function [] = ml_addpath()
%ML_ADDPATH Add Matlab files of the MORLAB toolbox to search path.
%
% SYNTAX:
%   ML_ADDPATH()
%
% DESCRIPTION:
%   This function adds the folder structure of the MORLAB toolbox to the
%   Matlab search path. For a permanent installation, the resulting search
%   path has to be saved using the Set Path tool of Matlab.
%
%   Note: This routine only works, if the folder structure is still in the
%         original state.
%
% INPUTS:
%   ---
%
% OUTPUTS:
%   ---
%
% See also ml_morlabopts, morlab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INITIALIZATION.                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Get separators for files and paths.
fsep = filesep();
psep = pathsep();

% Determines path of this function.
[rootpath, ~, ~] = fileparts(mfilename('fullpath'));
rootpath         = [rootpath fsep];

% Set default path to source files.
rootpathsrc = [rootpath 'src' fsep];


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MORLAB FUNCTIONS.                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Collect all paths into long string.
morlabpaths = [ ...
    rootpath psep ...
    rootpathsrc psep ...
    rootpathsrc 'checks' fsep psep ...
    rootpathsrc 'demos' fsep psep ...
    rootpathsrc 'demos' fsep 'data' fsep psep ...
    rootpathsrc 'demos' fsep 'datascripts' fsep psep ...
    rootpathsrc 'eqn_solvers' fsep psep ...
    rootpathsrc 'evaluation' fsep psep ...
    rootpathsrc 'mor' fsep psep ...
    rootpathsrc 'mor' fsep 'ct' fsep psep ...
    rootpathsrc 'mor' fsep 'ct' fsep 'dense' fsep psep ...
    rootpathsrc 'mor' fsep 'ct' fsep 'sparse' fsep psep ...
    rootpathsrc 'mor' fsep 'dt' fsep psep ...
    rootpathsrc 'mor' fsep 'dt' fsep 'dense' fsep psep ...
    rootpathsrc 'mor' fsep 'dt' fsep 'sparse' fsep psep ...
    rootpathsrc 'subroutines' fsep];

% Add all paths.
addpath(morlabpaths);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DOCUMENTATION.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Add paths of all Contents.m files.
addpath(genpath([rootpathsrc 'contents' fsep]));

% Add paths of html toolbox documentation if present.
if exist([rootpath fsep 'doc' fsep], 'dir')
    addpath([rootpath fsep 'doc' fsep]);
end

if exist([rootpath fsep 'doc' fsep 'html' fsep], 'dir')
    addpath(genpath([rootpath fsep 'doc' fsep 'html' fsep]));
end

if exist([rootpath fsep 'src' fsep 'demos' fsep 'html' fsep], 'dir')
    addpath(genpath([rootpath fsep 'src' fsep 'demos' fsep 'html' fsep]));
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% UNIT TEST FRAMEWORK / SCRIPTS.                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Add paths of scripts if present.
if exist([rootpath fsep '_scripts' fsep], 'dir')
    addpath([rootpath fsep '_scripts' fsep]);
end

% Add paths of unit test framework if present.
basetest = [rootpath fsep '_tests' fsep];
if exist(basetest, 'dir')
    testpaths = [ ...
        basetest psep ...
        basetest 'coverage' fsep psep ...
        basetest 'coverage' fsep 'MOcov' fsep psep ...
        basetest 'data' fsep psep ...
        basetest 'data' fsep 'eqn_solvers' fsep psep ...
        basetest 'data' fsep 'subroutines' fsep psep ...
        basetest 'data' fsep 'mor' fsep psep ...
        basetest 'data' fsep 'mor' fsep 'ct' fsep psep ...
        basetest 'data' fsep 'mor' fsep 'ct' fsep 'dense' fsep psep ...
        basetest 'data' fsep 'mor' fsep 'ct' fsep 'sparse' fsep psep ...
        basetest 'data' fsep 'mor' fsep 'dt' fsep psep ...
        basetest 'data' fsep 'mor' fsep 'dt' fsep 'dense' fsep psep ...
        basetest 'helpers' fsep psep ...
        basetest 'test_checks' fsep psep ...
        basetest 'test_demos' fsep psep ...
        basetest 'test_eqn_solvers' fsep psep ...
        basetest 'test_evaluation' fsep psep ...
        basetest 'test_mor' fsep psep ...
        basetest 'test_mor' fsep 'ct' fsep psep ...
        basetest 'test_mor' fsep 'ct' fsep 'dense' fsep psep ...
        basetest 'test_mor' fsep 'ct' fsep 'sparse' fsep psep ...
        basetest 'test_mor' fsep 'dt' fsep psep ...
        basetest 'test_mor' fsep 'dt' fsep 'dense' fsep psep ...
        basetest 'test_mor' fsep 'dt' fsep 'sparse' fsep psep ...
        basetest 'test_subroutines' fsep];
    addpath(testpaths);
end
