function [Z, info] = ml_signm(A, E, opts)
%ML_SIGNM Matrix sign function iteration.
%
% SYNTAX:
%   [Z, info] = ML_SIGNM(A)
%   [Z, info] = ML_SIGNM(A, [], opts)
%
%   [Z, info] = ML_SIGNM(A, E)
%   [Z, info] = ML_SIGNM(A, E, opts)
%
% DESCRIPTION:
%   The Newton iteration is used to compute the matrix sign function
%
%                         [ -eye(k)     0    ]
%       Z = sign(A) = T * [                  ]  * T^(-1),               (1)
%                         [  0      eye(n-k) ]
%
%   or the generalized matrix sign function
%
%                                      [ -eye(k)     0    ]
%       s*E - Z = sign(inv(E)*A) = T * [                  ]  * T^(-1),  (2)
%                                      [  0      eye(n-k) ]
%
%   with k eigenvalues of A (or s*E - A) are in the open left half-plane
%   and n-k are in the open right half-plane.
%
% INPUTS:
%   A    - matrix with dimensions n x n from (1) or (2)
%   E    - matrix with dimensions n x n from (2), if empty the standard
%          sign function is computed
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsTol          | nonnegative scalar, tolerance for the absolute    |
%   |                 | change in the last iteration step                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Info            | {0, 1}, used to disable/enable display of verbose |
%   |                 | status information during iteration steps         |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | MaxIter         | positive integer, maximum number of iteration     |
%   |                 | steps                                             |
%   |                 | (default 100)                                     |
%   +-----------------+---------------------------------------------------+
%   | RelTol          | nonnegative scalar, tolerance for the relative    |
%   |                 | change in the last iteration step                 |
%   |                 | (default 1.0e+02*n*eps)                           |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   Z    - matrix sign function of (1) or (2)
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErr          | vector, containing the absolute change of the     |
%   |                 | iteration matrix in each iteration step           |
%   +-----------------+---------------------------------------------------+
%   | IterationSteps  | number of performed iteration steps               |
%   +-----------------+---------------------------------------------------+
%   | RelErr          | vector, containing the relative change of the     |
%   |                 | iteration matrix in each iteration step           |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, Partial stabilization of descriptor systems using spectral
%   projectors, in: P. Van Dooren, S. P. Bhattacharyya, R. H. Chan, V.
%   Olshevsky, A.Routray (Eds.), Numerical Linear Algebra in Signals,
%   Systems and Control, Vol. 80 of Lect. Notes Electr. Eng., Springer
%   Netherlands, 2011, pp. 55--76.
%   https://doi.org/10.1007/978-94-007-0602-6_3
%
% See also ml_disk, ml_getqz.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 3);

if (nargin < 3) || isempty(opts)
    opts = struct();
end

% Check input matrices.
n = size(A, 1);

assert(isequal(size(A), [n n]), ...
    'MORLAB:data', ...
    'The matrix A has to be square!');

if issparse(A), A = full(A); end

if (nargin >= 2) && not(isempty(E))
    assert(isequal(size(E), [n n]), ...
        'MORLAB:data', ...
        'The matrix E must have the same dimensions as A!');

    if issparse(E), E = full(E); end

    hasE = 1;
else
    E    = eye(n);
    hasE = 0;
end

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'AbsTol')
    ml_assert_nonnegscalar(opts.AbsTol, 'opts.AbsTol');
else
    opts.AbsTol = 0;
end

if ml_field_set_to_value(opts, 'Info')
    ml_assert_boolean(opts.Info, 'opts.Info');
else
    opts.Info = false;
end

if ml_field_set_to_value(opts, 'MaxIter')
    ml_assert_posinteger(opts.MaxIter, 'opts.MaxIter');
else
    opts.MaxIter = 100;
end

if ml_field_set_to_value(opts, 'RelTol')
    ml_assert_nonnegscalar(opts.RelTol, 'opts.RelTol');
else
    opts.RelTol = 1.0e+02 * (n * eps);
end

% Case of empty data.
if isempty(A)
    Z    = [];
    info = struct([]);
    return;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INITIALIZATION.                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

niter  = 0;
Z      = A;
Znorm  = norm(Z, 'fro');

[abserr, relerr] = deal(zeros(1, opts.MaxIter));

% Set method to unconverged.
converged = 0;


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SIGN FUNCTION ITERATION.                                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

while (niter < opts.MaxIter) && not(converged)
    Z2   = Z;
    Zinv = Z \ E;

    if hasE
        Zinv = E * Zinv;
    end

    % Scaling factor for convergence acceleration.
    c  = sqrt(Znorm / norm(Zinv, 'fro'));

    % Update of iteration matrix.
    Z     = Z / (2.0 * c) + (0.5 * c) * Zinv;
    Znorm = norm(Z, 'fro');

    niter         = niter + 1;
    abserr(niter) = norm(Z - Z2 , 'fro');
    relerr(niter) = abserr(niter) / Znorm;

    % Information about current iteration step.
    if opts.Info
        fprintf(1, ['SIGNM step: %4d absolute change: %e' ...
            ' relative change: %e \n'], ...
            niter, abserr(niter), relerr(niter));
    end

    % Method is converged if absolute or relative changes are small enough.
    converged = (abserr(niter) <= opts.AbsTol) || ...
        (relerr(niter) <= opts.RelTol);
end

% Warning if iteration not converged.
if (niter == opts.MaxIter) && not(converged)
    warning('MORLAB:noConvergence', ...
        ['No convergence in %d iteration steps!\n' ...
        'Abs. tolerance: %e | Abs. change: %e\n' ...
        'Rel. tolerance: %e | Rel. change: %e\n' ...
        'Try to increase the tolerances or number of ' ...
        'iteration steps!'], ...
        niter, opts.AbsTol, min(abserr(niter)), ...
        opts.RelTol, min(relerr(niter)));
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN INFORMATION.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign information about iteration.
info = struct( ...
    'AbsErr'        , abserr(1:niter), ...
    'IterationSteps', niter, ...
    'RelErr'        , relerr(1:niter));
