function rom = ml_projtrunc_soss(sys, V, W)
%ML_PROJTRUNC_SOSS Compute reduced second-order model via projection.
%
% SYNATX:
%   rom = ML_PROJTRUNC_SOSS(sys, V, W)
%
% DESCRIPTION:
%   This function computes reduced second-order models using the projection
%   approach via the basis matrices V and W.
%
% INPUTS:
%   sys - structure, containing the improper descriptor system in the form:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | M               | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   | K               | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   | Bu              | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   | Cp              | matrix with dimensions p x n, might be empty      |
%   +-----------------+---------------------------------------------------+
%   | Cv              | matrix with dimensions p x n, might be empty      |
%   +-----------------+---------------------------------------------------+
%   | D               | matrix with dimensions p x m, might be empty      |
%   +-----------------+---------------------------------------------------+
%   V   - right projection basis or struct of right projection bases
%   {!}   with entries V.Vp and V.Vv
%   W   - left projection matrix or struct of left projection bases
%   {!}   with entries W.Wp and W.Wv
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom - struct, containing the reduced-order matrices as:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | M               | matrix with dimensions r x r                      |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions r x r                      |
%   +-----------------+---------------------------------------------------+
%   | K               | matrix with dimensions r x r                      |
%   +-----------------+---------------------------------------------------+
%   | Bu              | matrix with dimensions r x m                      |
%   +-----------------+---------------------------------------------------+
%   | Cp              | matrix with dimensions p x r, might be empty      |
%   +-----------------+---------------------------------------------------+
%   | Cv              | matrix with dimensions p x r, might be empty      |
%   +-----------------+---------------------------------------------------+
%   | D               | matrix with dimensions p x m, might be empty      |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_projtrunc_proper, ml_balproj_soss.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 3);

% Check if necessary functions and fields exist.
assert(ml_field_set_to_value(sys, 'SystemType') ...
    && ml_field_set_to_value(sys, 'DecidedSystemType') ...
    && sys.DecidedSystemType, ...
    'MORLAB:data', ...
    'Run first ml_decide_system_type.');

assert(ml_field_set_to_value(sys, 'PreparedSystemData') ...
    && sys.PreparedSystemData, ...
    'MORLAB:data', ...
    'Run first ml_prepare_system_data.');

if ml_field_set_to_value(sys, 'nM')
    n = sys.nM(end);
else
    n = size(sys.K, 1);
end

% Check projection bases.
if isa(V, 'cell')
    numRoms = length(V);
else
    V       = {V};
    numRoms = 1;
end

if not(isa(W, 'cell')), W = {W}; end

assert(length(W) == numRoms, ...
    'MORLAB:data', ...
    'There must be as many W as T matrices.');

for k = 1:numRoms
    if isa(V{k}, 'double')
        assert(isequal(size(V{k}), size(W{k})), ...
            'MORLAB:data', ...
            'The matrix T must have the same dimensions as W!');

        assert(size(V{k}, 1) == n, ...
            'MORLAB:data', ...
            'The matrix T must have the same number of rows as sys.K!');

        assert(size(W{k}, 1) == n, ...
            'MORLAB:data', ...
            'The matrix Winf must have the same number of rows as sys.K!');

        if issparse(V{k}), V{k} = full(V{k}); end
        if issparse(W{k}), W{k} = full(W{k}); end
    elseif isa(V{k}, 'struct')
        assert(isa(W{k}, 'struct'), ...
            'MORLAB:data', ...
            ['Both projection bases have to be given as structs for' ...
            ' second-order balancing.']);

        assert(isequal(size(V{k}.Vp), size(V{k}.Vv)) ...
            && isequal(size(V{k}.Vp), size(W{k}.Wp)) ...
            && isequal(size(V{k}.Vp), size(W{k}.Wv)), ...
            'MORLAB:data', ...
            'The matrices in V must have the same dimensions as in W!');

        assert(size(V{k}.Vp, 1) == n, ...
            'MORLAB:data', ...
            'The matrix Vp must have the same number of rows as sys.K!');

        assert(size(V{k}.Vv, 1) == n, ...
            'MORLAB:data', ...
            'The matrix Vv must have the same number of rows as sys.K!');

        assert(size(W{k}.Wp, 1) == n, ...
            'MORLAB:data', ...
            'The matrix Wp must have the same number of rows as sys.K!');

        assert(size(W{k}.Wv, 1) == n, ...
            'MORLAB:data', ...
            'The matrix Wv must have the same number of rows as sys.K!');

        if issparse(V{k}.Vp), V{k}.Vp = full(V{k}.Vp); end
        if issparse(V{k}.Vv), V{k}.Vv = full(V{k}.Vv); end
        if issparse(W{k}.Wp), W{k}.Wp = full(W{k}.Wp); end
        if issparse(W{k}.Wv), W{k}.Wv = full(W{k}.Wv); end
    else
        error('MORLAB:data', ...
            'Wrong data type for projection bases.');
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF REDUCED-ORDER MODELS.                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rom = cell(1, numRoms);

switch lower(sys.SystemType)
    case 'ct_d_soss'
        for k = 1:numRoms
            if isa(V{k}, 'struct')
                S = W{k}.Wp' * V{k}.Vv;

                rom{k} = struct( ...
                    'M' , S * ((W{k}.Wv' * (sys.M * V{k}.Vv)) / S), ...
                    'E' , S * ((W{k}.Wv' * (sys.E * V{k}.Vv)) / S), ...
                    'K' , S * (W{k}.Wv' * (sys.K * V{k}.Vp)), ...
                    'Bu', S * (W{k}.Wv' * sys.Bu));

                if ml_field_set_to_value(sys, 'Cp')
                    rom{k}.Cp = sys.Cp * V{k}.Vp;
                else
                    rom{k}.Cp = [];
                end

                if ml_field_set_to_value(sys, 'Cv')
                    rom{k}.Cv = (sys.Cv * V{k}.Vv) / S;
                else
                    rom{k}.Cv = [];
                end
            else
                rom{k} = struct( ...
                    'M' , W{k}' * (sys.M * V{k}), ...
                    'E' , W{k}' * (sys.E * V{k}), ...
                    'K' , W{k}' * (sys.K * V{k}), ...
                    'Bu', W{k}' * sys.Bu);

                if ml_field_set_to_value(sys, 'Cp')
                    rom{k}.Cp = sys.Cp * V{k};
                else
                    rom{k}.Cp = [];
                end

                if ml_field_set_to_value(sys, 'Cv')
                    rom{k}.Cv = sys.Cv * V{k};
                else
                    rom{k}.Cv = [];
                end
            end

            rom{k}.D = sys.D;
        end

    case 'ct_s_soss_so_1'
        for k = 1:numRoms
            if isa(V{k}, 'struct')
                S = W{k}.Wp' * V{k}.Vv;

                rom{k} = struct( ...
                    'M' , S * ((W{k}.Wv' * (sys.M * V{k}.Vv)) / S), ...
                    'E' , S * ((W{k}.Wv' * (sys.E * V{k}.Vv)) / S), ...
                    'K' , S * (W{k}.Wv' * (sys.K * V{k}.Vp)), ...
                    'Bu', S * (W{k}.Wv' * sys.pBu));

                if ml_field_set_to_value(sys, 'pCp')
                    rom{k}.Cp = sys.pCp * V{k}.Vp;
                else
                    rom{k}.Cp = [];
                end

                if ml_field_set_to_value(sys, 'pCv')
                    rom{k}.Cv = (sys.pCv * V{k}.Vv) / S;
                else
                    rom{k}.Cv = [];
                end
            else
                rom{k} = struct( ...
                    'M' , W{k}' * (sys.M * V{k}), ...
                    'E' , W{k}' * (sys.E * V{k}), ...
                    'K' , W{k}' * (sys.K * V{k}), ...
                    'Bu', W{k}' * sys.pBu);

                if ml_field_set_to_value(sys, 'pCp')
                    rom{k}.Cp = sys.pCp * V{k};
                else
                    rom{k}.Cp = [];
                end

                if ml_field_set_to_value(sys, 'pCv')
                    rom{k}.Cv = sys.pCv * V{k};
                else
                    rom{k}.Cv = [];
                end
            end

            rom{k}.D = sys.pD;
        end

    case 'ct_s_soss_dae_1_so'
        one = sys.nM;
        two = sys.zM;

        for k = 1:numRoms
            if isa(V{k}, 'struct')
                S = W{k}.Wp' * V{k}.Vv;

                rom{k} = struct( ...
                    'M' , S * ((W{k}.Wv' * (sys.M(one, one) ...
                    * V{k}.Vv)) / S), ...
                    'E' , S * ((W{k}.Wv' * (sys.E(one, one) ...
                    * V{k}.Vv)) / S), ...
                    'K' , S * (W{k}.Wv' * (sys.K(one, one) * V{k}.Vp ...
                    - sys.K(one, two) * (sys.K(two, two) ...
                    \ (sys.K(two, one) * V{k}.Vp)))), ...
                    'Bu', S * (W{k}.Wv' * sys.pBu));

                if ml_field_set_to_value(sys, 'pCp')
                    rom{k}.Cp = sys.pCp * V{k}.Vp;
                else
                    rom{k}.Cp = [];
                end

                if ml_field_set_to_value(sys, 'pCv')
                    rom{k}.Cv = (sys.pCv * V{k}.Vv) / S;
                else
                    rom{k}.Cv = [];
                end
            else
                rom{k} = struct( ...
                    'M' , W{k}' * (sys.M(one, one) * V{k}), ...
                    'E' , W{k}' * (sys.E(one, one) * V{k}), ...
                    'K' , W{k}' * (sys.K(one, one) * V{k} ...
                    - sys.K(one, two) * (sys.K(two, two) ...
                    \ (sys.K(two, one) * V{k}))), ...
                    'Bu', W{k}' * sys.pBu);

                if ml_field_set_to_value(sys, 'pCp')
                    rom{k}.Cp = sys.pCp * V{k};
                else
                    rom{k}.Cp = [];
                end

                if ml_field_set_to_value(sys, 'pCv')
                    rom{k}.Cv = sys.pCv * V{k};
                else
                    rom{k}.Cv = [];
                end
            end

            rom{k}.D = sys.pD;
        end

    otherwise
        error('MORLAB:data', ...
            ['There is no second-order balancing implemented for the' ...
            ' given system type.']);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FORMAT OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numRoms == 1
    rom = rom{:};
end
