function rom = ml_projtrunc_proper(sys, V, W)
%ML_PROJTRUNC_PROPER Compute reduced proper model via projection.
%
% SYNATX:
%   rom = ML_PROJTRUNC_PROPER(sys, V, W)
%
% DESCRIPTION:
%   This function computes reduced first-order models of proper models
%   using the projection approach via the basis matrices V and W.
%   The given models can be in standard form
%
%       x'(t) = A*x(t) + B*u(t), y(t) = C*x(t),
%
%   descriptor systems
%
%       E*x'(t) = A*x(t) + B*u(t), y(t) = C*x(t),
%
%   and second-order systems
%
%       M*x''(t) + E*x'(t) + K*x'(t) = Bu*u(t), y(t) = Cp*x(t) + Cv*x'(t).
%
% INPUTS:
%   sys - structure, containing the system matrices
%   V   - matrix, basis of right projection space
%   {!}
%   W   - matrix, basis of left projection space
%   {!}
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom - struct, containing the reduced-order matrices as:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | A               | matrix with dimensions r x r                      |
%   +-----------------+---------------------------------------------------+
%   | B               | matrix with dimensions r x m                      |
%   +-----------------+---------------------------------------------------+
%   | C               | matrix with dimensions p x r                      |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions r x r, if non-standard case|
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_projtrunc_improper, ml_balproj_proper.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 3);

% Check if necessary functions and fields exist.
assert(ml_field_set_to_value(sys, 'SystemType') ...
    && ml_field_set_to_value(sys, 'DecidedSystemType') ...
    && sys.DecidedSystemType, ...
    'MORLAB:data', ...
    'Run first ml_decide_system_type.');

assert(ml_field_set_to_value(sys, 'PreparedSystemData') ...
    && sys.PreparedSystemData, ...
    'MORLAB:data', ...
    'Run first ml_prepare_system_data.');

if strcmpi(sys.SystemType, 'ct_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'ct_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_so_1') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')

    assert(exist('mess_version', 'file') == 2, ...
        'MORLAB:mmess', ...
        ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
        ' installed!']);
end

% Check projection bases.
if isfield(sys, 'nE')
    n = sys.nE(end);
elseif isfield(sys, 'nM')
    n = 2 * sys.nM(end);
elseif isfield(sys, 'K')
    n = 2 * size(sys.K, 1);
else
    n = size(sys.A, 1);
end

if isa(V, 'cell')
    numRoms = length(V);
else
    V       = {V};
    numRoms = 1;
end

if not(isa(W, 'cell')), W = {W}; end

assert(length(W) == numRoms, ...
    'MORLAB:data', ...
    'There must be as many W as V matrices.');

for k = 1:numRoms
    assert(isequal(size(V{k}), size(W{k})), ...
        'MORLAB:data', ...
        'The matrix V must have the same dimensions as W!');

    assert(size(V{k}, 1) == n, ...
        'MORLAB:data', ...
        'The matrix V must have the same number of rows as sys.A!');

    assert(size(W{k}, 1) == n, ...
        'MORLAB:data', ...
        'The matrix W must have the same number of rows as sys.A!');

    if issparse(V{k}), V{k} = full(V{k}); end
    if issparse(W{k}), W{k} = full(W{k}); end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF REDUCED-ORDER MODELS.                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rom = cell(1, numRoms);

switch lower(sys.SystemType)
    case {'ct_d_ss', 'dt_d_ss', 'ct_s_ss_default', 'dt_s_ss_default'}
        for k = 1:numRoms
            rom{k} = struct( ...
                'A', W{k}' * (sys.A * V{k}), ...
                'B', W{k}' * sys.B, ...
                'C', sys.C * V{k}, ...
                'D', sys.D);
        end

    case {'ct_d_dss', 'dt_d_dss', 'ct_s_dss_default', 'dt_s_dss_default'}
        for k = 1:numRoms
            rom{k} = struct( ...
                'A', W{k}' * (sys.A * V{k}), ...
                'B', W{k}' * sys.B, ...
                'C', sys.C * V{k}, ...
                'D', sys.D, ...
                'E', W{k}' * (sys.E * V{k}));
        end

    case {'ct_s_dss_dae_1', 'dt_s_dss_dae_1'}
        oper = operatormanager(struct(), 'dae_1');
        eqn  = struct('E_', sys.E, 'A_', sys.A, ...
            'manifold_dim', sys.nE(end), 'haveE', true);

    case {'ct_s_dss_dae_2', 'dt_s_dss_dae_2'}
        oper = operatormanager(struct(), 'dae_2');
        eqn  = struct('E_', sys.E, 'A_', sys.A, ...
            'manifold_dim', sys.nE(end), 'haveE', true);

    case 'ct_s_soss_so_1'
        oper = operatormanager(struct(), 'so_1');
        eqn  = struct('M_', sys.M, 'E_', sys.E, 'K_', sys.K, ...
            'haveE', true);

    case 'ct_s_soss_dae_1_so'
        oper = operatormanager(struct(), 'dae_1_so');
        eqn  = struct('M_', sys.M, 'E_', sys.E, 'K_', sys.K, ...
            'manifold_dim', sys.nM(end), 'haveE', true);

    case 'ct_s_soss_dae_2_so'
        oper = operatormanager(struct(), 'dae_2_so');
        eqn  = struct('M_', sys.pM, 'E_', sys.pE, 'K_', sys.pK, ...
            'G_', sys.pG, 'haveE', true);

    case 'ct_s_soss_dae_3_so'
        oper = operatormanager(struct(), 'dae_3_so');
        eqn  = struct('M_', sys.pM, 'E_', sys.pE, 'K_', sys.pK, ...
            'G_', sys.pG, 'haveE', true);

    otherwise
        error('MORLAB:data', ...
            'The given system type is not implemented!');
end

if strcmpi(sys.SystemType, 'ct_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'ct_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_so_1') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')
    [eqn, ~, oper] = oper.mul_A_pre(eqn, [], oper);
    [eqn, ~, oper] = oper.mul_E_pre(eqn, [], oper);

    for k = 1:numRoms
        rom{k} = struct( ...
            'A', W{k}' * oper.mul_A(eqn, [], 'N', V{k}, 'N'), ...
            'B', W{k}' * sys.pB, ...
            'C', sys.pC * V{k}, ...
            'D', sys.pD, ...
            'E', W{k}' * oper.mul_E(eqn, [], 'N', V{k}, 'N'));
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FORMAT OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numRoms == 1
    rom = rom{:};
end
