function r = ml_order(hsv, rmin, rselect, opts)
%ML_ORDER Order computation for model reduction methods.
%
% SYNTAX:
%   nr = ML_ORDER(hsv, rmin, rselect)
%   nr = ML_ORDER(hsv, rmin, rselect, opts)
%
% DESCRIPTION:
%   This function computes the order of the reducable part of a model
%   based on a given vector of characteristic values and appropriate error
%   formulas.
%
% INPUTS:
%   hsv     - vector, containing the characteristic values of the system
%   rmin    - integer, minimum order reduced-order model resulting from
%             parts that are not present in the characteristic values hsv
%   rselect - integer, used to determine the computation method for the
%   {!}       order of the reduced-order model
%               0 - order is directly given by user
%               1 - computed by a relative tolerance for the hsv
%               2 - computed by a relative tolerance on the sum of hsv
%               3 - computed by absolute error bound of BT
%               4 - computed by relative error bound of BST
%               5 - computed by absolute error bound of LQGBT
%               6 - computed by absolute error bound of HinfBT
%   opts    - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Gamma           | positive scalar, scaling term from the H-infinity |
%   |                 | balanced truncation                               |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | rselect == 0                                      |
%   |                 | (default min(10,length(hsv)+rmin))                |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used in the         |
%   | {!}             | different error formulas                          |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   r - size of the reducable model part
%   {!}
%
% See also ml_balproj_proper, ml_balproj_improper.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 4);

if (nargin < 4) || isempty(opts)
    opts = struct();
end

% Check of input arguments.
ml_assert_nonnegvector(hsv, 'hsv');

hsv = sort(hsv(hsv > 0), 'descend');

ml_assert_nonnegint(rmin, 'rmin');

rselect = ml_check_cell_param(rselect, 'rselect', @ml_assert_nonnegint);

% Special case of no characteristic singular values.
if isempty(hsv)
    % still expect cell
    r = 0;
    return;
end

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

opts = ml_check_cell_param(opts, 'Order', @ml_assert_posinteger, ...
    min(10, length(hsv)) + rmin);

opts = ml_check_cell_param(opts, 'Tolerance', @ml_assert_nonnegscalar,...
    1.0e-02);

[rselect, opts.Order, opts.Tolerance] = ml_extend_cell(rselect,...
    opts.Order, ...
    opts.Tolerance);

numVariants = length(rselect);
r           = cell(1, numVariants);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHOOSE ERROR FORMULA.                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for k = 1:length(rselect)
    % Computation of the order.
    switch rselect{k}
        case 0 % Selection by a given order.
            nr = opts.Order{k} - rmin;

            if nr > length(hsv)
                nr = length(hsv);

                warning('MORLAB:data', ...
                    ['The desired size of the reduced-order model is' ...
                    ' too large. \nSet opts.Order = %d.'], ...
                    nr);
            elseif nr < 0
                nr = 0;

                warning('MORLAB:data', ...
                    ['The desired size of the reduced-order model is ' ...
                    'too small. \nSet opts.Order = %d.'], ...
                    nr + rmin);
            end

        case 1 % Selection by a given tolerance multiplied with the largest
            % characteristic value as lower bound.
            nr = sum(hsv > hsv(1) * opts.Tolerance{k});

        case 2 % Selection by a given tolerance multiplied with the largest
            % characteristic value as lower bound on the sum.
            abserr = 0;
            nr     = length(hsv) + 1;

            while abserr <= (hsv(1) * opts.Tolerance{k})
                nr = nr - 1;

                if nr == 0
                    break;
                end

                abserr = abserr + hsv(nr);
            end

        case 3 % Selection by a given tolerance and the error formula of
            % the balanced truncation.
            abserr = 0;
            nr     = length(hsv) + 1;

            while abserr <= (opts.Tolerance{k} / 2)
                nr = nr - 1;

                if nr == 0
                    break;
                end

                abserr = abserr + hsv(nr);
            end

        case 4 % Selection by a given tolerance and the error formula of
            % the balanced stochastic truncation.
            relerr = 1;
            nr     = length(hsv) + 1;

            while relerr <= (opts.Tolerance{k} + 1)
                nr     = nr - 1;

                if nr == 0
                    break;
                end

                relerr = relerr * (1 + hsv(nr)) / (1 - hsv(nr));
            end

        case 5 % Selection by a given tolerance and the error formula of
            % the LQG balanced truncation.
            abserr = 0;
            nr     = length(hsv) + 1;

            while abserr <= (opts.Tolerance{k} / 2)
                nr = nr - 1;

                if nr == 0
                    break;
                end

                abserr = abserr + hsv(nr) / sqrt(1 + hsv(nr)^2);
            end

        case 6 % Selection by a given tolerance and the error formula of
            % the Hinf balanced truncation.
            if ml_field_set_to_value(opts, 'Gamma')
                ml_assert_posscalar(opts.Gamma, 'opts.Gamma');
            else
                error('MORLAB:data', ...
                    'The parameter opts.Gamma is needed!');
            end

            abserr = 0;
            nr     = length(hsv) + 1;

            while abserr <= (opts.Tolerance{k} / 2)
                nr = nr - 1;

                if nr == 0
                    break;
                end

                abserr = abserr + hsv(nr) / sqrt(1 ...
                    + ((1 - 1 / opts.Gamma^2) * hsv(nr))^2);
            end

        otherwise
            error('MORLAB:data', ...
                'The given parameter rselect is not implemented!');
    end
    r{k} = nr;
end

% Unpack.
if length(r) == 1
    r = r{1};
end
