function [Q, Z, nu] = ml_getqz(A, E, Aspace, Espace, opts)
%ML_GETQZ Subspace extraction method.
%
% SYNTAX:
%   [Q, Z, nu] = ML_GETQZ(A, E, Aspace)
%   [Q, Z, nu] = ML_GETQZ(A, E, Aspace, [], opts)
%
%   [Q, Z, nu] = ML_GETQZ(A, E, Aspace, Espace)
%   [Q, Z, nu] = ML_GETQZ(A, E, Aspace, Espace, opts)
%
% DESCRIPTION:
%   Computes orthogonal transformation matrices Q and Z for a deflating
%   subspace of the matrix pencil s*E - A, where the basis is given by the
%   null space of the matrix Aspace. The transformation Q'*(s*E - A)*Z is
%   in block triangular form with the leading block corresponding to the
%   deflating subspace. If additionally the Espace is given, a stabilized
%   version of the algorithm is performed.
%
% INPUTS:
%   A      - a matrix with dimensions n x n
%   E      - a matrix with dimensions n x n
%   Aspace - a matrix with dimensions n x n, with its null space the
%            deflating subspace
%   Espace - a matrix with dimensions n x n, complementary subspace of
%            Aspace, is allowed to be empty for the classical subspace
%            extraction method
%   opts   - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Dimension       | integer, dimension of the deflating subspace,     |
%   |                 | negative if unknown                               |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%   | RankTol         | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of Aspace to determine the |
%   |                 | rank of Aspace, only used if Espace is not given  |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   Q  - right orthogonal transformation matrix onto the deflating subspace
%   Z  - left orthogonal transformation matrix onto the deflating subspace
%   nu - dimension of the deflating subspace
%
%
% REFERENCE:
%   P. Benner, Partial stabilization of descriptor systems using spectral
%   projectors, in: P. Van Dooren, S. P. Bhattacharyya, R. H. Chan, V.
%   Olshevsky, A.Routray (Eds.), Numerical Linear Algebra in Signals,
%   Systems and Control, Vol. 80 of Lect. Notes Electr. Eng., Springer
%   Netherlands, 2011, pp. 55--76.
%   https://doi.org/10.1007/978-94-007-0602-6_3
%
% See also ml_disk, ml_signm.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(3, 5);

if (nargin < 5) || isempty(opts)
    opts = struct();
end

% Check input matrices.
n = size(A, 1);

assert(isequal(size(A), [n n]), ...
    'MORLAB:data', ...
    'The matrix A has to be square!');

assert(isequal(size(E), [n n]), ...
    'MORLAB:data', ...
    'The matrix E must have the same dimensions as A!');

assert(isequal(size(Aspace), [n n]), ...
    'MORLAB:data', ...
    'The matrix Aspace must have the same dimensions as A!');

if issparse(A), A = full(A); end
if issparse(E), E = full(E); end
if issparse(Aspace), Aspace = full(Aspace); end

if (nargin > 3) && not(isempty(Espace))
    assert(isequal(size(Espace), [n n]), ...
        'MORLAB:data', ...
        'The matrix Espace must have the same dimensions as A!');

    if issparse(Espace), Espace = full(Espace); end
else
    Espace = [];
end

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'RankTol')
    ml_assert_nonnegscalar(opts.RankTol, 'opts.RankTol');
else
    opts.RankTol = log(n) * eps;
end

if ml_field_set_to_value(opts, 'Dimension')
    ml_assert_integer(opts.Dimension, 'opts.Dimension');

    assert(opts.Dimension <= n, ...
        'MORLAB:data', ...
        ['The parameter opts.Dimension has to be smaller or equal to ' ...
        'the dimension of the problem!']);
else
    opts.Dimension = -1;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SUBSPACE EXTRACTION METHOD.                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isempty(Espace))
    [U, ~] = qr([Espace(end:-1:1, end:-1:1), ...
        Aspace(end:-1:1, end:-1:1)]', 0);
    Aspace = U(end:-1:n+1, end:-1:1)';
end

if opts.Dimension < 0
    [~, S, V] = svd(Aspace);

    if isempty(Espace)
        if S(1, 1) <= opts.RankTol
            r = 0;
        else
            r = sum(diag(S) > S(1, 1) * opts.RankTol);
        end
    else
        r = sum(diag(S) > 0.5);
    end
else
    [V, ~, ~] = qr(Aspace');
    r         = n - opts.Dimension;
end

Z = V(: , r+1:n);

[Q, ~, ~] = qr([A * Z, E * Z]);
Z         = [Z, V(:, 1:r)];
nu        = n - r;
