function varargout = ml_extend_cell(varargin)
%ML_EXTEND_CELL Extends the given cell arrays to match in size.
%
% SYNTAX:
%   [cell1, cell2, ...] = ML_EXTEND_CELL(cell1, cell2, ...)
%
% DESCRIPTION:
%   Takes any number of 1D cell arrays and extends them to the length of
%   the largest one by repeating the first element in each one that is
%   shorter. A cell array may only have length one or the same as the
%   largest one.
%
% INPUTS:
%   cell1, cell2, ... - cell vectors
%
% OUTPUTS:
%   cell1, cell2, ... - cell vectors, with equal length
%
% See also ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

maxLen    = max(cellfun(@length, varargin));
numArgs   = length(varargin);
varargout = cell(1, numArgs);

for k = 1:numArgs
    len = length(varargin{k});

    % Extend to target size.
    if(len == 1)
        varargout{k} = repmat(varargin{k}, 1, maxLen);
    else
        assert(len == maxLen, ...
            'MORLAB:data', ...
            'All cell vectors must have the same length or length 1.');

        varargout{k} = varargin{k};
    end
end
