function [sys, info] = ml_dt_d_ss_adtf(sys, opts)
%ML_DT_D_SS_ADTF Add. dec. of dense standard system transfer functions.
%
% SYNTAX:
%   [sys, info] = ML_DT_D_SS_ADTF(sys)
%   [sys, info] = ML_DT_D_SS_ADTF(sys, opts)
%
% DESCRIPTION:
%   Consider a dense, standard discrete-time system of the form
%
%       x(t+1) = A*x(t) + B*u(t),                                       (1)
%        y(t)  = C*x(t).                                                (2)
%
%   This function computes an additive decomposition of the corresponding
%   transfer function by a block diagonalization of the matrix A. The
%   system is transformed, such that
%
%                [ As     ]       [ Bs ]
%       z(t+1) = [        ]z(t) + [    ]u(t),                           (3)
%                [     Au ]       [ Bu ]
%
%         y(t) = [ Cs, Cu ]z(t),                                        (4)
%
%   where As contains the eigenvalues with absolute values smaller than 1
%   and Au all the eigenvalues with absolute values larger than 1.
%
% INPUTS:
%   sys  - structure, containing the descriptor system in the form:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | A               | matrix with dimensions n x n in (1)               |
%   +-----------------+---------------------------------------------------+
%   | B               | matrix with dimensions n x m in (1)               |
%   +-----------------+---------------------------------------------------+
%   | C               | matrix with dimensions p x n in (2)               |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | stabsignmopts   | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_signm                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabdsylvopts   | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_dsylv_smith                         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | UnstabDim       | integer, dimension of the deflating anti-stable   |
%   |                 | subspace, negative if unknown                     |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   sys - structure, containing the transformed descriptor system:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | A               | matrix with dimensions ns x ns, see As in (3)     |
%   +-----------------+---------------------------------------------------+
%   | Au              | matrix with dimensions nu x nu in (3)             |
%   +-----------------+---------------------------------------------------+
%   | B               | matrix with dimensions ns x m, see Bs in (3)      |
%   +-----------------+---------------------------------------------------+
%   | Bu              | matrix with dimensions nu x m in (3)              |
%   +-----------------+---------------------------------------------------+
%   | C               | matrix with dimensions p x ns, see Cs in (4)      |
%   +-----------------+---------------------------------------------------+
%   | Cu              | matrix with dimensions p x nu in (4)              |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information about the
%          algorithms used for the additive decomposition
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoSTABSIGNM   | structure, containing information about the       |
%   |                 | matrix sign function method, see ml_signm         |
%   +-----------------+---------------------------------------------------+
%   | infoSTABDSYLV   | structure, containing information about the       |
%   |                 | Sylvester equation solver, see ml_dsylv_sgn       |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Number of identified stable eigenvalues           |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Number of identified anti-stable eigenvalues      |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   |                 | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   |                 | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   S. Werner, Hankel-norm approximation of descriptor systems, Master's
%   thesis, Otto von Guericke University, Magdeburg, Germany, 2016.
%   doi:10.25673/4507
%
% See also ml_ct_d_ss_adtf, ml_dt_d_dss_adtf.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check input standard system.
ml_assert_stdsys(sys);

if issparse(sys.A), sys.A = full(sys.A); end
if issparse(sys.B), sys.B = full(sys.B); end
if issparse(sys.C), sys.C = full(sys.C); end

n = size(sys.A, 1);

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'stabsignmopts')
    assert(isa(opts.stabsignmopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabsignmopts has to be a struct!');
else
    opts.stabsignmopts = struct();
end

if ml_field_set_to_value(opts, 'stabdsylvopts')
    assert(isa(opts.stabdsylvopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabdsylvopts has to be a struct!');
else
    opts.stabdsylvopts = struct();
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

if ml_field_set_to_value(opts, 'UnstabDim')
    ml_assert_integer(opts.UnstabDim, 'UnstabDim');
else
    opts.UnstabDim = -1;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION: ANTI-STABLE PART.                                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation deflating subspaces using the matrix sign function.
if opts.UnstabDim == 0
    nu = 0;
else
    [Z, infoSTABSIGNM] = ml_signm((sys.A + eye(n)) \ (sys.A - eye(n)), ...
        [], opts.stabsignmopts);

    if opts.UnstabDim > 0
        nu = opts.UnstabDim;
    else
        nu = (n + round(trace(Z))) / 2;
    end

    info.infoSTABSIGNM = infoSTABSIGNM;
end

% Block diagonalization of the system matrices.
if (nu > 0) && (nu < n)
    [Q, ~, ~] = qr(eye(n) + Z);

    sys.Au = Q(: , 1:nu)' * (sys.A * Q(: , 1:nu));
    Wa     = Q(: , 1:nu)' * (sys.A * Q(: , nu+1:end));
    sys.A  = Q(: , nu+1:end)' * (sys.A * Q(: , nu+1:end));
    Auinv  = sys.Au \ eye(nu);

    [Y, infoSTABDSYLV] = ml_dsylv_smith(Auinv, sys.A, -Auinv * Wa, ...
        [], [], opts.stabdsylvopts);

    sys.B  = Q' * sys.B;
    sys.Bu = sys.B(1:nu, :) - Y * sys.B(nu+1:end, :);
    sys.B  = sys.B(nu+1:end, :);

    sys.Cu = sys.C * Q;
    sys.C  = sys.Cu(: , nu+1:end) + sys.Cu(: , 1:nu) * Y;
    sys.Cu = sys.Cu(: , 1:nu);

    info.infoSTABDSYLV = infoSTABDSYLV;

    if opts.StoreProjection
        V = Q * [Y, eye(nu); eye(n - nu), zeros(size(Y'))];
        W = Q * [zeros(size(Y)), eye(nu); eye(n - nu), -Y'];
    else
        W = [];
        V = [];
    end
elseif nu == n
    sys.Au = sys.A;
    sys.Bu = sys.B;
    sys.Cu = sys.C;

    sys.A = [];
    sys.B = zeros(0, size(sys.Bu, 2));
    sys.C = zeros(size(sys.Cu, 1), 0);

    if opts.StoreProjection
        V = eye(n);
        W = eye(n);
    else
        W = [];
        V = [];
    end
else
    sys.Au = [];
    sys.Bu = zeros(0, size(sys.B, 2));
    sys.Cu = zeros(size(sys.C, 1), 0);

    if opts.StoreProjection
        V = eye(n);
        W = eye(n);
    else
        W = [];
        V = [];
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN INFORMATION.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Collection of all information.
info.Ns = n - nu;
info.Nu = nu;
info.V  = V;
info.W  = W;

[~, perm] = sort(lower(fieldnames(info)));
info      = orderfields(info, perm);
