function [V, W] = ml_domsubproj_soss(sys, R, L, rselect, opts)
%ML_DOMSUBPROJ_SOSS Compute projection basis for MOR of second-order parts.
%
% SYNTAX:
%   [V, W] = ML_DOMSUBPROJ_PROPER(sys, R, L, rmin, rselect)
%   [V, W] = ML_DOMSUBPROJ_PROPER(sys, R, L, rmin, rselect, opts)
%
% DESCRIPTION:
%   This function computes the structure-preserving dominant subspace
%   projection bases for second-order systems
%
%       M*x''(t) + E*x'(t) + K*x'(t) = Bu*u(t), y(t) = Cp*x(t) + Cv*x'(t).
%
% INPUTS:
%   sys     - structure, containing the system matrices
%   R       - basis matrix of right subspace
%   L       - basis matrix of left subspace
%   rmin    - nonnegative integer, allocated minimal reduced order for the
%             case that final reduced order is directly selected
%   rselect - integer, used to determine the computation method for the
%             order of the reduced-order model
%               0 - order is directly given by user
%               1 - computed by a relative tolerance for the hsv
%               2 - computed by a relative tolerance on the sum of hsv
%   opts    - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | rselect == 0                                      |
%   |                 | (default min(10,length(hsvp)))                    |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used in the         |
%   | {!}             | different error formulas                          |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | TwoSidedProj    | {0, 1}, if turned on two-sided projection is used |
%   |                 | rather than one-sided projection                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   V - right projection basis
%   {!}
%   W - left projection basis
%   {!}
%
%
% See also ml_balproj_soss, ml_projtrunc_proper, ml_order.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(4, 5);

if (nargin < 5) || isempty(opts)
    opts = struct();
end

assert(ml_field_set_to_value(sys, 'SystemType') ...
    && ml_field_set_to_value(sys, 'DecidedSystemType') ...
    && sys.DecidedSystemType, ...
    'MORLAB:data', ...
    'Run first ml_decide_system_type.');

assert(ml_field_set_to_value(sys, 'PreparedSystemData') ...
    && sys.PreparedSystemData, ...
    'MORLAB:data', ...
    'Run first ml_prepare_system_data.');

if strcmpi(sys.SystemType, 'ct_s_soss_so_1') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')

    assert(exist('mess_version', 'file') == 2, ...
        'MORLAB:mmess', ...
        ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
        ' installed!']);
end

% Check matrix equation factors.
if isfield(sys, 'nM')
    n = sys.nM(end);
else
    n = size(sys.K, 1);
end

if n == 0
    V = [];
    W = [];
    return;
end

assert(isa(R, 'double') && (size(R, 1) == n), ...
    'MORLAB:data', ...
    'The matrix R must have the same number of rows as sys.A!');

assert(isa(L, 'double') && (size(L, 1) == n), ...
    'MORLAB:data', ...
    'The matrix L must have the same number of rows as sys.A!');

if issparse(R), R = full(R); end
if issparse(L), L = full(L); end

rselect = ml_check_cell_param(rselect, 'rselect', @ml_assert_nonnegint);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'TwoSidedProj')
    ml_assert_boolean(opts.TwoSidedProj, 'opts.TwoSidedProj');
else
    opts.TwoSidedProj = false;
end

opts = ml_check_cell_param(opts, 'Tolerance', ...
    @ml_assert_nonnegscalar, 1.0e-02);
opts = ml_check_cell_param(opts, 'Order', @ml_assert_posinteger, []);

[rselect, opts.Tolerance, opts.Order] = ml_extend_cell(rselect, ...
    opts.Tolerance, opts.Order);

numProjections = length(rselect);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF REDUCED ORDERS & ORTHOGONALIZATION.                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if opts.TwoSidedProj
    [U1, S1, ~] = svd(R, 'econ');
    nrs1        = ml_order(diag(S1), 0, rselect, opts);

    [U2, S2, ~] = svd(L, 'econ');
    nrs2        = ml_order(diag(S2), 0, rselect, opts);
else
    [U1, S, ~] = svd([R, L], 'econ');
    U2         = U1;
    nrs1       = ml_order(diag(S), 0, rselect, opts);
    nrs2       = nrs1;
end

if not(isa(nrs1, 'cell')), nrs1 = {nrs1}; end
if not(isa(nrs2, 'cell')), nrs2 = {nrs2}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF PROJECTION.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

V = cell(1, numProjections);
W = cell(1, numProjections);

for k = 1:numProjections
    r    = min([nrs1{k}, nrs2{k}]);

    V{k} = U1(:, 1:r);
    W{k} = U2(:, 1:r);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FORMAT OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numProjections == 1
    W = W{:};
    V = V{:};
end
