function [V, W] = ml_domsubproj_proper(sys, R, L, rmin, rselect, opts)
%ML_DOMSUBPROJ_PROPER Compute projection basis for MOR of proper parts.
%
% SYNTAX:
%   [V, W] = ML_DOMSUBPROJ_PROPER(sys, R, L, rmin, rselect)
%   [V, W] = ML_DOMSUBPROJ_PROPER(sys, R, L, rmin, rselect, opts)
%
% DESCRIPTION:
%   This function computes the dominant subspace projection bases for
%   proper system parts for standard first-order systems
%
%       x'(t) = A*x(t) + B*u(t), y(t) = C*x(t),
%
%   descriptor systems
%
%       E*x'(t) = A*x(t) + B*u(t), y(t) = C*x(t),
%
%   and second-order systems
%
%       M*x''(t) + E*x'(t) + K*x'(t) = Bu*u(t), y(t) = Cp*x(t) + Cv*x'(t).
%
% INPUTS:
%   sys     - structure, containing the system matrices
%   R       - basis matrix of right subspace
%   L       - basis matrix of left subspace
%   rmin    - nonnegative integer, allocated minimal reduced order for the
%             case that final reduced order is directly selected
%   rselect - integer, used to determine the computation method for the
%             order of the reduced-order model
%               0 - order is directly given by user
%               1 - computed by a relative tolerance for the hsv
%               2 - computed by a relative tolerance on the sum of hsv
%   opts    - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | rselect == 0                                      |
%   |                 | (default min(10,length(sv)) + rmin)               |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used in the         |
%   | {!}             | different error formulas                          |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | TwoSidedProj    | {0, 1}, if turned on two-sided projection is used |
%   |                 | rather than one-sided projection                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   V - right projection basis
%   {!}
%   W - left projection basis
%   {!}
%
%
% See also ml_balproj_improper, ml_projtrunc_proper, ml_order.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(5, 6);

if (nargin < 6) || isempty(opts)
    opts = struct();
end

assert(ml_field_set_to_value(sys, 'SystemType') ...
    && ml_field_set_to_value(sys, 'DecidedSystemType') ...
    && sys.DecidedSystemType, ...
    'MORLAB:data', ...
    'Run first ml_decide_system_type.');

assert(ml_field_set_to_value(sys, 'PreparedSystemData') ...
    && sys.PreparedSystemData, ...
    'MORLAB:data', ...
    'Run first ml_prepare_system_data.');

if strcmpi(sys.SystemType, 'ct_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'ct_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_so_1') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')

    assert(exist('mess_version', 'file') == 2, ...
        'MORLAB:mmess', ...
        ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
        ' installed!']);
end

% Check matrix equation factors.
if isfield(sys, 'nE')
    n = sys.nE(end);
elseif isfield(sys, 'nM')
    n = 2 * sys.nM(end);
elseif isfield(sys, 'K')
    n = 2 * size(sys.K, 1);
else
    n = size(sys.A, 1);
end

assert(isa(R, 'double') && (size(R, 1) == n), ...
    'MORLAB:data', ...
    'The matrix R must have the same number of rows as sys.A!');

assert(isa(L, 'double') && (size(L, 1) == n), ...
    'MORLAB:data', ...
    'The matrix L must have the same number of rows as sys.A!');

if issparse(R), R = full(R); end
if issparse(L), L = full(L); end

ml_assert_nonnegint(rmin, 'rmin');

rselect = ml_check_cell_param(rselect, 'rselect', @ml_assert_nonnegint);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'TwoSidedProj')
    ml_assert_boolean(opts.TwoSidedProj, 'opts.TwoSidedProj');
else
    opts.TwoSidedProj = false;
end

opts = ml_check_cell_param(opts, 'Tolerance', ...
    @ml_assert_nonnegscalar, 1.0e-02);
opts = ml_check_cell_param(opts, 'Order', @ml_assert_posinteger, []);

[rselect, opts.Tolerance, opts.Order] = ml_extend_cell(rselect, ...
    opts.Tolerance, opts.Order);

numProjections = length(rselect);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF CHARACTERISTIC VALUES.                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Special case of empty data.
if n == 0
    V = [];
    W = [];
    return;
end

% Computation of characteristic values.
systype = lower(sys.SystemType);
switch systype
    case {'ct_d_ss', 'dt_d_ss', 'ct_s_ss_default', 'dt_s_ss_default', ...
            'ct_d_dss', 'dt_d_dss', 'ct_s_dss_default', 'dt_s_dss_default'}
        % Do nothing.

    case {'ct_s_dss_dae_1', 'dt_s_dss_dae_1'}
        oper = operatormanager(struct(), 'dae_1');
        eqn  = struct('E_', sys.E, 'A_', sys.A, ...
            'manifold_dim', sys.nE(end), 'haveE', true);

    case {'ct_s_dss_dae_2', 'dt_s_dss_dae_2'}
        oper = operatormanager(struct(), 'dae_2');
        eqn  = struct('E_', sys.E, 'A_', sys.A, ...
            'manifold_dim', sys.nE(end), 'haveE', true);

    case 'ct_s_soss_so_1'
        oper = operatormanager(struct(), 'so_1');
        eqn  = struct('M_', sys.M, 'E_', sys.E, 'K_', sys.K, ...
            'haveE', true);

    case 'ct_s_soss_dae_1_so'
        oper = operatormanager(struct(), 'dae_1_so');
        eqn  = struct('M_', sys.M, 'E_', sys.E, 'K_', sys.K, ...
            'manifold_dim', sys.nM(end), 'haveE', true);

    case 'ct_s_soss_dae_2_so'
        oper = operatormanager(struct(), 'dae_2_so');
        eqn  = struct('M_', sys.pM, 'E_', sys.pE, 'K_', sys.pK, ...
            'G_', sys.pG, 'haveE', true);

    case 'ct_s_soss_dae_3_so'
        oper = operatormanager(struct(), 'dae_3_so');
        eqn  = struct('M_', sys.pM, 'E_', sys.pE, 'K_', sys.pK, ...
            'G_', sys.pG, 'haveE', true);

    otherwise
        error('MORLAB:data', ...
            'The given system type is not implemented.');
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF REDUCED ORDERS & ORTHOGONALIZATION.                      %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if opts.TwoSidedProj
    [U1, S1, ~] = svd(R, 'econ');
    nrs1        = ml_order(diag(S1), rmin, rselect, opts);

    [U2, S2, ~] = svd(L, 'econ');
    nrs2        = ml_order(diag(S2), rmin, rselect, opts);
else
    [U1, S, ~] = svd([R, L], 'econ');
    U2         = U1;
    nrs1       = ml_order(diag(S), rmin, rselect, opts);
    nrs2       = nrs1;
end

if not(isa(nrs1, 'cell')), nrs1 = {nrs1}; end
if not(isa(nrs2, 'cell')), nrs2 = {nrs2}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF PROJECTION.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if strcmpi(sys.SystemType, 'ct_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_1') ...
        || strcmpi(sys.SystemType, 'ct_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'dt_s_dss_dae_2') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_so_1') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so') ...
        || strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')

    [eqn, ~, oper] = oper.mul_E_pre(eqn, [], oper);
end

V = cell(1, numProjections);
W = cell(1, numProjections);

for k = 1:numProjections
    r    = min([nrs1{k}, nrs2{k}]);

    V{k} = U1(:, 1:r);
    W{k} = U2(:, 1:r);

    switch lower(sys.SystemType)
        case {'ct_d_ss', 'dt_d_ss', ...
                'ct_s_ss_default', 'dt_s_ss_default'}
            W{k} = W{k} / (W{k}' * V{k})';

        case {'ct_d_dss', 'dt_d_dss', ...
                'ct_s_dss_default', 'dt_s_dss_default'}
            W{k} = W{k} / (W{k}' * (sys.E * V{k}))';

        otherwise
            EV   = oper.mul_E(eqn, [], 'N', V{k}, 'N');
            W{k} = W{k} / (W{k}' * (EV))';
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FORMAT OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numProjections == 1
    W = W{:};
    V = V{:};
end
