function [Aspace, Espace, info] = ml_disk(A, E, opts)
%ML_DISK Inverse free iteration for the matrix disk function.
%
% SYNTAX:
%   [Aspace, Espace, info] = ML_DISK(A, E)
%   [Aspace, Espace, info] = ML_DISK(A, E, opts)
%
% DESCRIPTION:
%   Computes the inverse free iteration for the matrix pencil s*E - A with
%   no eigenvalues on the unit circle. The results can be used to determine
%   the deflating subspaces of s*E - A corresponding to the eigenvalues
%   inside and outside the unit circle, respectively.
%
% INPUTS:
%   A    - a matrix with dimensions n x n
%   E    - a matrix with dimensions n x n
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsTol          | nonnegative scalar, tolerance for the absolute    |
%   |                 | change in the last iteration step                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Info            | {0, 1}, used to disable/enable display of verbose |
%   |                 | status information during iteration steps         |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | MaxIter         | positive integer, maximum number of iteration     |
%   |                 | steps                                             |
%   |                 | (default 100)                                     |
%   +-----------------+---------------------------------------------------+
%   | RelTol          | nonnegative scalar, tolerance for the relative    |
%   |                 | change in the last iteration step                 |
%   |                 | (default 1.0e+02*n*eps)                           |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   Aspace - a matrix with dimensions n x n, the null space of Aspace is
%            the deflating subspace corresponding to the eigenvalues
%            inside the unit circle
%   Espace - a matrix with dimensions n x n, the null space of Espace is
%            the deflating subspace corresponding to the eigenvalues
%            outside the unit circle
%   info   - structure, containing the following information about the
%            inverse free iteration:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErr          | vector, containing the absolute change of the     |
%   |                 | iteration matrix in each iteration step           |
%   +-----------------+---------------------------------------------------+
%   | IterationSteps  | number of performed iteration steps               |
%   +-----------------+---------------------------------------------------+
%   | RelErr          | vector, containing the relative change of the     |
%   |                 | iteration matrix in each iteration step           |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, Partial stabilization of descriptor systems using spectral
%   projectors, in: P. Van Dooren, S. P. Bhattacharyya, R. H. Chan, V.
%   Olshevsky, A.Routray (Eds.), Numerical Linear Algebra in Signals,
%   Systems and Control, Vol. 80 of Lect. Notes Electr. Eng., Springer
%   Netherlands, 2011, pp. 55--76.
%   https://doi.org/10.1007/978-94-007-0602-6_3
%
% See also ml_getqz, ml_signm.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 3);

if (nargin < 3) || isempty(opts)
    opts = struct();
end

% Check input matrices.
n = size(A, 1);

assert(isequal(size(A), [n n]), ...
    'MORLAB:data', ...
    'The matrix A has to be square!');

if issparse(A), A = full(A); end

if (nargin >= 2) && not(isempty(E))
    assert(isequal(size(E), [n n]), ...
        'MORLAB:data', ...
        'The matrix E must have the same dimensions as A!');

    if issparse(E), E = full(E); end
else
    E = eye(n);
end

% Check and assign optional parameters.
assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The parameter opts has to be a struct!');

if ml_field_set_to_value(opts, 'AbsTol')
    ml_assert_nonnegscalar(opts.AbsTol, 'opts.AbsTol');
else
    opts.AbsTol = 0;
end

if ml_field_set_to_value(opts, 'Info')
    ml_assert_boolean(opts.Info, 'opts.Info');
else
    opts.Info = false;
end

if ml_field_set_to_value(opts, 'MaxIter')
    ml_assert_posinteger(opts.MaxIter, 'opts.MaxIter');
else
    opts.MaxIter = 100;
end

if ml_field_set_to_value(opts, 'RelTol')
    ml_assert_nonnegscalar(opts.RelTol, 'opts.RelTol');
else
    opts.RelTol = 1.0e+02 * (n * eps);
end

% Case of empty data.
if isempty(A)
    Aspace = [];
    Espace = [];
    info   = struct([]);
    return;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INITIALIZATION.                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

niter  = 0;
Aspace = A;
Espace = E;
S      = eye(n);
abstol = opts.AbsTol;
reltol = opts.RelTol;

[abserr, relerr] = deal(zeros(1, opts.MaxIter));

% Set method to unconverged.
converged = 0;

% Adaptation of relative tolerance.
dif = min(svd([Aspace, Espace]));
if dif < reltol
    warning('MORLAB:data', ...
        'Matrix pencil is too close to singular pencil!');
end

reltol = reltol / min(1, dif);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% INVERSE FREE ITERATION.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

while (niter < opts.MaxIter) && not(converged)
    S2 = S;

    [U, S] = qr([Espace; -Aspace]);
    S      = S(1:n, :);
    S      = diag(sign(diag(S))) * S;

    % Update of iteration matrices.
    Aspace = U(1:n, n+1:2*n)' * Aspace;
    Espace = U(n+1:2*n, n+1:2*n)' * Espace;

    niter         = niter +  1;
    normS         = norm(S, 1);
    abserr(niter) = norm(S - S2, 1);
    relerr(niter) = abserr(niter) / normS;

    % Information about current iteration step.
    if opts.Info
        fprintf(1, ['DISK step: %4d absolute change: %e' ...
            ' relative change: %e \n'], ...
            niter, abserr(niter), relerr(niter));
    end

    % Method is converged if absolute or relative changes are small enough.
    converged = (abserr(niter) <= abstol) || (relerr(niter) <= reltol);
end

% Warning if iteration not converged.
if (niter == opts.MaxIter) && not(converged)
    warning('MORLAB:noConvergence', ...
        ['No convergence in %d iteration steps!\n' ...
        'Abs. tolerance: %e | Abs. change: %e\n' ...
        'Rel. tolerance: %e | Rel. change: %e\n' ...
        'Try to increase the tolerances or number of ' ...
        'iteration steps!'], ...
        niter, opts.AbsTol, min(abserr(niter)), ...
        opts.RelTol, min(relerr(niter)));
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN INFORMATION.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign information about iteration.
info = struct( ...
    'AbsErr'        , abserr(1:niter), ...
    'IterationSteps', niter, ...
    'RelErr'        , relerr(1:niter));
