function [V, W, hsv] = ml_balproj_soss(sys, R, L, rselect, opts)
%ML_BALPROJ_SOSS Compute projection bases for second-order balancing.
%
% SYNTAX:
%   [V, W, hsv] = ML_BALPROJ_SOSS(sys, R, L, rselect)
%   [V, W, hsv] = ML_BALPROJ_SOSS(sys, R, L, rselect, opts)
%
% DESCRIPTION:
%   This function computes projection bases for second-order balanced
%   truncation via the square-root or balancing-free square-root
%   method.
%
% INPUTS:
%   sys       - structure, containing the improper system in the form:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | M               | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   | E               | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   | K               | matrix with dimensions n x n                      |
%   +-----------------+---------------------------------------------------+
%   | Bu              | matrix with dimensions n x m                      |
%   +-----------------+---------------------------------------------------+
%   | Cp              | matrix with dimensions p x n, might be empty      |
%   +-----------------+---------------------------------------------------+
%   | Cv              | matrix with dimensions p x n, might be empty      |
%   +-----------------+---------------------------------------------------+
%   | D               | matrix with dimensions p x m, might be empty      |
%   +-----------------+---------------------------------------------------+
%   R       - Cholesky factor of the controllability Gramian with
%             dimensions 2*n x nr
%   L       - Cholesky factor of the observability Gramian with
%             dimensions 2*n x nl
%   rselect - integer, used to determine the computation method for the
%             order of the reduced-order model
%               0 - order is directly given by user
%               1 - computed by a relative tolerance for the hsv
%               2 - computed by a relative tolerance on the sum of hsv
%               3 - computed by absolute error bound of BT
%               4 - computed by relative error bound of BST
%               5 - computed by absolute error bound of LQGBT
%               6 - computed by absolute error bound of HinfBT
%   opts    - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | BalanceType     | character array, determining the type of          |
%   | {!}             | second-order balancing                            |
%   |                 |   'p'   - position balancing                      |
%   |                 |   'pm'  - position balancing (diagonalized M)     |
%   |                 |   'pv'  - position-velocity balancing             |
%   |                 |   'vp'  - velocity-position balancing             |
%   |                 |   'vpm' - velocity-position balancing (diag. M)   |
%   |                 |   'v'   - velocity balancing                      |
%   |                 |   'fv'  - free velocity balancing                 |
%   |                 |   'so'  - second-order balancing                  |
%   |                 | (default 'so')                                    |
%   +-----------------+---------------------------------------------------+
%   | Gamma           | positive scalar, scaling term from the H-infinity |
%   |                 | balanced truncation                               |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | rselect == 0                                      |
%   |                 | (default min(10,length(hsvp)) + nu + ni)          |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used in the         |
%   | {!}             | different error formulas                          |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   V   - right projection basis or struct of right projection bases in
%   {!}   case of opts.BalanceType = 'so' with entries V.Vp and V.Vv
%   W   - left projection matrix or struct of left projection bases in
%   {!}   case of opts.BalanceType = 'so' with entries W.Wp and W.Wv
%   hsv - vector or struct, containing the characteristic values
%   {!}
%
% REFERENCE:
%   S. W. R. Werner. Structure-Preserving Model Reduction for Mechanical
%   Systems. Dissertation, Otto-von-Guericke-Universität, Magdeburg,
%   Germany, 2021. doi:10.25673/38617
%
% See also mlprojtrunc_soss, ml_balproj_proper, ml_balproj_improper.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(4, 5);

if (nargin < 5) || isempty(opts)
    opts = struct();
end

assert(ml_field_set_to_value(sys, 'SystemType') ...
    && ml_field_set_to_value(sys, 'DecidedSystemType') ...
    && sys.DecidedSystemType, ...
    'MORLAB:data', ...
    'Run first ml_decide_system_type.');

assert(ml_field_set_to_value(sys, 'PreparedSystemData') ...
    && sys.PreparedSystemData, ...
    'MORLAB:data', ...
    'Run first ml_prepare_system_data.');

% Check system structure.
assert(strcmpi(sys.SystemType, 'ct_d_soss') ...
    ||strcmpi(sys.SystemType, 'ct_s_soss_so_1') ...
    || strcmpi(sys.SystemType, 'ct_s_soss_dae_1_so'), ...
    'MORLAB:data', ...
    'The given system cannot be balanced into second-order form.');

if ml_field_set_to_value(sys, 'nM')
    n = sys.nM(end);
else
    n = size(sys.K, 1);
end

rselect = ml_check_cell_param(rselect, 'rselect', @ml_assert_nonnegint);

% Check Gramian factors.
assert(isa(R, 'double') && (size(R, 1) == 2 * n), ...
    'MORLAB:data', ...
    'The matrix R must have double number of rows as sys.K!');

assert(isa(L, 'double') && (size(L, 1) == 2 * n), ...
    'MORLAB:data', ...
    'The matrix L must have double number of rows as sys.K!');

if issparse(R), R = full(R); end
if issparse(L), L = full(L); end

% Check and assign optional parameters.
opts = ml_check_cell_param(opts, 'Method', @ml_assert_char, 'sr');
opts = ml_check_cell_param(opts, 'Tolerance', ...
    @ml_assert_nonnegscalar, 1.0e-02);
opts = ml_check_cell_param(opts, 'Order', @ml_assert_posinteger, []);
opts = ml_check_cell_param(opts, 'BalanceType', @ml_assert_char, 'so');

[rselect, ...
    method, ...
    tolerance, ...
    order, ...
    balancetype] = ml_extend_cell(rselect, ...
    opts.Method, ...
    opts.Tolerance, ...
    opts.Order, ...
    opts.BalanceType);

numProjections = length(rselect);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TRANSFORM GRAMIAN FACTORS.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

one = 1:n;
two = n+1:2*n;

switch lower(sys.SystemType)
    case 'ct_d_soss'
        M = sys.M;

    case 'ct_s_soss_so_1'
        M         = sys.M;
        L(one, :) = -sys.K' * L(one, :);

    case 'ct_s_soss_dae_1_so'
        M = sys.M(one, one);
        L = [sys.E(one, one)' * L(one, :) + M' * L(two, :); L(one, :)];
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE PROJECTIONS.                                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

V   = cell(1, numProjections);
W   = cell(1, numProjections);
hsv = cell(1, numProjections);

for k = 1:numProjections
    switch lower(balancetype{k})
        case 'p'
            % Compute characteristic values.
            [~, S, T] = svd(L(one, :)' * R(one, :), 'econ');
            [U, ~, ~] = svd(L(two, :)' * (M * R(two, :)), ...
                'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    W{k} = (L(two, :) * U(:, 1:r)) * S;
                    V{k} = (R(one, :) * T(:, 1:r)) * S;

                case 'bfsr'
                    [W{k}, ~] = qr(L(two, :) * U(:, 1:r), 0);
                    [V{k}, ~] = qr(R(one, :) * T(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'pm'
            % Compute characteristic values.
            [U, S, T] = svd(L(one, :)' * R(one, :), 'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    W{k} = (M' \ (L(one, :) * U(:, 1:r))) * S;
                    V{k} = (R(one, :) * T(:, 1:r)) * S;

                case 'bfsr'
                    [W{k}, ~] = qr(M' \ (L(one, :) * U(:, 1:r)), 0);
                    [V{k}, ~] = qr(R(one, :) * T(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'pv'
            % Compute characteristic values.
            [U, S, T] = svd(L(two, :)' * (M * R(one, :)), 'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    W{k} = (L(two, :) * U(:, 1:r)) * S;
                    V{k} = (R(one, :) * T(:, 1:r)) * S;

                case 'bfsr'
                    [W{k}, ~] = qr(L(two, :) * U(:, 1:r), 0);
                    [V{k}, ~] = qr(R(one, :) * T(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'vp'
            % Compute characteristic values.
            [~, S, T] = svd(L(one, :)' * R(two, :), 'econ');
            [U, ~, ~] = svd(L(two, :)' * (M * R(one, :)), 'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    W{k} = (L(two, :) * U(:, 1:r)) * S;
                    V{k} = (R(two, :) * T(:, 1:r)) * S;

                case 'bfsr'
                    [W{k}, ~] = qr(L(two, :) * U(:, 1:r), 0);
                    [V{k}, ~] = qr(R(two, :) * T(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'vpm'
            % Compute characteristic values.
            [U, S, T] = svd(L(one, :)' * R(two, :), 'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    W{k} = (M' \ (L(one, :) * U(:, 1:r))) * S;
                    V{k} = (R(two, :) * T(:, 1:r)) * S;

                case 'bfsr'
                    [W{k}, ~] = qr(M' \ (L(one, :) * U(:, 1:r)), 0);
                    [V{k}, ~] = qr(R(two, :) * T(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'v'
            % Compute characteristic values.
            [U, S, T] = svd(L(two, :)' * (M * R(two, :)), 'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    W{k} = (L(two, :) * U(:, 1:r)) * S;
                    V{k} = (R(two, :) * T(:, 1:r)) * S;

                case 'bfsr'
                    [W{k}, ~] = qr(L(two, :) * U(:, 1:r), 0);
                    [V{k}, ~] = qr(R(two, :) * T(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'fv'
            % Compute characteristic values.
            [~, S, T] = svd(L(one, :)' * R(one, :), 'econ');
            hsv{k}    = diag(S);
            hsv{k}    = hsv{k}(1:min(length(hsv{k}), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r       = ml_order(hsv{k}, 0, rselect{k}, optstmp);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    S    = sparse(1:r, 1:r, 1 ./ sqrt(hsv{k}(1:r)));
                    V{k} = (R(one, :) * T(:, 1:r)) * S;
                    W{k} = V{k};

                case 'bfsr'
                    [V{k}, ~] = qr(R(one, :) * T(:, 1:r), 0);
                    W{k}      = V{k};

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

        case 'so'
            % Compute characteristic values.
            [Up, Sp, Vp] = svd(L(one, :)' * R(one, :), 'econ');
            hsvp         = diag(Sp);
            hsvp         = hsvp(1:min(length(hsvp), n));
            [Uv, Sv, Vv] = svd(L(two, :)' * (M * R(two, :)), 'econ');
            hsvv         = diag(Sv);
            hsvv         = hsvv(1:min(length(hsvv), n));

            % Compute reduced-order
            optstmp = struct('Tolerance', tolerance{k}, 'Order', order{k});
            r1      = ml_order(hsvp, 0, rselect{k}, optstmp);
            r2      = ml_order(hsvv, 0, rselect{k}, optstmp);
            r       = min(r1, r2);

            % Compute projection.
            switch lower(method{k})
                case 'sr'
                    Sp = sparse(1:r, 1:r, 1 ./ sqrt(hsvp(1:r)));
                    Wp = (L(one, :) * Up(:, 1:r)) * Sp;
                    Vp = (R(one, :) * Vp(:, 1:r)) * Sp;

                    Sv = sparse(1:r, 1:r, 1 ./ sqrt(hsvv(1:r)));
                    Wv = (L(two, :) * Uv(:, 1:r)) * Sv;
                    Vv = (R(two, :) * Vv(:, 1:r)) * Sv;

                case 'bfsr'
                    [Wp, ~] = qr(L(one, :) * Up(:, 1:r), 0);
                    [Vp, ~] = qr(R(one, :) * Vp(:, 1:r), 0);

                    [Wv, ~] = qr(L(two, :) * Uv(:, 1:r), 0);
                    [Vv, ~] = qr(R(two, :) * Vv(:, 1:r), 0);

                otherwise
                    error('MORLAB:data', ...
                        ['The requested balancing method is not' ...
                        ' implemented.']);
            end

            hsv{k} = struct('Hsvp', hsvp, 'Hsvv', hsvv);
            V{k}   = struct('Vp', Vp, 'Vv', Vv);
            W{k}   = struct('Wp', Wp, 'Wv', Wv);

        otherwise
            error('MORLAB:data', ...
                ['The requested balancing method is not' ...
                ' implemented!']);
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FORMAT OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if numProjections == 1
    hsv = hsv{:};
    W   = W{:};
    V   = V{:};
end
