function [varargout] = ml_dt_twostep_mor(varargin)
%ML_DT_TWOSTEP_MOR Discrete-time two-step model reduction method.
%
% SYNTAX:
%   [Ar, Br, Cr, Dr, info] = ML_DT_TWOSTEP_MOR(A, B, C, D)
%   [Ar, Br, Cr, Dr, info] = ML_DT_TWOSTEP_MOR(A, B, C, D, opts)
%
%   [Ar, Br, Cr, Dr, Er, info] = ML_DT_TWOSTEP_MOR(A, B, C, D, E)
%   [Ar, Br, Cr, Dr, Er, info] = ML_DT_TWOSTEP_MOR(A, B, C, D, E, opts)
%
%   [rom, info] = ML_DT_TWOSTEP_MOR(sys)
%   [rom, info] = ML_DT_TWOSTEP_MOR(sys, opts)
%
% DESCRIPTION:
%   This function computes the reduced-order model using a two-step
%   approach for a discrete-time sparse standard system of the form
%
%         x'(t) = A*x(t) + B*u(t),                                     (1a)
%          y(t) = C*x(t) + D*u(t),                                     (1b)
%
%   sparse descriptor system of the form
%
%       E*x'(t) = A*x(t) + B*u(t),                                     (2a)
%          y(t) = C*x(t) + D*u(t).                                     (2b)
%
%   The computed reduced-order models usually have the format of (1)
%   or (2), but for (3) there is the option to generate
%   structure-preserving reduced-order models.
%
% INPUTS:
%   A    - matrix from (1) or (2) with dimensions n x n
%   B    - matrix from (1) or (2) with dimensions n x m
%   C    - matrix from (1) or (2) with dimensions p x n
%   D    - matrix from (1) or (2) with dimensions p x m
%   E    - matrix from (1) or (2) with dimensions n x n
%   sys  - structure or state-space object (ss, dss, sparss),
%          containing the system's matrices
%   opts - structure, containing optional paramaters, the most important
%          are shown below, for details on parameters to change underlying
%          computational routines see *_twostep_mor functions:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | krylovopts      | struct, option structure for the Krylov           |
%   |                 | pre-reduction step; if not otherwise specified by |
%   |                 | the user, the number of expansion points is set   |
%   |                 | preserving one-sided projection bases of order    |
%   |                 | min(2*n, 1000), see ml_dt_s_foss_krylov           |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | KrylovROM       | struct, medium-scale reduced-order model that is  |
%   |                 | used instead of the Krylov pre-reduction step     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | MORMethod       | character array, determines the dense model       |
%   |                 | reduction method for the second step, possible    |
%   |                 | methods are:                                      |
%   |                 |  'brbt'   - bounded-real balanced truncation      |
%   |                 |  'bst'    - balanced stochastic truncation        |
%   |                 |  'bt'     - balanced truncation                   |
%   |                 |  'flbt'   - frequency-limited balanced truncation |
%   |                 |  'hinfbt' - H-infintiy balanced truncation        |
%   |                 |  'hna'    - Hankel-norm approximation             |
%   |                 |  'lqgbt'  - Linear-quadratic Gassian BT           |
%   |                 |  'mt'     - modal truncation                      |
%   |                 |  'prbt'   - positive-real balanced truncation     |
%   |                 |  'tlbt'   - time-limited balanced truncation      |
%   |                 | (default 'bt')                                    |
%   +-----------------+---------------------------------------------------+
%   | mormethodopts   | struct, option structure for the second reduction |
%   |                 | method specified by opts.MORMethod                |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreKrylovROM  | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed ROM from the Krylov pre-reduction step   |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   Ar   - output matrix for (1) or (2) with dimensions r x r
%   Br   - output matrix for (1) or (2) with dimensions r x m
%   Cr   - output matrix for (1) or (2) with dimensions p x r
%   Dr   - output matrix for (1) or (2) with dimensions p x m
%   Er   - output matrix for (1) or (2) with dimensions r x r
%   rom  - struct or state-space object (ss, dss), containing the
%          reduced-order system's matrices (naming of entries is identical
%          to input argument sys)
%   info - structure, containing the information about the computations and
%          models, depending on the system type more information might be
%          available, see *_twostep_mor routines for details:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoKRYLOV      | struct, information about the Krylov              |
%   |                 | pre-reduction step                                |
%   +-----------------+---------------------------------------------------+
%   | infoMORMETHOD   | struct, information about the dense model         |
%   |                 | reduction method                                  |
%   +-----------------+---------------------------------------------------+
%   | KrylovROM       | struct, reduced-order model from the Krylov       |
%   |                 | pre-reduction step, if opts.StoreKrylovROM == 1   |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_twostep_mor, ml_dt_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK AND REFORMAT INPUTS.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[sys, opts, ioformat] = ml_decide_system_type('dt', varargin{:});

if ioformat == 0
    % Check for maximum number of outputs in second-order case.
    nargoutchk(0, 8);
else
    % Struct or state-space object outputs.
    nargoutchk(0, 2);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SELECT BALANCED TRUNCATION ROUTINE.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start and end strings for system structures.
bstype = 'discrete-time ';
estr   = ' system';

% Selections of routines for different system structures.
switch lower(sys.SystemType)
    case {'dt_d_ss', 'dt_d_dss'}
        error('MORLAB:data', ...
            ['Two-step methods are not recommended for the reduction' ...
            ' of dense dynamical systems.\nPlease directly use your' ...
            ' model reduction method of choice instead.']);

    case 'dt_s_ss_default'
        [roms, info]    = ml_dt_s_foss_twostep_mor(sys, opts);
        info.SystemType = [bstype 'sparse standard state-space' estr];

    case 'dt_s_dss_default'
        [roms, info]    = ml_dt_s_foss_twostep_mor(sys, opts);
        info.SystemType = [bstype 'sparse generalized state-space' estr];

    case 'dt_s_dss_dae_1'
        [roms, info]    = ml_dt_s_foss_twostep_mor(sys, opts);
        info.SystemType = [bstype 'sparse index-1 descriptor' estr];

    case 'dt_s_dss_dae_2'
        [roms, info]    = ml_dt_s_foss_twostep_mor(sys, opts);
        info.SystemType = [bstype 'sparse index-2 descriptor' estr];

    case {'dt_d_soss', 'dt_s_soss_so_1', 'dt_s_soss_dae_1_so', ...
            'dt_s_soss_dae_2_so', 'dt_s_soss_dae_3_so'}
        error('MORLAB:notImplemented', ...
            ['Discrete-time second-order systems are currently not' ...
            ' implemented in MORLAB.']);

    otherwise
        error('MORLAB:data', 'Unknow system type!');
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isa(roms, 'cell')), roms = {roms}; end

[varargout{1:nargout}] = ml_format_output(roms, ioformat, info);
