function [varargout] = ml_dt_krylov(varargin)
%ML_DT_KRYLOV Discrete-time Krylov subspace method.
%
% SYNTAX:
%   [Ar, Br, Cr, Dr, info] = ML_DT_KRYLOV(A, B, C, D)
%   [Ar, Br, Cr, Dr, info] = ML_DT_KRYLOV(A, B, C, D, opts)
%
%   [Ar, Br, Cr, Dr, Er, info] = ML_DT_KRYLOV(A, B, C, D, E)
%   [Ar, Br, Cr, Dr, Er, info] = ML_DT_KRYLOV(A, B, C, D, E, opts)
%
%   [rom, info] = ML_DT_KRYLOV(sys)
%   [rom, info] = ML_DT_KRYLOV(sys, opts)
%
% DESCRIPTION:
%   This function computes the balanced truncation for a discrete-time
%   standard system of the form
%
%         x(t+1) = A*x(t) + B*u(t),                                    (1a)
%           y(t) = C*x(t) + D*u(t),                                    (1b)
%
%   descriptor system of the form
%
%       E*x(t+1) = A*x(t) + B*u(t),                                    (2a)
%           y(t) = C*x(t) + D*u(t).                                    (2b)
%
%   The computed reduced-order models have the same format as the systems
%   given as input.
%
% INPUTS:
%   A    - matrix from (1) or (2) with dimensions n x n
%   B    - matrix from (1) or (2) with dimensions n x m
%   C    - matrix from (1) or (2) with dimensions p x n
%   D    - matrix from (1) or (2) with dimensions p x m
%   E    - matrix from (1) or (2) with dimensions n x n
%   sys  - structure or state-space object (ss, dss, sparss),
%          containing the system's matrices
%   opts - structure, containing optional paramaters, the most important
%          are shown below, for details on parameters to change underlying
%          computational routines see *_krylov functions:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | krylovVopts     | struct, optional parameters to determine          |
%   |                 | construction of right Krylov basis, with the      |
%   |                 | optional fields:                                  |
%   |                 |   Directs  - matrix, tangential directions, set to|
%   |                 |              ones vector for matrix interpolation,|
%   |                 |              default setting changes to           |
%   |                 |              randn(m, NumPts) if inputs m > 4     |
%   |                 |              (default ones(1, NumPts))            |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |              frequency range, used as sampling    |
%   |                 |              range if Points are not given        |
%   |                 |              (default [-4, log10(pi)])            |
%   |                 |   NumPts   - positive integer, used as number of  |
%   |                 |              sampling points if Directs, Orders   |
%   |                 |              and Points are not set               |
%   |                 |              (default 50)                         |
%   |                 |   Orders   - vector, derivative orders            |
%   |                 |              (default ones(1, NumPts))            |
%   |                 |   Points   - vector, expansion points             |
%   |                 |              (default exp(1i*logspace(NumPts)))   |
%   |                 |   RealVal  - {0, 1}, turns off/on if basis is     |
%   |                 |              realified by splitting real and      |
%   |                 |              imaginary parts of vectors           |
%   |                 |              (default 1)                          |
%   +-----------------+---------------------------------------------------+
%   | krylovWopts     | struct, optional parameters to determine          |
%   |                 | construction of left Krylov basis, with the       |
%   |                 | optional fields:                                  |
%   |                 |   Directs  - matrix, tangential directions, set to|
%   |                 |              ones vector for matrix interpolation,|
%   |                 |              default setting changes to           |
%   |                 |              randn(m, NumPts) if outputs p > 4    |
%   |                 |              (default ones(1, NumPts))            |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |              frequency range, used as sampling    |
%   |                 |              range if Points are not given        |
%   |                 |              (default [-4, log10(pi)])            |
%   |                 |   NumPts   - positive integer, used as number of  |
%   |                 |              sampling points if Directs, Orders   |
%   |                 |              and Points are not set               |
%   |                 |              (default 50)                         |
%   |                 |   Orders   - vector, derivative orders            |
%   |                 |              (default ones(1, NumPts))            |
%   |                 |   Points   - vector, expansion points             |
%   |                 |              (default exp(1i*logspace(NumPts)))   |
%   |                 |   RealVal  - {0, 1}, turns off/on if basis is     |
%   |                 |              realified by splitting real and      |
%   |                 |              imaginary parts of vectors           |
%   |                 |              (default 1)                          |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the SVs,  |
%   |                 |  'sum'       - using rel. tolerance for sum of SVs|
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | StoreBases      | {0, 1}, used to disable/enable storing of the     |
%   |                 | uncompressed Krylov bases                         |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | TwoSidedProj    | {0, 1}, if turned on two-sided projection is used |
%   |                 | rather than one-sided projection                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | matrix, precomputed basis matrix right Krylov     |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | matrix, precomputed basis matrix left Krylov      |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   Ar   - output matrix for (1) or (2) with dimensions r x r
%   Br   - output matrix for (1) or (2) with dimensions r x m
%   Cr   - output matrix for (1) or (2) with dimensions p x r
%   Dr   - output matrix for (1) or (2) with dimensions p x m
%   Er   - output matrix for (1) or (2) with dimensions r x r
%   rom  - struct or state-space object (ss, dss), containing the
%          reduced-order system's matrices (naming of entries is identical
%          to input argument sys)
%   info - structure, containing the information about the computations and
%          models, depending on the system type more information might be
%          available, see *_krylov routines for details:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | SystemType      | character array, system structure that has been   |
%   |                 | determined for computations                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | Right Krylov basis of the controllability         |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | Left Krylov basis of the observability            |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_krylov, ml_dt_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK AND REFORMAT INPUTS.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[sys, opts, ioformat] = ml_decide_system_type('dt', varargin{:});

if ioformat == 0
    % Check for maximum number of outputs in second-order case.
    nargoutchk(0, 8);
else
    % Struct or state-space object outputs.
    nargoutchk(0, 2);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SELECT BALANCED TRUNCATION ROUTINE.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start and end strings for system structures.
bstype = 'discrete-time ';
estr   = ' system';

% Selections of routines for different system structures.
switch lower(sys.SystemType)
    case 'dt_d_ss'
        [roms, info]    = ml_dt_d_ss_krylov(sys, opts);
        info.SystemType = [bstype 'dense standard state-space' estr];

    case 'dt_d_dss'
        [roms, info]    = ml_dt_d_dss_krylov(sys, opts);
        info.SystemType = [bstype 'dense descriptor' estr];

    case 'dt_s_ss_default'
        [roms, info]    = ml_dt_s_foss_krylov(sys, opts);
        info.SystemType = [bstype 'sparse standard state-space' estr];

    case 'dt_s_dss_default'
        [roms, info]    = ml_dt_s_foss_krylov(sys, opts);
        info.SystemType = [bstype 'sparse generalized state-space' estr];

    case 'dt_s_dss_dae_1'
        [roms, info]    = ml_dt_s_foss_krylov(sys, opts);
        info.SystemType = [bstype 'sparse index-1 descriptor' estr];

    case 'dt_s_dss_dae_2'
        [roms, info]    = ml_dt_s_foss_krylov(sys, opts);
        info.SystemType = [bstype 'sparse index-2 descriptor' estr];

    case {'dt_d_soss', 'dt_s_soss_so_1', 'dt_s_soss_dae_1_so', ...
            'dt_s_soss_dae_2_so', 'dt_s_soss_dae_3_so'}
        error('MORLAB:notImplemented', ...
            ['Discrete-time second-order systems are currently not' ...
            ' implemented in MORLAB.']);

    otherwise
        error('MORLAB:data', 'Unknow system type!');
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isa(roms, 'cell')), roms = {roms}; end

[varargout{1:nargout}] = ml_format_output(roms, ioformat, info);
