function [rom, info] = ml_dt_d_ss_krylov(sys, opts)
%ML_DT_D_SS_KRYLOV Krylov subspace method for standard systems.
%
% SYNTAX:
%   [rom, info] = ML_DT_D_SS_KRYLOV(sys)
%   [rom, info] = ML_DT_D_SS_KRYLOV(sys, opts)
%
% DESCRIPTION:
%   This function computes Krylov subspace approximations for dense,
%   standard systems of the form
%
%       x(t+1) = A*x(t) + B*u(t),                                       (1)
%         y(t) = C*x(t) + D*u(t).                                       (2)
%
%   Therefore, shifted linear systems in expansion points are solved. As a
%   result a reduced-order system of the form
%
%       x(t+1) = Ar*x(t) + Br*u(t),                                     (3)
%         y(t) = Cr*x(t) + Dr*u(t)                                      (4)
%
%   is computed.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | krylovVopts     | struct, optional parameters to determine          |
%   |                 | construction of right Krylov basis, with the      |
%   |                 | optional fields:                                  |
%   |                 |   Directs - matrix, tangential directions, set to |
%   |                 |             ones vector for matrix interpolation, |
%   |                 |             default setting changes to            |
%   |                 |             randn(m, NumPts) if inputs m > 4      |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |             frequency range, used as sampling     |
%   |                 |             range if Points are not given         |
%   |                 |             (default [-4, log10(pi)])             |
%   |                 |   NumPts  - positive integer, used as number of   |
%   |                 |             sampling points if Directs, Orders    |
%   |                 |             and Points are not set                |
%   |                 |             (default 50)                          |
%   |                 |   Orders  - vector, derivative orders             |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   Points  - vector, expansion points              |
%   |                 |             (default exp(1i*logspace(NumPts)))    |
%   |                 |   RealVal - {0, 1}, turns off/on if basis is      |
%   |                 |             realified by splitting real and       |
%   |                 |             imaginary parts of vectors            |
%   |                 |             (default 1)                           |
%   +-----------------+---------------------------------------------------+
%   | krylovWopts     | struct, optional parameters to determine          |
%   |                 | construction of left Krylov basis, with the       |
%   |                 | optional fields:                                  |
%   |                 |   Directs - matrix, tangential directions, set to |
%   |                 |             ones vector for matrix interpolation, |
%   |                 |             default setting changes to            |
%   |                 |             randn(m, NumPts) if outputs p > 4     |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |             frequency range, used as sampling     |
%   |                 |             range if Points are not given         |
%   |                 |             (default [-4, log10(pi)])             |
%   |                 |   NumPts  - positive integer, used as number of   |
%   |                 |             sampling points if Directs, Orders    |
%   |                 |             and Points are not set                |
%   |                 |             (default 50)                          |
%   |                 |   Orders  - vector, derivative orders             |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   Points  - vector, expansion points              |
%   |                 |             (default exp(1i*logspace(NumPts)))    |
%   |                 |   RealVal - {0, 1}, turns off/on if basis is      |
%   |                 |             realified by splitting real and       |
%   |                 |             imaginary parts of vectors            |
%   |                 |             (default 1)                           |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(SVs)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the SVs,  |
%   |                 |  'sum'       - using rel. tolerance for sum of SVs|
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | StoreBases      | {0, 1}, used to disable/enable storing of the     |
%   |                 | uncompressed Krylov bases                         |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | TwoSidedProj    | {0, 1}, if turned on two-sided projection is used |
%   |                 | rather than one-sided projection                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | matrix, precomputed basis matrix right Krylov     |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | matrix, precomputed basis matrix left Krylov      |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order system:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | Right Krylov basis of the controllability         |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | Left Krylov basis of the observability            |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_d_ss_krylov, ml_dt_d_ss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('dt', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'dt_d_ss'
        % No extra action in main supported case.

    case 'dt_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to full.' ...
                ' Use ml_dt_s_foss_krylov to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_dt_s_foss_krylov to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'dt_d_ss';
sys            = ml_prepare_system_data(sys);

n = size(sys.A, 1);
m = size(sys.B, 2);
p = size(sys.C, 1);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'krylovVopts')
    assert(isa(opts.krylovVopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovVopts has to be a struct!');
else
    opts.krylovVopts = struct();
end

opts.krylovVopts = ml_check_krylovopts(opts.krylovVopts, m, 'dt');

if ml_field_set_to_value(opts, 'krylovWopts')
    assert(isa(opts.krylovWopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovWopts has to be a struct!');
else
    opts.krylovWopts = struct();
end

opts.krylovWopts = ml_check_krylovopts(opts.krylovWopts, p, 'dt');

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreBases')
    ml_assert_boolean(opts.StoreBases, 'opts.StoreBases');
else
    opts.StoreBases = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();

% Save sampling time.
if isfield(sys, 'Ts')
    Ts    = sys.Ts;
    hasTs = 1;
else
    hasTs = 0;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE KRYLOV BASES.                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if n > 0
    E = eye(n);

    % Right basis.
    if ml_field_set_to_value(opts, 'VBasis')
        Vbase = opts.VBasis;
    else
        tang = not(isempty(opts.krylovVopts.Directs));

        if tang
            upt = 1;
        else
            upt = m;
        end

        Vbase = zeros(n, sum(opts.krylovVopts.Orders) * upt);
        idx   = 1:upt;

        for k = 1:length(opts.krylovVopts.Points)
            % Current interpolation point.
            s = opts.krylovVopts.Points(k);

            % Solve linear systems.
            if tang
                Vbase(:, idx) = (s * E - sys.A) ...
                    \ (sys.B * opts.krylovVopts.Directs(:, k));
            else
                Vbase(:, idx) = (s * E - sys.A) \ sys.B;
            end

            % Solve Hermite systems.
            for j = 2:opts.krylovVopts.Orders(k)
                idx           = idx + upt;
                Vbase(:, idx) = (s * E - sys.A) \ Vbase(:, idx - upt);
            end

            % Update index range.
            idx = idx + upt;
        end
    end

    % Realification of basis.
    if opts.krylovVopts.RealVal
        R = [real(Vbase), imag(Vbase)];
    else
        R = Vbase;
    end

    % Left basis.
    if ml_field_set_to_value(opts, 'WBasis')
        Wbase = opts.WBasis;
    else
        tang = not(isempty(opts.krylovWopts.Directs));

        if tang
            upt = 1;
        else
            upt = p;
        end

        Wbase = zeros(n, sum(opts.krylovWopts.Orders) * upt);
        idx   = 1:upt;

        for k = 1:length(opts.krylovWopts.Points)
            % Current interpolation point.
            s = opts.krylovWopts.Points(k);

            % Solve linear systems.
            if tang
                Wbase(:, idx) = (s * E - sys.A)' ...
                    \ (sys.C' * opts.krylovWopts.Directs(:, k));
            else
                Wbase(:, idx) = (s * E - sys.A)' \ sys.C';
            end

            % Solve Hermite systems.
            for j = 2:opts.krylovWopts.Orders(k)
                idx           = idx + upt;
                Wbase(:, idx) = (s * E - sys.A)' \ Wbase(:, idx - upt);
            end

            % Update index range.
            idx = idx + upt;
        end
    end

    % Realification of basis.
    if opts.krylovWopts.RealVal
        L = [real(Wbase), imag(Wbase)];
    else
        L = Wbase;
    end
else
    [R, L, Vbase, Wbase] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


% Compute singular values of projection bases.
[V, W] = ml_domsubproj_proper(sys, R, L, 0, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce the system.
rom = ml_projtrunc_proper(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end

numRoms = length(rom);
if hasTs
    for k = 1:numRoms
        rom{k}.Ts = Ts;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Information about the method.
n = cell(1, numRoms);
for k = 1:numRoms
    n{k} = size(rom{k}.A, 1);
end
info.N = n;

% Store final projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreBases
    info.VBasis = Vbase;
    info.WBasis = Wbase;
else
    info.VBasis = [];
    info.WBasis = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
