function [rom, info] = ml_dt_d_dss_lqgbt(sys, opts)
%ML_DT_D_DSS_LQGBT LQG balanced truncation for discrete-time desc. systems.
%
% SYNTAX:
%   [rom, info] = ML_DT_D_DSS_LQGBT(sys)
%   [rom, info] = ML_DT_D_DSS_LQGBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the generalized linear-quadratic Gaussian
%   balanced truncation for a discrete-time descriptor system of the form
%
%       E*x(t+1) = A*x(t) + B*u(t),                                     (1)
%           y(t) = C*x(t) + D*u(t).                                     (2)
%
%   Therefore, first an additive decomposition of the system is performed
%   using the matrix disk function, such that
%
%            [ Ei  0 ]       [ Ai  0 ]                        [ Ci ]
%       E2 = [       ], A2 = [       ], B2 = [ Bi, Bp ], C2 = [    ],
%            [ 0  Ep ]       [ 0  Ap ]                        [ Cp ]
%
%   with (Ei, Ai, Bi, Ci, D) belonging to the polynomial part and
%   (Ep, Ap, Bp, Cp, 0) belonging to the strictly proper part.
%   Now, the two generalized discrete-time Riccati equations
%
%       Ap*Pp*Ap' - Ep*Pp*Ep' + Bp*Bp' - (Ap*Pp*Cp' + Bp*M')
%           * inv(Cp*Pp*Cp' + Rb) * (Ap*Pp*Cp' + Bp*M')' = 0,
%       Ap'*Qp*Ap - Ep*Qp*Ep' + Cp'*Cp - (Bp'*Qp*Ap + M'*Cp)'
%           * inv(Bp*Qp*Bp' + Rc) * (Bp'*Qp*Ap + M'*Cp) = 0
%
%   are solved for the reduction of the strictly proper part, with
%
%       Rb = I + M*M',
%       Rc = I + M'*M,
%
%   where M = D - Ci * inv(Ai) * Bi. Also, the two generalized
%   discrete-time Lyapunov equations
%
%       Ai*Pi*Ai' - Ei*Pi*Ei' - Bi*Bi' = 0,
%       Ai'*Qi*Ai - Ei'*Qi*Ei - Ci'*Ci = 0
%
%   are solved for the reduction of the polynomial part. As result, a
%   reduced-order system of the form
%
%       Er*x(t+1) = Ar*x(t) + Br*u(t),                                  (3)
%            y(t) = Cr*x(t) + Dr*u(t)                                   (4)
%
%   is computed, such that for the original transfer function G with the
%   (right) coprime factorization G = N*inv(M) and the reduced-order
%   transfer function Gr with an r-th order strictly proper part and the
%   (right) coprime factorization Gr = Nr*inv(Mr) it holds
%
%       ||[N; M] - [Nr; Mr]||_{\infty} <= 2 * (Hsvp(r+1)/sqrt(1
%           + Hsvp(r+1)^2) + ... + Hsvp(n)/sqrt(1 + Hsvp(n)^2)),
%
%   with Hsvp the vector containing the characteristic LQG singular values
%   of the system.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Beta            | nonnegative scalar, used as shift of the          |
%   |                 | in Bass' algorithm for better conditioning        |
%   |                 | (default 0.1)                                     |
%   +-----------------+---------------------------------------------------+
%   | dareopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see ml_daredl_sda_fac|
%   |                 | and ml_dare_nwt_fac                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest improper Hankel singular value of the     |
%   |                 | system to truncate the improper part              |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsvp)) + Nu + Ni)          |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'newton' - Newton iteration                      |
%   |                 |  'sda'    - dual structure-preserving doubling    |
%   |                 | (default 'sda')                                   |
%   +-----------------+---------------------------------------------------+
%   | stabdecopts     | structure, containing the optional parameters for |
%   |                 | the decomposition of the stable and unstable parts|
%   |                 | of the system using the sign function and subspace|
%   |                 | extraction method, only used if                   |
%   |                 | RiccatiSolver = 'newton',                         |
%   |                 | see ml_signm and ml_getqz                         |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabmethodopts  | structure, containing the optional parameters for |
%   |                 | the Smith iteration based Lyapunov equation solver|
%   |                 | used for the stabilization, only used if          |
%   |                 | RiccatiSolver = 'newton', see ml_dlyap_smith      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | (right) coprime factorization in H-infinity norm  |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | Hsvp            | a vector, containing the computed characteristic  |
%   |                 | LQG singular values of the proper part of the     |
%   |                 | system                                            |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoDARE        | structure, containing information about the       |
%   |                 | structure-preserving doubling solver for the dual |
%   |                 | Riccati equations, see ml_daredl_sda_fac          |
%   +-----------------+---------------------------------------------------+
%   | infoDAREC       | structure, containing information about the       |
%   |                 | Newton solver for the regulator Riccati equation, |
%   |                 | see ml_dare_nwt_fac                               |
%   +-----------------+---------------------------------------------------+
%   | infoDAREO       | structure, containing information about the       |
%   |                 | Newton solver for the filter Riccati equation,    |
%   |                 | see ml_dare_nwt_fac                               |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite, finite stable and finite anti-stable    |
%   |                 | parts, see ml_dt_dss_adtf                         |
%   +-----------------+---------------------------------------------------+
%   | infoGDLYAP      | structure, containing information about the       |
%   |                 | generalized discrete-time Lyapunov equation solver|
%   |                 | for the improper Gramians,                        |
%   |                 | see ml_gdlyapdl_smith_fac                         |
%   +-----------------+---------------------------------------------------+
%   | infoPARTSTABC   | structure, containing information about the       |
%   |                 | partial stabilization used for the controllability|
%   |                 | Riccati equation, see ml_dt_dss_partstab          |
%   +-----------------+---------------------------------------------------+
%   | infoPARTSTABO   | structure, containing information about the       |
%   |                 | partial stabilization used for the observability  |
%   |                 | Riccati equation, see ml_dt_dss_partstab          |
%   +-----------------+---------------------------------------------------+
%   | Nf              | Dimension of the finite reduced part in the       |
%   | {!}             | reduced-order model                               |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the improper part in the reduced-    |
%   | {!}             | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   M. R. Opmeer, R. F. Curtain, Linear quadratic Gaussian balancing for
%   discrete-time infinite-dimensional linear systems, SIAM J. Control
%   Optim., 43(4), 1196--1221. https://doi.org/10.1137/S0363012903431189
%
% See also ml_dt_d_ss_lqgbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('dt', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'dt_d_ss'
        warning('MORLAB:data', ...
            ['No descriptor matrix E found, set to identity E = I_n.' ...
            ' Use ml_dt_d_ss_lqgbt to handle standard systems.']);

        sys.E = eye(size(sys.A));

    case 'dt_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['No descriptor matrix E found, set to identity.', ...
                ' System matrices were converted from sparse to full.']);
            sys.E = eye(size(sys.A));
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse standard system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    case 'dt_d_dss'
        % No extra action in main supported case.

    case {'dt_s_dss_default', 'dt_s_dss_dae_1', 'dt_s_dss_dae_2'}
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                'System matrices were converted from sparse to full.');
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse descriptor system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'dt_d_dss';
sys            = ml_prepare_system_data(sys);

% Fill the D matrix.
if isempty(sys.D)
    sys.D = zeros(size(sys.C, 1), size(sys.B, 2));
end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'dareopts')
    assert(isa(opts.dareopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.dareopts has to be a struct!');
else
    opts.dareopts = struct();
end

if ml_field_set_to_value(opts, 'gdlyapopts')
    assert(isa(opts.gdlyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.gdlyapopts has to be a struct!');
else
    opts.gdlyapopts = struct();
end

if ml_field_set_to_value(opts, 'infdecopts')
    assert(isa(opts.infdecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.infdecopts has to be a struct!');
else
    opts.infdecopts = struct();
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 5;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'newton') ...
        || strcmpi(opts.RiccatiSolver, 'sda'), ...
        'MORLAB:data', ...
        'The desired Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'sda';
end

if ml_field_set_to_value(opts, 'stabdecopts')
    assert(isa(opts.stabdecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.stabdecopts has to be a struct!');
else
    opts.stabdecopts = struct();
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Handling of anti-stable dimension for decomposition or stabilization.
if ml_field_set_to_value(opts.stabdecopts, 'Dimension')
    ml_assert_integer(opts.stabdecopts.Dimension, ...
        'opts.stabdecopts.Dimension');
else
    opts.stabdecopts.Dimension = -1;
end

nu                         = opts.stabdecopts.Dimension;
opts.stabdecopts.Dimension = 0;

% Initial info structure.
info = struct();

% Save sampling time.
if isfield(sys, 'Ts')
    Ts    = sys.Ts;
    hasTs = 1;
else
    hasTs = 0;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into finite and infinite parts.
[sys, infoADTF] = ml_dt_d_dss_adtf(sys, opts);

ninf = size(sys.Ainf, 1);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PARTIAL STABILIZATION AND SOLVE MATRIX EQUATIONS.                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of the constant polynomial part.
if ninf > 0
    M = sys.D - sys.Cinf * (sys.Ainf \ sys.Binf);
else
    M = sys.D;
end

if size(sys.A, 1) > 0
    % Computation of matrices for Riccati equations.
    Rb = eye(size(sys.C, 1)) + M * M';
    Lb = chol(Rb);
    Rc = eye(size(sys.B, 2)) + M' * M;
    Lc = chol(Rc);

    A = sys.A - sys.B * M' * (Rb \ sys.C);
    B = sys.B / Lc;
    C = Lb' \ sys.C;

    if strcmpi(opts.RiccatiSolver, 'sign')
        if ml_field_set_to_value(opts, 'GramFacC') ...
                && ml_field_set_to_value(opts, 'GramFacO')
            % Both precomputed Gramian factors.
            R = opts.GramFacC;
            L = opts.GramFacO;
        elseif ml_field_set_to_value(opts, 'GramFacC')
            % Precomputed controllability factor.
            opts.careopts.EqnType = 'primal';
            [L, ~, infoDARE]      = ml_daredl_sda_fac(A, B, C, sys.E, ...
                -1, opts.careopts);
            R                     = opts.GramFacC;
            info.infoDARE         = infoDARE;
        elseif ml_field_set_to_value(opts, 'GramFacO')
            % Precomputed observability factor.
            opts.careopts.EqnType = 'dual';
            [~, R, infoDARE]      = ml_daredl_sda_fac(A, B, C, sys.E, ...
                -1, opts.careopts);
            L                     = opts.GramFacO;
            info.infoDARE         = infoDARE;
        else
            % No Gramian factors precomputed.
            opts.careopts.EqnType = 'both';
            [L, R, infoDARE]      = ml_daredl_sda_fac(A, B, C, sys.E, ...
                -1, opts.careopts);
            info.infoDARE         = infoDARE;
        end
    else
        opts.stabdecopts.Dimension = nu;
        opts.infdecopts.Dimension  = 0;

        if ml_field_set_to_value(opts, 'GramFacC') ...
                && ml_field_set_to_value(opts, 'GramFacO')
            % Both precomputed Gramian factors.
            R = opts.GramFacC;
            L = opts.GramFacO;
        elseif ml_field_set_to_value(opts, 'GramFacC')
            % Precomputed controllability factor.
            [K2, infoPARTSTABO] = ml_dt_d_dss_partstab(A, B, sys.E, opts);
            info.infoPARTSTABO  = infoPARTSTABO;

            opts.careopts.K0 = K2;
            [L, infoDAREO]   = ml_dare_nwt_fac(A, B, C, sys.E, ...
                opts.careopts);
            info.infoDAREO   = infoDAREO;

            R = opts.GramFacC;
        elseif ml_field_set_to_value(opts, 'GramFacO')
            % Precomputed observability factor.
            [K1, infoPARTSTABC] = ml_dt_d_dss_partstab(A', C', sys.E', ...
                opts);
            info.infoPARTSTABC  = infoPARTSTABC;

            opts.careopts.K0 = K1;
            [R, infoDAREC]   = ml_dare_nwt_fac(A', C', B', sys.E', ...
                opts.careopts);
            info.infoDAREC   = infoDAREC;

            L = opts.GramFacO;
        else
            % No Gramian factors precomputed.
            [K1, infoPARTSTABC] = ml_dt_d_dss_partstab(A', C', sys.E', ...
                opts);
            info.infoPARTSTABC  = infoPARTSTABC;

            opts.careopts.K0 = K1;
            [R, infoDAREC]   = ml_dare_nwt_fac(A', C', B', sys.E', ...
                opts.careopts);
            info.infoDAREC   = infoDAREC;

            [K2, infoPARTSTABO] = ml_dt_d_dss_partstab(A, B, sys.E, opts);
            info.infoPARTSTABO  = infoPARTSTABO;

            opts.careopts.K0 = K2;
            [L, infoDAREO]   = ml_dare_nwt_fac(A, B, C, sys.E, ...
                opts.careopts);
            info.infoDAREO   = infoDAREO;
        end
    end
else
    [R, L] = deal([]);
end

% Computation of the full-rank factors of the improper Gramians.
if ninf > 0
    if ml_field_set_to_value(opts, 'IGramFacC') ...
            && ml_field_set_to_value(opts, 'IGramFacO')
        % Both precomputed Gramian factors.
        Rinf = opts.IGramFacC;
        Linf = opts.IGramFacO;
    elseif ml_field_set_to_value(opts, 'IGramFacC')
        % Precomputed controllability factor.
        [Linf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf', ...
            sys.Cinf', sys.Einf', opts.gdlyapopts);
        Rinf               = opts.IGramFacC;
        info.infoGDLYAP    = infoGDLYAP;
    elseif ml_field_set_to_value(opts, 'IGramFacO')
        % Precomputed observability factor.
        [Rinf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf, ...
            sys.Binf, sys.Einf, opts.gdlyapopts);
        Linf               = opts.IGramFacO;
        info.infoGDLYAP    = infoGDLYAP;
    else
        % No Gramian factors precomputed.
        [Rinf, Linf, infoGDLYAP] = ml_gdlyapdl_smith_fac( ...
            sys.Ainf, sys.Binf, sys.Cinf, sys.Einf, opts.gdlyapopts);
        info.infoGDLYAP          = infoGDLYAP;
    end
else
    [Rinf, Linf] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection bases.
[Vinf, Winf, hsvi] = ml_balproj_improper(sys, Rinf, Linf, opts);

if not(isa(Vinf, 'cell')), Vinf = {Vinf}; end
if not(isa(Winf, 'cell')), Winf = {Winf}; end

[V, W, hsvp] = ml_balproj_proper(sys, R, L, ...
    size(sys.Au, 1) + size(Vinf{1}, 2), rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

[V, W, Vinf, Winf] = ml_extend_cell(V, W, Vinf, Winf);

% Compute reduced-order model parts.
stabrom = ml_projtrunc_proper(sys, V, W);
infrom  = ml_projtrunc_improper(sys, Vinf, Winf);

% Construct final ROMs from stable, unstable and improper parts.
if not(isa(stabrom, 'cell')), stabrom = {stabrom}; end
if not(isa(infrom, 'cell')), infrom = {infrom}; end

numRoms = length(stabrom);
rom     = cell(1, numRoms);
for k = 1:numRoms
    rom{k}.A = blkdiag(stabrom{k}.A, infrom{k}.A);
    rom{k}.B = [stabrom{k}.B; infrom{k}.B];
    rom{k}.C = [stabrom{k}.C, infrom{k}.C];
    rom{k}.D = sys.D;
    rom{k}.E = blkdiag(stabrom{k}.E, infrom{k}.E);

    if hasTs
        rom{k}.Ts = Ts;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
nf          = cell(1, numRoms);
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    nf{k}          = size(stabrom{k}.A, 1);
    absErrBound{k} = 2 * sum(hsvp(nf{k}+1:end) ...
        ./ sqrt(1 + hsvp(nf{k}+1:end).^2));

    if hasTs
        rom{k}.Ts = Ts;
    end
end

info.AbsErrBound = absErrBound;
info.Hsvi        = hsvi;
info.Hsvp        = hsvp;
info.Nf          = nf;
info.Ni          = size(infrom{1}.A, 1);

% Store projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = infoADTF.V * blkdiag(V{k}, Vinf{k});
        info.W{k} = infoADTF.W * blkdiag(W{k}, Winf{k});
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC  = R;
    info.GramFacO  = L;
    info.IGramFacC = Rinf;
    info.IGramFacO = Linf;
else
    info.GramFacC  = [];
    info.GramFacO  = [];
    info.IGramFacC = [];
    info.IGramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
