function [rom, info] = ml_dt_d_dss_krylov(sys, opts)
%ML_DT_D_DSS_KRYLOV Krylov subspace method for dense descriptor systems.
%
% SYNTAX:
%   [rom, info] = ML_DT_D_DSS_KRYLOV(sys)
%   [rom, info] = ML_DT_D_DSS_KRYLOV(sys, opts)
%
% DESCRIPTION:
%   This function computes Krylov subspace approximations for dense,
%   discrete-time descriptor system of the form
%
%       E*x(t+1) = A*x(t) + B*u(t),                                     (1)
%           y(t) = C*x(t) + D*u(t).                                     (2)
%
%   Therefore, first an additive decomposition of the system is performed
%   using the matrix disk function, such that
%
%            [ Ei  0 ]       [ Ai  0 ]                        [ Ci ]
%       E2 = [       ], A2 = [       ], B2 = [ Bi, Bp ], C2 = [    ],
%            [ 0  Ep ]       [ 0  Ap ]                        [ Cp ]
%
%   with (Ei, Ai, Bi, Ci, D) belonging to the polynomial part and
%   (Ep, Ap, Bp, Cp, 0) belonging to the strictly proper part. Then,
%   shifted linear systems in expansion points are solved on the strictly
%   proper system part. As a result a reduced-order system of the form
%
%       Er*x(t+1) = Ar*x(t) + Br*u(t),                                  (3)
%            y(t) = Cr*x(t) + Dr*u(t)                                   (4)
%
%   is computed.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest improper Hankel singular value of the     |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | krylovVopts     | struct, optional parameters to determine          |
%   |                 | construction of right Krylov basis, with the      |
%   |                 | optional fields:                                  |
%   |                 |   Directs - matrix, tangential directions, set to |
%   |                 |             ones vector for matrix interpolation, |
%   |                 |             default setting changes to            |
%   |                 |             randn(m, NumPts) if inputs m > 4      |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |             frequency range, used as sampling     |
%   |                 |             range if Points are not given         |
%   |                 |             (default [-4, log10(pi)])             |
%   |                 |   NumPts  - positive integer, used as number of   |
%   |                 |             sampling points if Directs, Orders    |
%   |                 |             and Points are not set                |
%   |                 |             (default 50)                          |
%   |                 |   Orders  - vector, derivative orders             |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   Points  - vector, expansion points              |
%   |                 |             (default exp(1i*logspace(NumPts)))    |
%   |                 |   RealVal - {0, 1}, turns off/on if basis is      |
%   |                 |             realified by splitting real and       |
%   |                 |             imaginary parts of vectors            |
%   |                 |             (default 1)                           |
%   +-----------------+---------------------------------------------------+
%   | krylovWopts     | struct, optional parameters to determine          |
%   |                 | construction of left Krylov basis, with the       |
%   |                 | optional fields:                                  |
%   |                 |   Directs - matrix, tangential directions, set to |
%   |                 |             ones vector for matrix interpolation, |
%   |                 |             default setting changes to            |
%   |                 |             randn(m, NumPts) if outputs p > 4     |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   FreqRange - vector of length 2, exponents of    |
%   |                 |             frequency range, used as sampling     |
%   |                 |             range if Points are not given         |
%   |                 |             (default [-4, log10(pi)])             |
%   |                 |   NumPts  - positive integer, used as number of   |
%   |                 |             sampling points if Directs, Orders    |
%   |                 |             and Points are not set                |
%   |                 |             (default 50)                          |
%   |                 |   Orders  - vector, derivative orders             |
%   |                 |             (default ones(1, NumPts))             |
%   |                 |   Points  - vector, expansion points              |
%   |                 |             (default exp(1i*logspace(NumPts)))    |
%   |                 |   RealVal - {0, 1}, turns off/on if basis is      |
%   |                 |             realified by splitting real and       |
%   |                 |             imaginary parts of vectors            |
%   |                 |             (default 1)                           |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(SVs)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the SVs,  |
%   |                 |  'sum'       - using rel. tolerance for sum of SVs|
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | StoreBases      | {0, 1}, used to disable/enable storing of the     |
%   |                 | uncompressed Krylov bases adn improper Gramians   |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | TwoSidedProj    | {0, 1}, if turned on two-sided projection is used |
%   |                 | rather than one-sided projection                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | matrix, precomputed basis matrix right Krylov     |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | matrix, precomputed basis matrix left Krylov      |
%   |                 | subspace                                          |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order descriptor system:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite, finite stable and finite anti-stable    |
%   |                 | parts, see ml_dt_d_dss_adtf                       |
%   +-----------------+---------------------------------------------------+
%   | infoGDLYAP      | structure, containing information about the       |
%   |                 | generalized discrete-time Lyapunov equation solver|
%   |                 | for the improper Gramians,                        |
%   |                 | see ml_gdlyapdl_smith_fac                         |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the improper part in the reduced-    |
%   | {!}             | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Np              | Dimension of the proper part in the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | VBasis          | Right Krylov basis of the controllability         |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | WBasis          | Left Krylov basis of the observability            |
%   |                 | subspace, if opts.StoreGramians == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_d_dss_krylov, ml_dt_d_dss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('dt', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'dt_d_ss'
        warning('MORLAB:data', ...
            ['No descriptor matrix E found, set to identity E = I_n.' ...
            ' Use ml_dt_d_ss_krylov to handle standard systems.']);

        sys.E = eye(size(sys.A));

    case 'dt_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['No descriptor matrix E found, set to identity.', ...
                ' System matrices were converted from sparse to full.' ...
                ' Use ml_dt_s_foss_krylov to handle sparse systems.']);
            sys.E = eye(size(sys.A));
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_dt_s_foss_krylov to handle such systems.']);
        end

    case 'dt_d_dss'
        % No extra action in main supported case.

    case {'dt_s_dss_default', 'dt_s_dss_dae_1', 'dt_s_dss_dae_2'}
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to full.' ...
                ' Use ml_dt_s_foss_krylov to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse descriptor system detected.' ...
                ' Use ml_dt_s_foss_krylov to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'dt_d_dss';
sys            = ml_prepare_system_data(sys);

m = size(sys.B, 2);
p = size(sys.C, 1);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'gdlyapopts')
    assert(isa(opts.gdlyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.gdlyapopts has to be a struct!');
else
    opts.gdlyapopts = struct();
end

if ml_field_set_to_value(opts, 'krylovVopts')
    assert(isa(opts.krylovVopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovVopts has to be a struct!');
else
    opts.krylovVopts = struct();
end

opts.krylovVopts = ml_check_krylovopts(opts.krylovVopts, m, 'dt');

if ml_field_set_to_value(opts, 'krylovWopts')
    assert(isa(opts.krylovWopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovWopts has to be a struct!');
else
    opts.krylovWopts = struct();
end

opts.krylovWopts = ml_check_krylovopts(opts.krylovWopts, p, 'dt');

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreBases')
    ml_assert_boolean(opts.StoreBases, 'opts.StoreBases');
else
    opts.StoreBases = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();

% Save sampling time.
if isfield(sys, 'Ts')
    Ts    = sys.Ts;
    hasTs = 1;
else
    hasTs = 0;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into unstable, finite and infinite parts.
opts.stabdecopts = struct('Dimension', 0);
[sys, infoADTF]  = ml_dt_d_dss_adtf(sys, opts);

ninf = size(sys.Ainf, 1);
n    = size(sys.A, 1);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE LYAPUNOV EQUATIONS.                                               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of the full-rank factors of the improper Gramians.
if ninf > 0
    if ml_field_set_to_value(opts, 'IGramFacC') ...
            && ml_field_set_to_value(opts, 'IGramFacO')
        % Both precomputed Gramian factors.
        Rinf = opts.IGramFacC;
        Linf = opts.IGramFacO;
    elseif ml_field_set_to_value(opts, 'IGramFacC')
        % Precomputed controllability factor.
        [Linf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf', ...
            sys.Cinf', sys.Einf', opts.gdlyapopts);
        Rinf               = opts.IGramFacC;
        info.infoGDLYAP    = infoGDLYAP;
    elseif ml_field_set_to_value(opts, 'IGramFacO')
        % Precomputed observability factor.
        [Rinf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf, ...
            sys.Binf, sys.Einf, opts.gdlyapopts);
        Linf               = opts.IGramFacO;
        info.infoGDLYAP    = infoGDLYAP;
    else
        % No Gramian factors precomputed.
        [Rinf, Linf, infoGDLYAP] = ml_gdlyapdl_smith_fac( ...
            sys.Ainf, sys.Binf, sys.Cinf, sys.Einf, opts.gdlyapopts);
        info.infoGDLYAP          = infoGDLYAP;
    end
else
    [Rinf, Linf] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE KRYLOV BASES.                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Right basis.
if n > 0
    if ml_field_set_to_value(opts, 'VBasis')
        Vbase = opts.VBasis;
    else
        tang = not(isempty(opts.krylovVopts.Directs));

        if tang
            upt = 1;
        else
            upt = m;
        end

        Vbase = zeros(n, sum(opts.krylovVopts.Orders) * upt);
        idx   = 1:upt;

        for k = 1:length(opts.krylovVopts.Points)
            % Current interpolation point.
            s = opts.krylovVopts.Points(k);

            % Solve linear systems.
            if tang
                Vbase(:, idx) = (s * sys.E - sys.A) ...
                    \ (sys.B * opts.krylovVopts.Directs(:, k));
            else
                Vbase(:, idx) = (s * sys.E - sys.A) \ sys.B;
            end

            % Solve Hermite systems.
            for j = 2:opts.krylovVopts.Orders(k)
                idx           = idx + upt;
                Vbase(:, idx) = (s * sys.E - sys.A) ...
                    \ (sys.E * Vbase(:, idx - upt));
            end

            % Update index range.
            idx = idx + upt;
        end
    end

    % Realification of basis.
    if opts.krylovVopts.RealVal
        R = [real(Vbase), imag(Vbase)];
    else
        R = Vbase;
    end

    % Left basis.
    if ml_field_set_to_value(opts, 'WBasis')
        Wbase = opts.WBasis;
    else
        tang = not(isempty(opts.krylovWopts.Directs));

        if tang
            upt = 1;
        else
            upt = p;
        end

        Wbase = zeros(n, sum(opts.krylovWopts.Orders) * upt);
        idx   = 1:upt;

        for k = 1:length(opts.krylovWopts.Points)
            % Current interpolation point.
            s = opts.krylovWopts.Points(k);

            % Solve linear systems.
            if tang
                Wbase(:, idx) = (s * sys.E - sys.A)' ...
                    \ (sys.C' * opts.krylovWopts.Directs(:, k));
            else
                Wbase(:, idx) = (s * sys.E - sys.A)' \ sys.C';
            end

            % Solve Hermite systems.
            for j = 2:opts.krylovWopts.Orders(k)
                idx           = idx + upt;
                Wbase(:, idx) = (s * sys.E - sys.A)' ...
                    \ (sys.E' * Wbase(:, idx - upt));
            end

            % Update index range.
            idx = idx + upt;
        end
    end

    % Realification of basis.
    if opts.krylovWopts.RealVal
        L = [real(Wbase), imag(Wbase)];
    else
        L = Wbase;
    end
else
    [R, L, Vbase, Wbase] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection bases.
[Vinf, Winf, hsvi] = ml_balproj_improper(sys, Rinf, Linf, opts);

if not(isa(Vinf, 'cell')), Vinf = {Vinf}; end
if not(isa(Winf, 'cell')), Winf = {Winf}; end

[V, W] = ml_domsubproj_proper(sys, R, L, ninf, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

[V, W, Vinf, Winf] = ml_extend_cell(V, W, Vinf, Winf);

% Reduce the system.
romfin = ml_projtrunc_proper(sys, V, W);
infrom  = ml_projtrunc_improper(sys, Vinf, Winf);

% Construct final ROMs from proper and improper parts.
if not(isa(romfin, 'cell')), romfin = {romfin}; end
if not(isa(infrom, 'cell')), infrom = {infrom}; end

numRoms = length(romfin);
rom     = cell(1, numRoms);
for k = 1:numRoms
    rom{k}.A = blkdiag(romfin{k}.A, infrom{k}.A);
    rom{k}.B = [romfin{k}.B; infrom{k}.B];
    rom{k}.C = [romfin{k}.C, infrom{k}.C];
    rom{k}.D = sys.D;
    rom{k}.E = blkdiag(romfin{k}.E, infrom{k}.E);

    if hasTs
        rom{k}.Ts = Ts;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Information about the method.
numRoms = length(rom);
np      = cell(1, numRoms);
for k = 1:numRoms
    np{k} = size(romfin{k}.A, 1);
end

info.Hsvi = hsvi;
info.Np   = np;
info.Ni   = size(infrom{1}.A, 1);

% Store final projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = infoADTF.V * blkdiag(V{k}, Vinf{k});
        info.W{k} = infoADTF.W * blkdiag(W{k}, Winf{k});
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreBases
    info.VBasis    = Vbase;
    info.WBasis    = Wbase;
    info.IGramFacC = Rinf;
    info.IGramFacO = Linf;
else
    info.VBasis = [];
    info.WBasis = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
