function [rom, info] = ml_ct_s_soss_twostep_mor(sys, opts)
%ML_CT_S_SOSS_TWOSTEP_MOR Two-step MOR methods for second-order systems.
%
%
% SYNTAX:
%   [rom, info] = ML_CT_S_SOSS_TWOSTEP_MOR(sys)
%   [rom, info] = ML_CT_S_SOSS_TWOSTEP_MOR(sys, opts)
%
% DESCRIPTION:
%   This function computes allows to compute reduced-order models of
%   sparse, second-order systems of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   via a two-step process. In the first step, an accurate medium-scale
%   reduction is computed using Krylov subspace methods. In the second
%   step, dense model reduction methods are applied to the intermediate
%   model to compute reduced-order models of the form
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   or second-order form
%
%       Mr*x''(t) = -Kr*x(t) -  Er*x'(t) + Bur*u(t)                     (5)
%            y(t) = Cpr*x(t) + Cvr*x'(t) +  Dr*u(t)                     (6)
%
% INPUTS:
%   sys  - structure, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | krylovopts      | struct, option structure for the Krylov           |
%   |                 | pre-reduction step; if not otherwise specified by |
%   |                 | the user, the number of expansion points is set   |
%   |                 | preserving one-sided projection bases of order    |
%   |                 | min(2*n, 1000), see ml_ct_s_soss_krylov           |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | KrylovROM       | struct, medium-scale reduced-order model that is  |
%   |                 | used instead of the Krylov pre-reduction step     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | MORMethod       | character array, determines the dense model       |
%   |                 | reduction method for the second step, possible    |
%   |                 | methods are:                                      |
%   |                 |  'brbt'   - bounded-real balanced truncation      |
%   |                 |  'bst'    - balanced stochastic truncation        |
%   |                 |  'bt'     - balanced truncation                   |
%   |                 |  'flbt'   - frequency-limited balanced truncation |
%   |                 |  'hinfbt' - H-infintiy balanced truncation        |
%   |                 |  'hna'    - Hankel-norm approximation             |
%   |                 |  'lqgbt'  - Linear-quadratic Gassian BT           |
%   |                 |  'mt'     - modal truncation                      |
%   |                 |  'prbt'   - positive-real balanced truncation     |
%   |                 |  'tlbt'   - time-limited balanced truncation      |
%   |                 | (default 'bt')                                    |
%   +-----------------+---------------------------------------------------+
%   | mormethodopts   | struct, option structure for the second reduction |
%   |                 | method specified by opts.MORMethod                |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreKrylovROM  | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed ROM from the Krylov pre-reduction step   |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order model, with the
%   {!}    following entries if opts.mormethodopts.OutputModel = 'fo'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%          the following entries if opts.mormethodopts.OutputModel = 'so'
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (5) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (5) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (6) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (6) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoKRYLOV      | struct, information about the Krylov              |
%   |                 | pre-reduction step                                |
%   +-----------------+---------------------------------------------------+
%   | infoMORMETHOD   | struct, information about the dense model         |
%   |                 | reduction method                                  |
%   +-----------------+---------------------------------------------------+
%   | KrylovROM       | struct, reduced-order model from the Krylov       |
%   |                 | pre-reduction step, if opts.StoreKrylovROM == 1   |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_s_foss_twostep_mor, ml_ct_s_soss_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)        
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        % No extra action in main supported case.
        
        % TODO: Remove when M-M.E.S.S. is fixed.
        assert(not(strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')) ...
            && not(strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')), ...
            'MORLAB:notImplemented', ...
            'This system structure is not yet supported.');
        
    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

% Check and assign optional MORLAB parameters.
if ml_field_set_to_value(opts, 'krylovopts')
    assert(isa(opts.krylovopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.krylovopts has to be a struct!');
else
    opts.krylovopts = struct();
end

if ml_field_set_to_value(opts, 'KrylovROM')
    assert(isa(opts.KrylovROM, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.KrylovROM has to be a struct!');
else
    opts.KrylovROM = struct();
end

if ml_field_set_to_value(opts, 'MORMethod')
    ml_assert_char(opts.MORMethod, 'opts.MORMethod');

    assert(strcmpi(opts.MORMethod, 'brbt') || ...
        strcmpi(opts.MORMethod, 'bst') || ...
        strcmpi(opts.MORMethod, 'bt') || ...
        strcmpi(opts.MORMethod, 'flbt') || ...
        strcmpi(opts.MORMethod, 'hinfbt') || ...
        strcmpi(opts.MORMethod, 'hna') || ...
        strcmpi(opts.MORMethod, 'lqgbt') || ...
        strcmpi(opts.MORMethod, 'mt') || ...
        strcmpi(opts.MORMethod, 'prbt') || ...
        strcmpi(opts.MORMethod, 'tlbt'), ...
        'MORLAB:data', ...
        ['The requested model reduction method is either not' ...
        ' implemented or not permissible for two-step reduction.']);
else
    opts.MORMethod = 'bt';
end

if ml_field_set_to_value(opts, 'mormethodopts')
    assert(isa(opts.mormethodopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.mormethodopts has to be a struct!');
else
    opts.mormethodopts = struct();
end

if ml_field_set_to_value(opts, 'StoreKrylovROM')
    ml_assert_boolean(opts.StoreKrylovROM, 'opts.StoreKrylovROM');
else
    opts.StoreKrylovROM = false;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% KRYLOV PRE-REDUCTION.                                                   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if ml_field_set_to_value(opts, 'KrylovROM') ...
        && not(isempty(opts.KrylovROM) ...
        || isempty(fieldnames(opts.KrylovROM)))
    krylovrom       = opts.KrylovROM;
    info.infoKRYLOV = [];
else
    n = size(sys.K, 1);
    m = size(sys.pB, 2);
    p = size(sys.pC, 1);

    if isempty(opts.krylovopts) || isempty(fieldnames(opts.krylovopts))
        if m > 4
            incols = 1;
        else
            incols = m;
        end

        if p > 4
            outcols = 1;
        else
            outcols = p;
        end

        numcols = min(2 * n, 1000);
        numpts  = ceil(numcols / (2 * incols + 2 * outcols));

        opts.krylovopts.krylovVopts      = struct( ...
            'NumPts' , numpts, ...
            'RealVal', true);
        opts.krylovopts.krylovWopts      = struct( ...
            'NumPts' , numpts, ...
            'RealVal', true);
        opts.krylovopts.TwoSidedProj     = false;
        opts.krylovopts.OrderComputation = 'tolerance';
        opts.krylovopts.Tolerance        = n * eps;
        opts.krylovopts.OutputModel      = 'so';
    end

    [krylovrom, krylovinfo] = ml_ct_s_soss_krylov(sys, opts.krylovopts);
    info.infoKRYLOV         = krylovinfo;    
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MAIN REDUCTION.                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

switch lower(opts.MORMethod)
    case 'brbt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_brbt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_brbt(krylovrom, opts.mormethodopts);
        end
    case 'bst'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_bst(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_bst(krylovrom, opts.mormethodopts);
        end
    case 'bt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_bt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_bt(krylovrom, opts.mormethodopts);
        end
    case 'flbt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_flbt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_flbt(krylovrom, opts.mormethodopts);
        end
    case 'hinfbt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_hinfbt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_hinfbt(krylovrom, opts.mormethodopts);
        end
    case 'hna'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_hna(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_hna(krylovrom, opts.mormethodopts);
        end
    case 'lqgbt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_lqgbt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_lqgbt(krylovrom, opts.mormethodopts);
        end
    case 'mt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_mt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_mt(krylovrom, opts.mormethodopts);
        end
    case 'prbt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_prbt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_prbt(krylovrom, opts.mormethodopts);
        end
    case 'tlbt'
        if isfield(krylovrom, 'A')
            [rom, morinfo] = ...
                ml_ct_d_dss_tlbt(krylovrom, opts.mormethodopts);
        else
            [rom, morinfo] = ...
                ml_ct_d_soss_tlbt(krylovrom, opts.mormethodopts);
        end
    otherwise
        error('MORLAB:data', ...
            ['The requested model reduction method is either not' ...
            ' implemented or not permissible for two-step reduction.']);
end

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
info.infoMORMETHOD = morinfo;

% Store Gramian factors.
if opts.StoreKrylovROM
    info.KrylovROM = krylovrom;
else
    info.KrylovROM = struct();
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
