function [rom, info] = ml_ct_s_soss_hna(sys, opts)
%ML_CT_S_SOSS_HNA Hankel-norm approximation for sparse sec.-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_SOSS_HNA(sys)
%   [rom, info] = ML_CT_S_SOSS_HNA(sys, opts)
%
% DESCRIPTION:
%   This function computes the Hankel-norm approximation for a sparse,
%   standard or descriptor system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t)                         (1)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t)                         (2)
%
%   Therefore, first a balanced realization is computed by using the
%   balanced truncation square-root method with an appropriate tolerance
%   for the minimal realization of the given system. Then the system is
%   transformed using the formulas given in the reference below. As
%   result, a reduced-order system of the form
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   is computed, such that for the original transfer function G and the
%   r-th order transfer function Gr it holds
%
%       ||G - Gr||_{H}       = Hsv(r+1),
%       ||G - Gr||_{\infty} <= 2 * (Hsv(r+1) + ... + Hsv(n)),
%
%   with Hsv, a vector containing the Hankel singular values of the system.
%
%   Note: The system is assumed to be asymptotically stable.
%
% INPUTS:
%   sys  - structure, containing the second-order system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        M        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        K        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        Bu       | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        Cp       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        Cv       | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | hankeldecopts   | structure, containing the optional parameters for |
%   |                 | the disk function used for the decomposition      |
%   |                 | after the transformation to an all-pass system    |
%   |                 | see ml_disk and ml_getqz                          |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | hankelsignmopts | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition after the transformation of an      |
%   |                 | all-pass system, see ml_signm                     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | hankelsylvopts  | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition after the transformation of an      |
%   |                 | all-pass system, see ml_sylv_sgn                  |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapcopts       | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | controllability Lyapunov equation,                |
%   |                 | see mess_lradi                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapoopts       | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | observability Lyapunov equation,                  |
%   |                 | see mess_lradi                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | MinRelTol       | nonnegative scalar, tolerance multiplied with the |
%   | {!}             | largest characteristic value to determine a       |
%   |                 | minimal realization                               |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm            |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed Hankel singular |
%   |                 | values                                            |
%   +-----------------+---------------------------------------------------+
%   | infoLYAPC       | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the  |
%   |                 | controllability equation, see mess_lradi          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAPO       | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the  |
%   |                 | observability equation, see mess_lradi            |
%   +-----------------+---------------------------------------------------+
%   | N               | Dimension of the reduced-order model              |
%   | {!}             |                                                   |
%   +-----------------+---------------------------------------------------+
%   | Sigma           | Chosen Hankel singular value, exact approximation |
%   | {!}             | error in the Hankel-norm                          |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%
% See also ml_ct_s_foss_hna, ml_ct_d_soss_hna, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so', ...
            'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        % No extra action in main supported case.

        % TODO: Remove when M-M.E.S.S. is fixed.
        assert(not(strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')) ...
            && not(strcmpi(sys.SystemType, 'ct_s_soss_dae_3_so')), ...
            'MORLAB:notImplemented', ...
            'This system structure is not yet supported.');

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

% Check and assign optional MORLAB parameters.
if ml_field_set_to_value(opts, 'hankeldecopts')
    assert(isa(opts.hankeldecopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.hankeldecopts has to be a struct!');
else
    opts.hankeldecopts = struct();
end

if ml_field_set_to_value(opts, 'hankelsignmopts')
    assert(isa(opts.hankelsignmopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.hankelsignmopts has to be a struct!');
else
    opts.hankelsignmopts = struct();
end

if ml_field_set_to_value(opts, 'hankelsylvopts')
    assert(isa(opts.hankelsylvopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.hankelsylvopts has to be a struct!');
else
    opts.hankelsylvopts = struct();
end

opts = ml_check_cell_param(opts, 'MinRelTol', ...
    @ml_assert_nonnegscalar, log(size(sys.K, 1)) * eps);

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 3;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

% Check and assign optional M-M.E.S.S. parameters.
if ml_field_set_to_value(opts, 'lyapcopts')
    assert(isa(opts.lyapcopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapcopts has to be a struct!');
else
    opts.lyapcopts = struct();
end

opts.lyapcopts = ml_check_messopts({'global', 'shifts', 'adi'}, ...
    opts.lyapcopts);

if ml_field_set_to_value(opts, 'lyapoopts')
    assert(isa(opts.lyapoopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapoopts has to be a struct!');
else
    opts.lyapoopts = struct();
end

opts.lyapoopts = ml_check_messopts({'global', 'shifts', 'adi'}, ...
    opts.lyapoopts);

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Create equation object.
switch lower(sys.SystemType)
    case {'ct_s_soss_so_1', 'ct_s_soss_dae_1_so'}
        eqn  = struct( ...
            'M_'   , sys.M, ...
            'E_'   , sys.E, ...
            'K_'   , sys.K, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_so_1')
            oper = operatormanager(struct(), 'so_1');
        else
            eqn.manifold_dim = sys.nM(end);
            oper             = operatormanager(struct(), 'dae_1_so');
        end

    case {'ct_s_soss_dae_2_so', 'ct_s_soss_dae_3_so'}
        eqn = struct( ...
            'M_'   , sys.pM, ...
            'E_'   , sys.pE, ...
            'K_'   , sys.pK, ...
            'G_'   , sys.pG, ...
            'alpha', 0, ...
            'haveE', true);

        if strcmpi(sys.SystemType, 'ct_s_soss_dae_2_so')
            oper    = operatormanager(struct(), 'dae_2_so');
            wstring = 'index-2';
        else
            oper    = operatormanager(struct(), 'dae_3_so');
            wstring = 'index-3';
        end

        if strcmpi(opts.OutputModel, 'so')
            warning('MORLAB:data', ...
                ['Structure-preserving balancing is not supported for' ...
                ' %s systems.\nChange to Opts.OutputModel = ''fo''.'], ...
                wstring);

            opts.OutputModel = 'fo';
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

eqn.B = ml_compress_fac(sys.pB, eps, 'column');
eqn.C = ml_compress_fac(sys.pC, eps, 'row');

% Computation of low-rank Gramian factors.
if ml_field_set_to_value(opts, 'GramFacC')
    R              = opts.GramFacC;
    info.infoLYAPC = [];
else
    eqn.type       = 'N';
    out            = mess_lradi(eqn, opts.lyapcopts, oper);
    R              = out.Z;
    info.infoLYAPC = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAPC.IterationSteps = length(out.res);
        info.infoLYAPC.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAPC.IterationSteps = length(out.rc);
        info.infoLYAPC.Residuals      = out.rc;
    end
end

if ml_field_set_to_value(opts, 'GramFacO')
    L              = opts.GramFacO;
    info.infoLYAPO = [];
else
    eqn.type       = 'T';
    out            = mess_lradi(eqn, opts.lyapoopts, oper);
    L              = out.Z;
    info.infoLYAPO = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAPO.IterationSteps = length(out.res);
        info.infoLYAPO.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAPO.IterationSteps = length(out.rc);
        info.infoLYAPO.Residuals      = out.rc;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
if ml_field_set_to_value(opts, 'Tolerance')
    tmp = opts.Tolerance;
else
    tmp = [];
end
opts.Tolerance = opts.MinRelTol;
opts.Method    = 'sr';
[V, W, hsv]    = ml_balproj_proper(sys, R, L, 0, 1, opts);
opts.Tolerance = tmp;

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
minsys = ml_projtrunc_proper(sys, V, W);

if not(isa(minsys, 'cell')), minsys = {minsys}; end

nh = ml_order(hsv, 0, rselect, opts);

if not(isa(nh, 'cell')), nh = {nh}; end

[sysFull, nh] = ml_extend_cell(minsys, nh);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TRANSFORMATION TO REDUCED-ORDER MODEL.                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

numRoms   = length(nh);
rom       = cell(1, numRoms);
sigmaFull = cell(1, numRoms);
nhFull    = nh;

for r = 1:numRoms
    nh  = nhFull{r};
    sys = sysFull{r};
    n   = size(sys.A, 1);

    if nh < n
        sigmaFull{r} = hsv(nh + 1);
        sigma        = sigmaFull{r};

        % Get multiplicity of the chosen Hankel singular value.
        k = 1;

        while k < n - nh
            if abs(sigma - hsv(nh + 1 + k)) > sigma * log(n) * sqrt(eps)
                break;
            end

            k = k + 1;
        end

        % Permutation of the system.
        if nh + k < n
            p = [1:nh, nh+k+1:n, nh+1:nh+k];

            sys.A = sys.A(p,p);
            sys.B = sys.B(p, :);
            sys.C = sys.C(: , p);
            hsvh  = hsv(p);
        else
            hsvh = hsv(1:n);
        end

        % Transformation to an all-pass error system.
        nk = n - k;

        U = sigma * pinv(sys.C(: , nk+1:n)') * sys.B(nk+1:n, :);
        S = sparse(1:nk, 1:nk, hsvh(1:nk));
        A = sys.A(1:nk, 1:nk);
        B = sys.B(1:nk, :);
        C = sys.C(: , 1:nk);

        if isempty(sys.D)
            sys.D = zeros(size(sys.C, 1), size(sys.B, 2));
        end

        % Additive decomposition.
        model = struct( ...
            'A', sigma^2 * A' + S * A * S + C' * U * B', ...
            'B', S * B - C' * U, ...
            'C', C * S - U * B', ...
            'D', sys.D + U, ...
            'E', diag(hsvh(1:nk).^2 - sigma^2));

        hankelopts                       = struct();
        hankelopts.infdecopts.Dimension  = 0;
        hankelopts.stabdecopts           = opts.hankeldecopts;
        hankelopts.stabdecopts.Dimension = nk - nh;

        [model, infoHANKEL] = ml_ct_d_dss_adtf(model, hankelopts);

        % Assign information about additive decomposition.
        info.infoHAADTF = infoHANKEL;
    else
        sigmaFull{r} = 0;
        model        = sys;
    end

    rom{r} = struct( ...
        'A', model.A, ...
        'B', model.B, ...
        'C', model.C, ...
        'D', model.D, ...
        'E', model.E);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms     = length(rom);
n           = cell(1, numRoms);
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    n{k}           = size(rom{k}.A, 1);
    absErrBound{k} = 2 * sum(hsv(n{k}+1:end));
end

info.AbsErrBound = absErrBound;
info.Hsv         = hsv;
info.N           = n;
info.Sigma       = sigmaFull;

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
