function [rom, info] = ml_ct_s_foss_bt(sys, opts)
%ML_CT_S_FOSS_BT Balanced truncation for sparse first-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_FOSS_BT(sys)
%   [rom, info] = ML_CT_S_FOSS_BT(sys, opts)
%
% DESCRIPTION:
%   This function computes the balanced truncation for a sparse, standard
%   or descriptor system of the form
%
%    [E]x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, the two standard Lyapunov equations
%
%       A*P*[E'] + [E]*P*A' + B*B' = 0,
%       A'*Q*[E] + [E']*Q*A + C'*C = 0
%
%   are solved for the system Gramians P and Q. As result, a reduced-order
%   system of the form
%
%     [Er]*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed, such that for the original transfer function G and the
%   r-th order transfer function Gr it holds
%
%       ||G - Gr||_{\infty} <= 2 * (Hsv(r+1) + ... + Hsv(n)),
%
%   with Hsv, a vector containing the Hankel singular values of the system.
%
%   Note: The system is assumed to be asymptotically stable.
%
% INPUTS:
%   sys  - structure, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | lyapcopts       | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | controllability Lyapunov equation,                |
%   |                 | see mess_lradi                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapoopts       | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | observability Lyapunov equation,                  |
%   |                 | see mess_lradi                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm            |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed Hankel singular |
%   |                 | values                                            |
%   +-----------------+---------------------------------------------------+
%   | infoLYAPC       | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the  |
%   |                 | controllability equation, see mess_lradi          |
%   +-----------------+---------------------------------------------------+
%   | infoLYAPO       | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the  |
%   |                 | observability equation, see mess_lradi            |
%   +-----------------+---------------------------------------------------+
%   | N               | Dimension of the reduced-order model              |
%   | {!}             |                                                   |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_d_ss_bt, ml_ct_d_dss_bt, ml_ct_s_soss_bt.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

switch lower(sys.SystemType)
    case {'ct_s_ss_default', 'ct_s_dss_default', 'ct_s_dss_dae_1', ...
            'ct_s_dss_dae_2'}
        % No extra action in main supported case.

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

% Check and assign optional MORLAB parameters.
opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 3;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Check and assign optional M-M.E.S.S. parameters.
if ml_field_set_to_value(opts, 'lyapcopts')
    assert(isa(opts.lyapcopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapcopts has to be a struct!');
else
    opts.lyapcopts = struct();
end

opts.lyapcopts = ml_check_messopts({'global', 'shifts', 'adi'}, ...
    opts.lyapcopts);

if ml_field_set_to_value(opts, 'lyapoopts')
    assert(isa(opts.lyapoopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapoopts has to be a struct!');
else
    opts.lyapoopts = struct();
end

opts.lyapoopts = ml_check_messopts({'global', 'shifts', 'adi'}, ...
    opts.lyapoopts);

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Create equation object.
eqn = struct( ...
    'A_', sys.A, ...
    'B' , ml_compress_fac(sys.pB, eps, 'column'), ...
    'C' , ml_compress_fac(sys.pC, eps, 'row'));

% Select user-supplied function handles and mass matrices.
switch lower(sys.SystemType)
    case 'ct_s_ss_default'
        eqn.haveE = false;
        oper      = operatormanager(struct(), 'default');

    case 'ct_s_dss_default'
        if norm(sys.E - speye(size(sys.E, 1)), 'inf') == 0
            eqn.haveE = false;
        else
            eqn.E_    = sys.E;
            eqn.haveE = true;
        end
        oper = operatormanager(struct(), 'default');

    case {'ct_s_dss_dae_1', 'ct_s_dss_dae_2'}
        eqn.E_           = sys.E;
        eqn.haveE        = true;
        eqn.manifold_dim = sys.nE(end);

        if strcmpi(sys.SystemType, 'ct_s_dss_dae_1')
            oper = operatormanager(struct(), 'dae_1');
        else
            oper = operatormanager(struct(), 'dae_2');
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

% Computation of low-rank Gramian factors.
if ml_field_set_to_value(opts, 'GramFacC')
    R              = opts.GramFacC;
    info.infoLYAPC = [];
else
    eqn.type       = 'N';
    out            = mess_lradi(eqn, opts.lyapcopts, oper);
    R              = out.Z;
    info.infoLYAPC = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAPC.IterationSteps = length(out.res);
        info.infoLYAPC.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAPC.IterationSteps = length(out.rc);
        info.infoLYAPC.Residuals      = out.rc;
    end
end

if ml_field_set_to_value(opts, 'GramFacO')
    L              = opts.GramFacO;
    info.infoLYAPO = [];
else
    eqn.type       = 'T';
    out            = mess_lradi(eqn, opts.lyapoopts, oper);
    L              = out.Z;
    info.infoLYAPO = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAPO.IterationSteps = length(out.res);
        info.infoLYAPO.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAPO.IterationSteps = length(out.rc);
        info.infoLYAPO.Residuals      = out.rc;
    end
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
[V, W, hsv] = ml_balproj_proper(sys, R, L, 0, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_proper(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms     = length(rom);
n           = cell(1, numRoms);
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    n{k}           = size(rom{k}.A, 1);
    absErrBound{k} = 2 * sum(hsv(n{k}+1:end));
end

info.AbsErrBound = absErrBound;
info.Hsv         = hsv;
info.N           = n;

% Store projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
