function [rom, info] = ml_ct_s_foss_bst(sys, opts)
%ML_CT_S_FOSS_BST BST for sparse first-order systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_S_FOSS_BST(sys)
%   [rom, info] = ML_CT_S_FOSS_BST(sys, opts)
%
% DESCRIPTION:
%   This function computes the balanced stochastic truncation for a sparse,
%   standard or descriptor system of the form
%
%    [E]x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, one Lyupunov equation
%
%       A*P*[E'] + [E]*P*A' + B*B' = 0,
%
%   and one Riccati equation
%
%       A'*Q*E + E'*Q*A + (C - Bw'*Q*E)' * inv(D*D') * (C - Bw'*Q*E) = 0
%
%   are solved for the system Gramians P and Q. As result, a reduced-order
%   system of the form
%
%     [Er]*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed, such that for the original transfer function G and the
%   r-th order transfer function Gr it holds
%
%       ||G - Gr||_{\infty} / ||G||_{\infty}
%       <= ((1 + Hsv(r+1))/(1 - Hsv(r+1)) * ...
%          * (1 + Hsv(n))/(1 - Hsv(n))) - 1,
%
%   with Hsv, a vector containing the stochastic singular values of the
%   system.
%
%   Note: The system is assumed to be asymptotically stable.
%
% INPUTS:
%   sys  - structure, containing the system matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | careopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see mess_lrrri       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Epsilon         | positive scalar, used in the case of a            |
%   |                 | non-full-rank D term for epsilon regularization   |
%   |                 | by multiplying with an identity matrix of         |
%   |                 | appropriate size                                  |
%   |                 | (default 1.0e-03)                                 |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time            |
%   |                 | Lyapunov equation, see mess_lradi                 |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)))                     |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using relative error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | Riccati equation                                  |
%   |                 |  'ri' - Riccati iteration                         |
%   |                 | (default 'ri')                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m, optional   |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions n x n, optional   |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramian == 1                            |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed stochastic      |
%   |                 | singular values                                   |
%   +-----------------+---------------------------------------------------+
%   | infoCARE        | structure, containing information about the       |
%   |                 | Riccati equation solver                           |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation solver for the, |
%   |                 | see mess_lradi                                    |
%   +-----------------+---------------------------------------------------+
%   | N               | Dimension of the reduced-order model              |
%   | {!}             |                                                   |
%   +-----------------+---------------------------------------------------+
%   | RelErrBound     | computed error bound for the relative error of the|
%   | {!}             | of the reduced-order model in H-infinity norm     |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%
% See also ml_ct_d_ss_bst, ml_ct_d_dss_bst, ml_ct_s_soss_bst.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

assert(exist('mess_version', 'file') == 2, ...
    'MORLAB:mmess', ...
    ['For sparse methods, M-M.E.S.S. version 3.0 or later must be' ...
    ' installed!']);

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

switch lower(sys.SystemType)
    case {'ct_s_ss_default', 'ct_s_dss_default', 'ct_s_dss_dae_1', ...
            'ct_s_dss_dae_2'}
        % No extra action in main supported case.

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

% Prepare system data.
sys = ml_prepare_system_data(sys);

% Check and assign optional MORLAB parameters.
if ml_field_set_to_value(opts, 'Epsilon')
    ml_assert_posscalar(opts.Epsilon, 'opts.Epsilon');
else
    opts.Epsilon = 1.0e-03;
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 4;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'ri'), ...
        'MORLAB:data', ...
        'The requested Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'ri';
end

if strcmpi(opts.RiccatiSolver, 'ri')
    riccati_solver = @mess_lrri;
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Check and assign optional M-M.E.S.S. parameters.
if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

opts.lyapopts = ml_check_messopts([], opts.lyapopts);

if ml_field_set_to_value(opts, 'careopts')
    assert(isa(opts.careopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careopts has to be a struct!');
else
    opts.careopts = struct();
end

opts.careopts = ml_check_messopts([], opts.careopts);

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

m = size(sys.pB, 2);
p = size(sys.pC, 1);

if m < p
    A = sys.A';
    B = sys.pC';
    C = sys.pB';
    D = sys.pD';

    if isfield(sys, 'E')
        E = sys.E';
    end
else
    A = sys.A;
    B = sys.pB;
    C = sys.pC;
    D = sys.pD;

    if isfield(sys, 'E')
        E = sys.E;
    end
end

% Create equation object.
eqn = struct( ...
    'A_', A, ...
    'B' , ml_compress_fac(B, eps, 'column'), ...
    'C' , ml_compress_fac(C, eps, 'row'));

% Select user-supplied function handles and mass matrices.
switch lower(sys.SystemType)
    case 'ct_s_ss_default'
        eqn.haveE = false;
        oper      = operatormanager(struct(), 'default');

    case 'ct_s_dss_default'
        if norm(E - speye(size(E, 1)), 'inf') == 0
            eqn.haveE = false;
        else
            eqn.E_    = E;
            eqn.haveE = true;
        end
        oper = operatormanager(struct(), 'default');

    case {'ct_s_dss_dae_1', 'ct_s_dss_dae_2'}
        eqn.E_           = E;
        eqn.haveE        = true;
        eqn.manifold_dim = sys.nE(end);

        if strcmpi(sys.SystemType, 'ct_s_dss_dae_1')
            oper = operatormanager(struct(), 'dae_1');
        else
            oper = operatormanager(struct(), 'dae_2');
        end

    otherwise
        error('MORLAB:data', ...
            ['The selected system structure is not implemented in this' ...
            ' function!']);
end

% Computation of low-rank Gramian factors.
if (m >= p) && ml_field_set_to_value(opts, 'GramFacC')
    R             = opts.GramFacC;
    info.infoLYAP = [];
elseif (m < p) && ml_field_set_to_value(opts, 'GramFacO')
    R             = opts.GramFacO;
    info.infoLYAP = [];
else
    eqn.type       = 'N';
    out            = mess_lradi(eqn, opts.lyapopts, oper);
    R              = out.Z;
    info.infoLYAP = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoLYAP.IterationSteps = length(out.res);
        info.infoLYAP.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoLYAP.IterationSteps = length(out.rc);
        info.infoLYAP.Residuals      = out.rc;
    end
end

if ml_field_set_to_value(eqn, 'haveE') && eqn.haveE
    [eqn, ~, oper] = oper.mul_E_pre(eqn, struct(), oper);
    ER             = oper.mul_E(eqn, struct(), 'N', R, 'N');
    [eqn, ~, oper] = oper.mul_E_post(eqn, struct(), oper);
else
    ER = R;
end

if (rank(D) < min(m, p)) ...
        || (norm(D, 'fro') < 1.0e+02 * eps)
    D  = opts.Epsilon * eye(min(m, p));
    Bw = B * opts.Epsilon * eye(min(m, p), max(m, p))' + ...
        ER * (R' * C');
else
    Bw     = B * D' + ER * (R' * C');
    [~, D] = qr(D', 0);
end

eqn        = rmfield(eqn, {'B', 'C'});
eqn.B1     = ml_compress_fac(Bw / D, eps, 'column');
eqn.B2     = zeros(size(sys.pB, 1), 0);
eqn.B      = zeros(size(sys.pB, 1), 0);
eqn.C1     = ml_compress_fac(D' \ C, eps, 'row');
eqn.C2     = zeros(0, size(sys.pC, 2));
eqn.C      = zeros(0, size(sys.pC, 2));
eqn.U      = -Bw;
eqn.V      = C' / (D' * D);
eqn.haveUV = true;

if (m >= p) && ml_field_set_to_value(opts, 'GramFacO')
    L             = opts.GramFacO;
    info.infoCARE = [];
elseif (m < p) && ml_field_set_to_value(opts, 'GramFacC')
    L             = opts.GramFacC;
    info.infoCARE = [];
else
    eqn.type       = 'T';
    out            = riccati_solver(eqn, opts.careopts, oper);
    L              = out.Z;
    info.infoCARE = struct();
    if ml_field_set_to_value(out, 'res')
        info.infoCARE.IterationSteps = length(out.res);
        info.infoCARE.Residuals      = out.res;
    end
    if ml_field_set_to_value(out, 'rc')
        info.infoCARE.IterationSteps = length(out.rc);
        info.infoCARE.Residuals      = out.rc;
    end
end

% Swap Gramian factors if system was transposed.
if m < p
    [R, L] = deal(L, R);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
[V, W, hsv] = ml_balproj_proper(sys, R, L, 0, rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
rom = ml_projtrunc_proper(sys, V, W);

if not(isa(rom, 'cell')), rom = {rom}; end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
numRoms     = length(rom);
n           = cell(1, numRoms);
relErrBound = cell(1, numRoms);
for k = 1:numRoms
    n{k}           = size(rom{k}.A, 1);
    relErrBound{k} = prod((1 + hsv(n{k}+1:end)) ...
        ./ (1 - hsv(n{k}+1:end))) - 1;
end

info.Hsv         = hsv;
info.N           = n;
info.RelErrBound = relErrBound;

% Store projection bases.
if opts.StoreProjection
    info.V = V;
    info.W = W;
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
