function [varargout] = ml_ct_mt(varargin)
%ML_CT_MT Continuous-time modal truncation method.
%
% SYNTAX:
%   [Ar, Br, Cr, Dr, info] = ML_CT_MT(A, B, C, D)
%   [Ar, Br, Cr, Dr, info] = ML_CT_MT(A, B, C, D, opts)
%
%   [Ar, Br, Cr, Dr, Er, info] = ML_CT_MT(A, B, C, D, E)
%   [Ar, Br, Cr, Dr, Er, info] = ML_CT_MT(A, B, C, D, E, opts)
%
%   [rom, info] = ML_CT_MT(sys)
%   [rom, info] = ML_CT_MT(sys, opts)
%
% DESCRIPTION:
%   This function computes the modal truncation for a continuous-time
%   standard system of the form
%
%         x'(t) = A*x(t) + B*u(t),                                     (1a)
%          y(t) = C*x(t) + D*u(t),                                     (1b)
%
%   descriptor system of the form
%
%       E*x'(t) = A*x(t) + B*u(t),                                     (2a)
%          y(t) = C*x(t) + D*u(t),                                     (2b)
%
%   or second-order system of the form
%
%       M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t),                       (3a)
%           y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t).                       (3b)
%
%   The computed reduced-order models usually have the format of (1)
%   or (2). In the case of (3), unstructured reduced-order models of the
%   form (2) are computed.
%
% INPUTS:
%   A    - matrix from (1) or (2) with dimensions n x n
%   B    - matrix from (1) or (2) with dimensions n x m
%   C    - matrix from (1) or (2) with dimensions p x n
%   D    - matrix from (1), (2) or (3) with dimensions p x m
%   E    - matrix from (1), (2) or (3) with dimensions n x n
%   M    - matrix from (3) with dimensions n x n
%   K    - matrix from (3) with dimensions n x n
%   Bu   - matrix from (3) with dimensions n x m
%   Cp   - matrix from (3) with dimensions p x n
%   Cv   - matrix from (3) with dimensions p x n
%   sys  - structure or state-space object (ss, dss, sparss, mechss),
%          containing the system's matrices
%   opts - structure, containing optional paramaters, the most important
%          are shown below, for details on parameters to change underlying
%          computational routines see *_mt functions:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Alpha           | real scalar, such that all finite eigenvalues with|
%   | {!}             | the real part smaller than Alpha are truncated    |
%   |                 | (default -1.0)                                    |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   Ar   - output matrix for (1) or (2) with dimensions r x r
%   Br   - output matrix for (1) or (2) with dimensions r x m
%   Cr   - output matrix for (1) or (2) with dimensions p x r
%   Dr   - output matrix for (1) or (2) with dimensions p x m
%   Er   - output matrix for (1) or (2) with dimensions r x r
%   rom  - struct or state-space object (ss, dss), containing the
%          reduced-order system's matrices (naming of entries is identical
%          to input argument sys)
%   info - structure, containing the information about the computations and
%          models, depending on the system type more information might be
%          available, see *_mt routines for details:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | SystemType      | character array, system structure that has been   |
%   |                 | determined for computations                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% See also ml_ct_mt, ml_dt_bt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK AND REFORMAT INPUTS.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[sys, opts, ioformat] = ml_decide_system_type('ct', varargin{:});

if ioformat == 0
    % Check for maximum number of outputs in second-order case.
    nargoutchk(0, 8);
else
    nargoutchk(0, 2);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SELECT BALANCED TRUNCATION ROUTINE.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Start and end strings for system structures.
bstype = 'continuous-time ';
estr   = ' system';

% Selections of routines for different system structures.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        [roms, info]    = ml_ct_d_ss_mt(sys, opts);
        info.SystemType = [bstype 'dense standard state-space' estr];

    case 'ct_d_dss'
        [roms, info]    = ml_ct_d_dss_mt(sys, opts);
        info.SystemType = [bstype 'dense descriptor' estr];

    case 'ct_d_soss'
        [roms, info]    = ml_ct_d_soss_mt(sys, opts);
        info.SystemType = [bstype 'dense second-order' estr];

    case {'ct_s_dss_default', 'ct_s_ss_default', 'ct_s_dss_dae_1', ...
            'ct_s_dss_dae_2', 'ct_s_soss_so_1', ...
            'ct_s_soss_dae_1_so', 'ct_s_soss_dae_2_so', ...
            'ct_s_soss_dae_3_so'}
        error('MORLAB:notImplemented', ...
            ['Modal truncation for sparse continuous-time systems is' ...
            ' not implemented yet.']);

    otherwise
        error('MORLAB:data', 'Unknow system type!');
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if not(isa(roms, 'cell')), roms = {roms}; end

[varargout{1:nargout}] = ml_format_output(roms, ioformat, info);
