function [rom, info] = ml_ct_d_ss_mt(sys, opts)
%ML_CT_D_SS_MT Modal truncation for continuous-time dense, standard systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SS_MT(sys)
%   [rom, info] = ML_CT_D_SS_MT(sys, opts)
%
% DESCRIPTION:
%   This function computes the modal truncation for a standard system of
%   the form
%
%       x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, a block diagonalization of the matrix A is performed using
%   the matrix sign function and a Sylvester equation, such that
%
%            [ Ar  0 ]                        [ Cr ]
%       A2 = [       ], B2 = [ Br, B1 ], C2 = [    ], Dr = D,
%            [ 0  A1 ]                        [ C1 ]
%
%   where Ar contains all the eigenvalues of A, which have a larger real
%   part than a given alpha. As result, the reduced-order system is given
%   by
%
%       x'(t) = Ar*x(t) + Br*u(t),                                      (3)
%        y(t) = Cr*x(t) + Dr*u(t).                                      (4)
%
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Alpha           | real scalar, such that all finite eigenvalues with|
%   | {!}             | the real part smaller than Alpha are truncated    |
%   |                 | (default -1.0)                                    |
%   +-----------------+---------------------------------------------------+
%   | signmopts       | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition of the eigenvalues, see ml_signm    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | sylvopts        | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition of the eigenvalues, see ml_sylv_sgn |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | infoSIGNM       | structure, containing information about the       |
%   | {!}             | matrix sign function method, see ml_signm         |
%   +-----------------+---------------------------------------------------+
%   | infoSYLV        | structure, containing information about the       |
%   | {!}             | Sylvester equation solver, see ml_sylv_sgn        |
%   +-----------------+---------------------------------------------------+
%   | N {!}           | Dimension of the reduced-order model              |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, E. S. Quintana-Orti, Model reduction based on spectral
%   projection methods, in: P. Benner, V. Mehrmann, D. Sorensen (Eds.),
%   Dimension Reduction of Large-Scale Systems, Vol. 45 of Lect. Notes
%   Comput. Sci. Eng., Springer, Berlin/Heidelberg, Germany, 2005,
%   pp. 5--45. https://doi.org/10.1007/3-540-27909-1_1
%
% See also ml_ct_d_dss_mt, ml_dt_d_ss_mt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        % No extra action in main supported case.

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                'System matrices are converted from sparse to full.');
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse standard system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_ss';
sys            = ml_prepare_system_data(sys);

% Check and assign optional parameters.
opts               = ml_check_cell_param(opts, 'Alpha', ...
    @ml_assert_scalar, -1.0);
[opts.Alpha, Inds] = sort(cell2mat(opts.Alpha));
numAlpha           = length(opts.Alpha);

if ml_field_set_to_value(opts, 'signmopts')
    assert(isa(opts.signmopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.signmopts has to be a struct!');
else
    opts.signmopts = struct();
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

if ml_field_set_to_value(opts, 'sylvopts')
    assert(isa(opts.sylvopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.sylvopts has to be a struct!');
else
    opts.sylvopts = struct();
end

% Initial info structure.
info           = struct();
info.infoSIGNM = cell(1, numAlpha);
info.infoSYLV  = cell(1, numAlpha);
info.N         = cell(1, numAlpha);

if opts.StoreProjection
    info.V = cell(1, numAlpha);
    info.W = cell(1, numAlpha);
else
    info.V = [];
    info.W = [];
end

rom = cell(1, numAlpha);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for k = 1:numAlpha
    % Decomposition of desired and undesired eigenvalues.
    n              = size(sys.A, 1);
    [Z, infoSIGNM] = ml_signm(sys.A - opts.Alpha(k) * eye(n), [], ...
        opts.signmopts);

    % Assign information about sign function.
    info.infoSIGNM{Inds(k)} = infoSIGNM;

    n1 = (n + round(trace(Z))) / 2;

    if (n1 > 0) && (n1 < n)
        [Q, ~, ~] = qr(eye(n) + Z);

        A1     = Q(: , n1+1:end)' * (sys.A * Q(: , n1+1:end));
        Wa     = Q(: , 1:n1)' * (sys.A * Q(: , n1+1:end));
        sys.A  = Q(: , 1:n1)' * (sys.A * Q(: , 1:n1));

        [Y, infoSYLV] = ml_sylv_sgn(-sys.A + opts.Alpha(k) * eye(n1), ...
            A1 - opts.Alpha(k) * eye(n - n1), -Wa, [], [], opts.sylvopts);

        sys.B = Q' * sys.B;
        sys.B = sys.B(1:n1, :) - Y * sys.B(n1+1:end, :);

        sys.C = sys.C * Q(: , 1:n1);

        % Assign information about Sylvester equation solver.
        info.infoSYLV{Inds(k)} = infoSYLV;

        if opts.StoreProjection
            if k == 1
                info.V{k} = Q(: , 1:n1);
                info.W{k} = Q * [eye(n1); -Y'];
            else
                info.V{k} = info.V{k-1} * Q(: , 1:n1);
                info.W{k} = info.W{k-1} * (Q * [eye(n1); -Y']);
            end
        end
    elseif n1 == 0
        [sys.A, sys.B, sys.C] = deal([]);

        if opts.StoreProjection
            info.V{k} = zeros(n, 0);
            info.W{k} = zeros(0, n);
        end
    else
        if opts.StoreProjection
            info.V{k} = eye(n);
            info.W{k} = eye(n);
        end
    end

    rom{Inds(k)} = sys;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
for k = 1:numAlpha
    info.N{k} = size(rom{k}.A, 1);
end

[rom, info] = ml_format_output(rom, 1, info);
