function [rom, info] = ml_ct_d_ss_hna(sys, opts)
%ML_CT_D_SS_HNA Hankel-norm approximation for dense, standard systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SS_HNA(sys)
%   [rom, info] = ML_CT_D_SS_HNA(sys, opts)
%
% DESCRIPTION:
%   This function computes the Hankel-norm approximation of a standard
%   system of the form
%
%       x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, first a balanced realization is computed by using the
%   balanced truncation square-root method with an appropriate tolerance
%   for the minimal realization of the given system. Then the system is
%   transformed using the formulas given in the reference below. As
%   result, a reduced-order system of the form
%
%       x'(t) = Ar*x(t) + Br*u(t),                                      (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed, such that for the original transfer function G and and the
%   r-th order transfer function Gr it holds
%
%       ||G - Gr||_{H}       = Hsv(r+1),
%       ||G - Gr||_{\infty} <= 2 * (Hsv(r+1) + ... + Hsv(n)),
%
%   with Hsv, a vector containing the Hankel singular values of the system.
%
%   Note: For unstable systems, first an additive decomposition into the
%         stable and anti-stable parts is performed and then only the
%         stable part will be reduced. That does not change the error
%         formulas.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | hankelsignmopts | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition after the transformation of an      |
%   |                 | all-pass system, see ml_signm                     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | hankelsylvopts  | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition after the transformation of an      |
%   |                 | all-pass system, see ml_sylv_sgn                  |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time Lyapunov   |
%   |                 | equations, see ml_lyapdl_sgn_fac                  |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | MinRelTol       | nonnegative scalar, tolerance multiplied with the |
%   | {!}             | largest characteristic value to determine a       |
%   |                 | minimal realization                               |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)) + Nu)                |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using absolute error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | stabsignmopts   | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_signm                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabsylvopts    | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_sylv_sgn                            |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | UnstabDim       | integer, dimension of the deflating anti-stable   |
%   |                 | subspace, negative if unknown                     |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm            |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed Hankel singular |
%   |                 | values                                            |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | stable and anti-stable parts, see ml_ct_ss_adtf   |
%   +-----------------+---------------------------------------------------+
%   | infoHAADTF      | structure, containing information about the       |
%   |                 | additive decomposition of the Hankel-norm         |
%   |                 | transformed system, see ml_ct_ss_adtf             |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time dual Lyapunov equations solver,   |
%   |                 | see ml_lyapdl_sgn_fac                             |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Dimension of the stable part of the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Dimension of the anti-stable part of the reduced- |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Sigma           | Chosen Hankel singular value, exact approximation |
%   | {!}             | error in the Hankel-norm                          |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, E. S. Quintana-Orti, G. Quintana-Orti, Computing optimal
%   Hankel norm approximations of large-scale systems, in: Proc. 43rd IEEE
%   Conf. Decision Contr., Omnipress, Madison, WI, 2004, pp. 3078--3083.
%
% See also ml_ct_d_dss_hna, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        % No extra action in main supported case.

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_hna to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_ct_s_foss_hna to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_ss';
sys            = ml_prepare_system_data(sys);

% Fill the D matrix.
if isempty(sys.D)
    sys.D = zeros(size(sys.C, 1), size(sys.B, 2));
end

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'hankelsignmopts')
    assert(isa(opts.hankelsignmopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.hankelsignmopts has to be a struct!');
else
    opts.hankelsignmopts = struct();
end

if ml_field_set_to_value(opts, 'hankelsylvopts')
    assert(isa(opts.hankelsylvopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.hankelsylvopts has to be a struct!');
else
    opts.hankelsylvopts = struct();
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

opts = ml_check_cell_param(opts, 'MinRelTol', ...
    @ml_assert_nonnegscalar, log(size(sys.A, 1)) * eps);

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 3;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into the stable and anti-stable parts.
[sys, infoADTF] = ml_ct_d_ss_adtf(sys, opts);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE MINIMAL, BALANCED REALIZATION.                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of the full-rank factors of the system Gramians.
if size(sys.A, 1) > 0
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;
    elseif ml_field_set_to_value(opts, 'GramFacC')
        % Precomputed controllability factor.
        [L, infoLYAP] = ml_lyap_sgn_fac(sys.A', sys.C', [], opts.lyapopts);
        R             = opts.GramFacC;
        info.infoLYAP = infoLYAP;
    elseif ml_field_set_to_value(opts, 'GramFacO')
        % Precomputed observability factor.
        [R, infoLYAP] = ml_lyap_sgn_fac(sys.A, sys.B, [], opts.lyapopts);
        L             = opts.GramFacO;
        info.infoLYAP = infoLYAP;
    else
        % No Gramian factors precomputed.
        [R, L, infoLYAP] = ml_lyapdl_sgn_fac(sys.A, sys.B, sys.C, [], ...
            opts.lyapopts);
        info.infoLYAP    = infoLYAP;
    end
else
    [R, L] = deal([]);
end

if ml_field_set_to_value(opts, 'Tolerance')
    tmp = opts.Tolerance;
else
    tmp = [];
end
opts.Tolerance = opts.MinRelTol;
opts.Method    = 'sr';
[V, W, hsv]    = ml_balproj_proper(sys, R, L, 0, 1, opts);
opts.Tolerance = tmp;

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

minsys = ml_projtrunc_proper(sys, V, W);

if not(isa(minsys, 'cell')), minsys = {minsys}; end

for k = 1:length(minsys)
    minsys{k}.Au = sys.Au;
    minsys{k}.Bu = sys.Bu;
    minsys{k}.Cu = sys.Cu;
end

nh = ml_order(hsv, size(minsys{1}.Au, 1), rselect, opts);

if not(isa(nh, 'cell')), nh = {nh}; end

[sysFull, nh] = ml_extend_cell(minsys, nh);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% TRANSFORMATION TO REDUCED-ORDER MODEL.                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

numRoms   = length(nh);
rom       = cell(1, numRoms);
sigmaFull = cell(1, numRoms);
nhFull    = nh;

for r = 1:numRoms
    nh  = nhFull{r};
    sys = sysFull{r};
    n   = size(sys.A, 1);

    if nh < n
        sigmaFull{r} = hsv(nh + 1);
        sigma        = sigmaFull{r};

        % Get multiplicity of the chosen Hankel singular value.
        k = 1;

        while k < n - nh
            if abs(sigma - hsv(nh + 1 + k)) > sigma * log(n) * sqrt(eps)
                break;
            end

            k = k + 1;
        end

        % Permutation of the system.
        if nh + k < n
            p = [1:nh, nh+k+1:n, nh+1:nh+k];

            sys.A = sys.A(p,p);
            sys.B = sys.B(p, :);
            sys.C = sys.C(: , p);
            hsvh  = hsv(p);
        else
            hsvh = hsv(1:n);
        end

        % Transformation to an all-pass error system.
        nk = n - k;

        U = sigma * pinv(sys.C(: , nk+1:n)') * sys.B(nk+1:n, :);
        G = sparse(1:nk, 1:nk, 1 ./ (hsvh(1:nk).^2 - sigma^2));
        S = sparse(1:nk, 1:nk, hsvh(1:nk));
        A = sys.A(1:nk, 1:nk);
        B = sys.B(1:nk, :);
        C = sys.C(: , 1:nk);

        model = struct( ...
            'A', G * (sigma^2 * A' + S * A * S + C' * U * B'), ...
            'B', G * (S * B - C' * U), ...
            'C', C * S - U * B', ...
            'D', sys.D + U);

        % Additive decomposition.
        hankelopts               = struct();
        hankelopts.stabsignmopts = opts.hankelsignmopts;
        hankelopts.stabsylvopts  = opts.hankelsylvopts;
        hankelopts.UnstabDim     = nk - nh;

        [model, infoHANKEL] = ml_ct_d_ss_adtf(model, hankelopts);

        % Assign information about additive decomposition.
        info.infoHAADTF = infoHANKEL;
    else
        sigmaFull{r} = 0;
        model        = sys;
    end

    rom{r} = struct( ...
        'A', blkdiag(model.A, sysFull{k}.Au), ...
        'B', [model.B; sysFull{k}.Bu], ...
        'C', [model.C, sysFull{k}.Cu], ...
        'D', model.D);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
absErrBound = cell(1, numRoms);
for k = 1:numRoms
    absErrBound{k} = 2 * sum(hsv(nhFull{k}+1:end));
end

info.AbsErrBound = absErrBound;
info.Hsv         = hsv;
info.Ns          = nhFull;
info.Nu          = size(sys.Au, 1);
info.Sigma       = sigmaFull;

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
