function [rom, info] = ml_ct_d_ss_flbt(sys, opts)
%ML_CT_D_SS_FLBT Frequency-limited balanced truncation for standard systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_SS_FLBT(sys)
%   [rom, info] = ML_CT_D_SS_FLBT(sys, opts)
%
% DESCRIPTION:
%   This function computes the frequency-limited balanced truncation for a
%   standard system of the form
%
%       x'(t) = A*x(t) + B*u(t),                                        (1)
%        y(t) = C*x(t) + D*u(t).                                        (2)
%
%   Therefore, the two standard Lyapunov equations
%
%       A*P  + P*A' + BF*B' + B*BF' = 0,
%       A'*Q + Q*A  + CF'*C + C'*CF = 0,
%
%   where BF and CF are frequency-dependent matrices, are solved for the
%   frequency-limited Gramians P and Q. As result, a reduced-order
%   system of the form
%
%       x'(t) = Ar*x(t) + Br*u(t),                                      (3)
%        y(t) = Cr*x(t) + Dr*u(t)                                       (4)
%
%   is computed. For enforcing stability in the reduced-order model, the
%   modified Gramian approach can be used, which also gives a global error
%   bound of the form
%
%       ||G - Gr||_{\infty} <= 2*||JB||*||JC||(Hsv(r+1) + ... + Hsv(n)),
%
%   with Hsv, a vector containing the freuqency-limited Hankel singular
%   values of the system.
%
%   Note: For unstable systems, first an additive decomposition into the
%         stable and anti-stable parts is performed and then only the
%         stable part will be reduced.
%
% INPUTS:
%   sys  - structure, containing the standard system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | FreqRange       | nonnegative vector, frequency intervals such that |
%   |                 | [w(1), w(2)] ... [w(2k-1), w(2k)] are approximated|
%   |                 | (default [0, 1.0e+03])                            |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the generalized continuous-time|
%   |                 | Lyapunov equations, see ml_lyapdl_sgn_ldl if      |
%   |                 | ModGramian = 0 and ml_lyapdl_sgn_fac if           |
%   |                 | ModGramian = 1                                    |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | ModGramian      | {0, 1}, used to disable/enable the modified       |
%   |                 | Gramian approach                                  |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsv)) + Nu)                |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using rel. tolerance for the hsv   |
%   |                 |  'sum'       - using rel. tolerance for sum of hsv|
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | stabsignmopts   | structure, containing the optional parameters for |
%   |                 | the matrix sign function used for the             |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_signm                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | stabsylvopts    | structure, containing the optional parameters for |
%   |                 | the Sylvester equation solver used for the        |
%   |                 | decomposition into stable and anti-stable system  |
%   |                 | parts, see ml_sylv_sgn                            |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | if 'tolerance' or 'sum' is set for                |
%   |                 | OrderComputation                                  |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%   | UnstabDim       | integer, dimension of the deflating anti-stable   |
%   |                 | subspace, negative if unknown                     |
%   |                 | (default -1)                                      |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, with the following entries:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErrBound     | computed error bound for the absolute error of the|
%   | {!}             | reduced-order model in H-infinity norm, only for  |
%   |                 | the modified Gramian approach                     |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsv             | a vector, containing the computed Hankel singular |
%   |                 | values                                            |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | stable and anti-stable parts, see ml_ct_ss_adtf   |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time dual Lyapunov equations solver,   |
%   |                 | see ml_lyapdl_sgn_ldl or ml_lyapdl_sgn_fac        |
%   +-----------------+---------------------------------------------------+
%   | Ns              | Dimension of the stable part of the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Dimension of the anti-stable part of the reduced- |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, P. Kurschner, J. Saak, Frequency-limited balanced truncation
%   with low-rank approximations, SIAM J. Sci. Comput. 38 (1) (2016)
%   A471--A499. https://doi.org/ 10.1137/15M1030911
%
% See also ml_ct_d_ss_bt, ml_ct_d_ss_tlbt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        % No extra action in main supported case.

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                'System matrices were converted from sparse to full.');
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse standard system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_ss';
sys            = ml_prepare_system_data(sys);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'FreqRange')
    ml_assert_nonnegvector(opts.FreqRange, 'opts.FreqRange');

    assert(mod(length(opts.FreqRange), 2) == 0, ...
        'MORLAB:data', ...
        'The parameter opts.FreqRange has to be an even vector!');

    opts.FreqRange = sort(opts.FreqRange);
else
    opts.FreqRange = [0, 1.0e+03];
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

if ml_field_set_to_value(opts, 'ModGramian')
    ml_assert_boolean(opts.ModGramian, 'opts.ModGramian');
else
    opts.ModGramian = 0;
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 1;
    elseif strcmpi(opts.OrderComputation{k}, 'sum')
        rselect{k} = 2;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into the stable and anti-stable parts.
[sys, infoADTF] = ml_ct_d_ss_adtf(sys, opts);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

n = size(sys.A, 1);
m = size(sys.B, 2);
p = size(sys.C, 1);
E = eye(n);

% Computation of the full-rank factors of the proper Gramians.
if n > 0
    if ml_field_set_to_value(opts, 'GramFacC') ...
            && ml_field_set_to_value(opts, 'GramFacO')
        % Both precomputed Gramian factors.
        R = opts.GramFacC;
        L = opts.GramFacO;

        if opts.ModGramian
            if opts.FreqRange(1) == 0
                F      = -sys.A - 1i * opts.FreqRange(2) * E;
                kstart = 2;
            else
                F      = E;
                kstart = 1;
            end

            for k = kstart:1:length(opts.FreqRange)/2
                F = F * ((sys.A + 1i * opts.FreqRange(2*k-1) * E) ...
                    \ (sys.A + 1i * opts.FreqRange(2*k) * E));
            end

            F  = (1i / pi) * logm(F);
            BF = real(F * sys.B);
            CF = real(sys.C * F);
            BQ = [zeros(m), eye(m); eye(m), zeros(m)];
            CQ = [zeros(p), eye(p); eye(p), zeros(p)];

            [CF, Y] = ml_compress_ldl([sys.C; CF], CQ, eps, 'row');
            Y       = abs(diag(Y));
            JC      = norm(sys.C * CF' * diag(1 ./ sqrt(Y)), 2);

            [BF, Y] = ml_compress_ldl([sys.B, BF], BQ, eps, 'col');
            Y       = abs(diag(Y));
            JB      = norm(diag(1 ./ sqrt(Y)) * BF' * sys.B, 2);
        end
    else
        % Compute frequency-dependent right-hand side.
        if opts.FreqRange(1) == 0
            F      = -sys.A - 1i * opts.FreqRange(2) * E;
            kstart = 2;
        else
            F      = E;
            kstart = 1;
        end

        for k = kstart:1:length(opts.FreqRange)/2
            F = F * ((sys.A + 1i * opts.FreqRange(2*k-1) * E) ...
                \ (sys.A + 1i * opts.FreqRange(2*k) * E));
        end

        F  = (1i / pi) * logm(F);
        BF = real(F * sys.B);
        CF = real(sys.C * F);
        BQ = [zeros(m), eye(m); eye(m), zeros(m)];
        CQ = [zeros(p), eye(p); eye(p), zeros(p)];

        if opts.ModGramian
            [CF, Y] = ml_compress_ldl([sys.C; CF], CQ, eps, 'row');
            Y       = abs(diag(Y));
            JC      = norm(sys.C * CF' * diag(1 ./ sqrt(Y)), 2);
            CF      = diag(sqrt(Y)) * CF;

            [BF, Y] = ml_compress_ldl([sys.B, BF], BQ, eps, 'col');
            Y       = abs(diag(Y));
            JB      = norm(diag(1 ./ sqrt(Y)) * BF' * sys.B, 2);
            BF      = BF * diag(sqrt(Y));
        end

        if ml_field_set_to_value(opts, 'GramFacC')
            % Precomputed controllability factor.
            if opts.ModGramian
                % Modified Gramian approach.
                [L, infoLYAP] = ml_lyap_sgn_fac(sys.A', CF', [], ...
                    opts.lyapopts);
            else
                % Classical method.
                [L, LY, infoLYAP] = ml_lyap_sgn_ldl(sys.A', ...
                    [sys.C; CF]', CQ', [], opts.lyapopts);

                LY  = diag(LY);
                idx = (LY > 0);
                L   = L(:, idx) * diag(sqrt(LY(idx)));
            end

            R = opts.GramFacC;
        elseif ml_field_set_to_value(opts, 'GramFacO')
            % Precomputed observability factor.
            if opts.ModGramian
                % Modified Gramian approach.
                [R, infoLYAP] = ml_lyap_sgn_fac(sys.A, BF, [], ...
                    opts.lyapopts);
            else
                % Classical method.
                [R, RY, infoLYAP] = ml_lyap_sgn_ldl(sys.A, ...
                    [sys.B, BF], BQ, [], opts.lyapopts);

                RY  = diag(RY);
                idx = (RY > 0);
                R   = R(:, idx) * diag(sqrt(RY(idx)));
            end

            L = opts.GramFacO;
        else
            % No Gramian factors precomputed.
            if opts.ModGramian
                % Modified Gramian approach.
                [R, L, infoLYAP] = ml_lyapdl_sgn_fac(sys.A, BF, CF, [], ...
                    opts.lyapopts);
            else
                % Classical method.
                [R, RY, L, LY, infoLYAP] = ml_lyapdl_sgn_ldl(sys.A, ...
                    [sys.B, BF], BQ, [sys.C; CF], CQ, [], opts.lyapopts);

                RY  = diag(RY);
                idx = (RY > 0);
                R   = R(:, idx) * diag(sqrt(RY(idx)));

                LY  = diag(LY);
                idx = (LY > 0);
                L   = L(:, idx) * diag(sqrt(LY(idx)));
            end
        end

        % Assig information about Lyapunov equation solver.
        info.infoLYAP = infoLYAP;
    end
else
    [R, L] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection basis matrices.
[V, W, hsv] = ml_balproj_proper(sys, R, L, size(sys.Au, 1), rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

% Reduce stable part of the system.
stabrom = ml_projtrunc_proper(sys, V, W);

% Construct final reduced-order models from stable and unstable parts.
if not(isa(stabrom, 'cell')), stabrom = {stabrom}; end

numRoms = length(stabrom);
rom    = cell(1, numRoms);
for k = 1:numRoms
    rom{k}.A = blkdiag(stabrom{k}.A, sys.Au);
    rom{k}.B = [stabrom{k}.B; sys.Bu];
    rom{k}.C = [stabrom{k}.C, sys.Cu];
    rom{k}.D = sys.D;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
n = cell(1, numRoms);
for k = 1:numRoms
    n{k} = size(stabrom{k}.A, 1);
end
if opts.ModGramian
    absErrBound = cell(1, numRoms);
    for k = 1:numRoms
        absErrBound{k} = 2 * JB * JC * sum(hsv(n{k}+1:end));
    end
    info.AbsErrBound = absErrBound;
end

info.Hsv = hsv;
info.Ns  = n;
info.Nu  = size(sys.Au, 1);

% Store final projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = infoADTF.V * blkdiag(V{k}, eye(info.Nu));
        info.W{k} = infoADTF.W * blkdiag(W{k}, eye(info.Nu));
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC = R;
    info.GramFacO = L;
else
    info.GramFacC = [];
    info.GramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
