function [rom, info] = ml_ct_d_dss_mt(sys, opts)
%ML_CT_D_DSS_MT Modal truncation for descriptor systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_DSS_MT(sys)
%   [rom, info] = ML_CT_D_DSS_MT(sys, opts)
%
% DESCRIPTION:
%   This function computes the generalized modal truncation for a
%   descriptor system of the form
%
%       E*x'(t) = A*x(t) + B*u(t),                                      (1)
%          y(t) = C*x(t) + D*u(t).                                      (2)
%
%   Therefore, a block diagonalization of the matrix pencil s*E - A is
%   performed using the matrix disk function, such that
%
%            [ Ei 0  0  ]       [ Ai 0  0  ]
%       E2 = [ 0  E0 0  ], A2 = [ 0  A0 0  ],
%            [ 0  0  E1 ]       [ 0  0  A1 ]
%
%                                 [ Ci ]
%       B2 = [ Bi, B0, B1 ], C2 = [ C0 ],
%                                 [ C1 ]
%
%   where the matrix pencil s*Ei - Ai contains all infinite eigenvalues
%   and s*E0 - A0 the finite eigenvalues with the real part larger than a
%   given alpha. As result, the reduced-order system is given by
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t),                                   (4)
%
%   with
%
%            [ E0  0 ]       [ A0  0 ]                        [ C0 ]
%       Er = [       ], A2 = [       ], B2 = [ B0, Bi ], C2 = [    ].
%            [ 0  Ei ]       [ 0  Ai ]                        [ Ci ]
%
%
% INPUTS:
%   sys  - structure or state-space object, containing the descriptor
%          system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Alpha           | real scalar, such that all finite eigenvalues with|
%   | {!}             | the real part smaller than Alpha are truncated    |
%   |                 | (default -1.0)                                    |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest improper Hankel singular value of the     |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | signmopts       | structure, containing the optional parameters for |
%   |                 | the sign function method, see ml_signm            |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | RankTol         | nonnegative scalar, tolerance used for the        |
%   |                 | determination of deflating subspaces              |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be a cell array containing
%         multiple arguments. In this case an cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure or state-space object, containing the reduced-order
%          descriptor system:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoSIGNM       | structure, containing information about the sign  |
%   | {!}             | function method used for the modal truncation of  |
%   |                 | the system, see ml_signm                          |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite and finite parts, see ml_ct_d_dss_adtf   |
%   +-----------------+---------------------------------------------------+
%   | Np              | Dimension of the finite part in the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the infinite part in the reduced-    |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, E. S. Quintana-Orti, Model reduction based on spectral
%   projection methods, in: P. Benner, V. Mehrmann, D. Sorensen (Eds.),
%   Dimension Reduction of Large-Scale Systems, Vol. 45 of Lect. Notes
%   Comput. Sci. Eng., Springer, Berlin/Heidelberg, Germany, 2005,
%   pp. 5--45. https://doi.org/10.1007/3-540-27909-1_1
%
% See also ml_dt_d_dss_mt, ml_ct_d_ss_mt, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        warning('MORLAB:data', ...
            ['No descriptor matrix E found, set to identity E = I_n.' ...
            ' Use ml_ct_d_ss_mt to handle standard systems.']);

        sys.E = eye(size(sys.A));

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['No descriptor matrix E found, set to identity.', ...
                ' System matrices are converted from sparse to full.']);
            sys.E = eye(size(sys.A));
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse standard system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    case 'ct_d_dss'
        % No extra action in main supported case.

    case {'ct_s_dss_default', 'ct_s_dss_dae_1', 'ct_s_dss_dae_2'}
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                'System matrices are converted from sparse to full.');
        else
            error('MORLAB:notImplemented', ...
                ['Large-scale sparse descriptor system detected.' ...
                ' For this system structure there is no method' ...
                ' implemented yet.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_dss';
sys            = ml_prepare_system_data(sys);

% Check and assign optional parameters.
opts               = ml_check_cell_param(opts, 'Alpha', ...
    @ml_assert_scalar, -1.0);
[opts.Alpha, Inds] = sort(cell2mat(opts.Alpha));
numAlpha           = length(opts.Alpha);

if ml_field_set_to_value(opts, 'gdlyapopts')
    assert(isa(opts.gdlyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.gdlyapopts has to be a struct!');
else
    opts.gdlyapopts = struct();
end

if ml_field_set_to_value(opts, 'signmopts')
    assert(isa(opts.signmopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.signmopts has to be a struct!');
else
    opts.signmopts = struct();
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info           = struct();
info.Np        = cell(1, numAlpha);
info.infoSIGNM = cell(1, numAlpha);

if opts.StoreProjection
    info.V = cell(1, numAlpha);
    info.W = cell(1, numAlpha);
else
    info.V = [];
    info.W = [];
end

stabrom = cell(1, numAlpha);


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into finite and infinite parts.
opts.stabdecopts.Dimension = 0;
[sys, infoADTF]            = ml_ct_d_dss_adtf(sys, opts);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});

if opts.StoreProjection
    Iinf = eye(size(sys.Ainf));
end

for k = 1:numAlpha
    n = size(sys.A, 1);

    % Decomposition of desired and undesired eigenvalues.
    if n > 0
        % In vectorized mode this field could be set from a previous
        % iteration.
        opts.Dimension      = [];
        [Aspace, infoSIGNM] = ml_signm(sys.A - opts.Alpha(k) * sys.E, ...
            sys.E, opts.signmopts);
        AE1                 = Aspace + sys.E;
        AE2                 = Aspace - sys.E;
        [Q, ~, n1]          = ml_getqz(sys.A, sys.E, AE1, AE2, opts);

        if (n1 > 0) && (n1 < n)
            opts.Dimension = n - n1;
            [~, V, ns]     = ml_getqz(sys.A, sys.E, AE2, AE1, opts);

            sys.A = Q(: , n1+1:end)' * sys.A * V(: , 1:ns);
            sys.E = Q(: , n1+1:end)' * sys.E * V(: , 1:ns);
            sys.B = Q(: , n1+1:end)' * sys.B;
            sys.C = sys.C * V(: , 1:ns);

            if opts.StoreProjection
                if k == 1
                    info.V{k} = infoADTF.V * blkdiag(V(: , 1:ns), Iinf);
                    info.W{k} = infoADTF.W ...
                        * blkdiag(Q(: , n1+1:end), Iinf);
                else
                    info.V{k} = info.V{k-1} * blkdiag(V(: , 1:ns), Iinf);
                    info.W{k} = info.W{k-1} ...
                        * blkdiag(Q(: , n1+1:end), Iinf);
                end
            end
        elseif n1 == n
            [sys.A, sys.B, sys.C, sys.E] = deal([]);

            if opts.StoreProjection
                info.V{k} = infoADTF.V * blkdiag(zeros(n, 0), Iinf);
                info.W{k} = infoADTF.W * blkdiag(zeros(n ,0), Iinf);
            end
        else
            if opts.StoreProjection
                info.V{k} = infoADTF.V;
                info.W{k} = infoADTF.W;
            end
        end

        % Assign information about the disk function.
        info.infoSIGNM{Inds(k)} = infoSIGNM;
    else
        if opts.StoreProjection
            info.V{k} = infoADTF.V * blkdiag(zeros(n, 0), Iinf);
            info.W{k} = infoADTF.W * blkdiag(zeros(n, 0), Iinf);
        end
    end

    stabrom{Inds(k)} = sys;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% REDUCTION OF IMPROPER PART.                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of the full-rank factors of the improper Gramians.
if size(sys.Ainf, 1) > 0
    if ml_field_set_to_value(opts, 'IGramFacC') ...
            && ml_field_set_to_value(opts, 'IGramFacO')
        % Both precomputed Gramian factors.
        Rinf = opts.IGramFacC;
        Linf = opts.IGramFacO;
    elseif ml_field_set_to_value(opts, 'IGramFacC')
        % Precomputed controllability factor.
        [Linf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf', ...
            sys.Cinf', sys.Einf', opts.gdlyapopts);
        Rinf               = opts.IGramFacC;
        info.infoGDLYAP    = infoGDLYAP;
    elseif ml_field_set_to_value(opts, 'IGramFacO')
        % Precomputed observability factor.
        [Rinf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf, ...
            sys.Binf, sys.Einf, opts.gdlyapopts);
        Linf               = opts.IGramFacO;
        info.infoGDLYAP    = infoGDLYAP;
    else
        % No Gramian factors precomputed.
        [Rinf, Linf, infoGDLYAP] = ml_gdlyapdl_smith_fac( ...
            sys.Ainf, sys.Binf, sys.Cinf, sys.Einf, opts.gdlyapopts);
        info.infoGDLYAP          = infoGDLYAP;
    end
else
    [Rinf, Linf] = deal([]);
end

opts.Method        = 'bfsr';
[Vinf, Winf, hsvi] = ml_balproj_improper(sys, Rinf, Linf, opts);
infrom             = ml_projtrunc_improper(sys, Vinf, Winf);

% Construct final ROMs from modal truncated and improper parts.
numRoms = length(stabrom);
rom    = cell(1, numRoms);

for k = 1:numRoms
    rom{k}.A = blkdiag(stabrom{k}.A, infrom.A);
    rom{k}.B = [stabrom{k}.B; infrom.B];
    rom{k}.C = [stabrom{k}.C, infrom.C];
    rom{k}.D = sys.D;
    rom{k}.E = blkdiag(stabrom{k}.E, infrom.E);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
for k = 1:numAlpha
    info.Np{k} = size(stabrom{k}.A, 1);
end

info.Ni   = size(infrom.A, 1);
info.Hsvi = hsvi;

% Store projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = info.V{k} * blkdiag(eye(info.Np{k}), Vinf);
        info.W{k} = info.W{k} * blkdiag(eye(info.Np{k}), Winf);
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.IGramFacC = Rinf;
    info.IGramFacO = Linf;
else
    info.IGramFacC = [];
    info.IGramFacO = [];
end

[rom, info] = ml_format_output(rom, 1, info);
