function [rom, info] = ml_ct_d_dss_bst(sys, opts)
%ML_CT_D_DSS_BST Balanced stochastic truncation for descriptor systems.
%
% SYNTAX:
%   [rom, info] = ML_CT_D_DSS_BST(sys)
%   [rom, info] = ML_CT_D_DSS_BST(sys, opts)
%
% DESCRIPTION:
%   This function computes the generalized balanced stochastic truncation
%   for a descriptor system of the form
%
%       E*x'(t) = A*x(t) + B*u(t),                                      (1)
%          y(t) = C*x(t) + D*u(t),                                      (2)
%
%   Therefore, first an additive decomposition of the system is performed
%   using the matrix disk function, such that
%
%            [ Ei  0 ]       [ Ai  0 ]                        [ Ci ]
%       E2 = [       ], A2 = [       ], B2 = [ Bi, Bp ], C2 = [    ],
%            [ 0  Ep ]       [ 0  Ap ]                        [ Cp ]
%
%   with (Ei, Ai, Bi, Ci, D) belonging to the polynomial part and
%   (Ep, Ap, Bp, Cp, 0) belonging to the strictly proper part.
%   Now, for the reduction of the strictly proper part the generalized
%   continuous-time Lyapunov equation
%
%       Ap*Pp*Ep' + Ep*Pp*Ap' + Bp*Bp' = 0
%
%   is solved for Pp and then, the corresponding generalized Riccati
%   equation
%
%       Ap'*Qp*Ep + Ep'*Qp*Ap
%       + (Cp - Bw' * Qp * Ep)' * inv(M*M') * (Cp - Bw' * Qp * Ep) = 0
%
%   is solved for Qp, with M = D - Ci * inv(Ai) * Bi. Also, the the two
%   generalized discrete-time Lyapunov equations
%
%       Ai*Pi*Ai' - Ei*Pi*Ei' - Bi*Bi' = 0,
%       Ai'*Qi*Ai - Ei'*Qi*Ei - Ci'*Ci = 0
%
%   are solved for the reduction of the polynomial part. As result, a
%   reduced-order system of the form
%
%       Er*x'(t) = Ar*x(t) + Br*u(t),                                   (3)
%           y(t) = Cr*x(t) + Dr*u(t)                                    (4)
%
%   is computed, such that for the original proper transfer function G and
%   the reduced-order transfer function Gr with an r-th order strictly
%   proper part it holds
%
%       ||G - Gr||_{\infty} / ||G||_{\infty}
%       <= ((1 + Hsvp(r+1))/(1 - Hsvp(r+1)) * ...
%          * (1 + Hsvp(n))/(1 - Hsvp(n))) - 1,
%
%   with Hsvp, a vector containing the proper characteristic stochastic
%   singular values of the system. If the transfer function is invertible
%   it holds
%
%       ||inv(G)*(G - Gr)||_{\infty}
%       <= ((1 + Hsvp(r+1))/(1 - Hsvp(r+1)) * ...
%          * (1 + Hsvp(n))/(1 - Hsvp(n))) + 1.
%
%   Notes:
%     1) The equations above are defined for the case of p < m. Otherwise
%        the system is transposed, then reduced and back transposed.
%     2) In case of a rank-deficient M term, an epsilon regularization is
%        performed, which replaces the M during the computations with an
%        identity matrix scaled by a given epsilon.
%     3) For unstable systems, first an additive decomposition into the
%        stable and anti-stable parts is performed and then only the
%        stable part will be reduced.
%
% INPUTS:
%   sys  - structure, containing the descriptor system's matrices:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (1) with dimensions n x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (2) with dimensions p x n             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (2) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (1) with dimensions n x n             |
%   +-----------------+---------------------------------------------------+
%   opts - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | careopts        | structure, containing the optional parameters for |
%   |                 | the Riccati equation solver, see ml_caredl_sgn_fac|
%   |                 | and ml_care_nwt_fac                               |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | DecompEig       | positive scalar, overestimation of the absolute   |
%   |                 | value of the largest finite eigenvalue of s*E - A,|
%   |                 | if set, replaces the computation with DecompTol   |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | DecompTol       | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest singular value of E to determine the      |
%   |                 | smallest non-quasi-zero singular value of E       |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Epsilon         | positive scalar, used in the case of a            |
%   |                 | non-full-rank M + M' term for epsilon             |
%   |                 | regularization by multiplying with an identity    |
%   |                 | matrix of appropriate size                        |
%   |                 | (default 1.0e-03)                                 |
%   +-----------------+---------------------------------------------------+
%   | gdlyapopts      | structure, containing the optional parameters for |
%   |                 | the computation of the generalized discrete-time  |
%   |                 | Lyapunov equations, see ml_gdlyapdl_smith_fac     |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian    |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian      |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the observability improper     |
%   |                 | Gramian                                           |
%   |                 | (default [])                                      |
%   +-----------------+---------------------------------------------------+
%   | ImproperTrunc   | nonnegative scalar, tolerance multiplied with the |
%   |                 | largest improper Hankel singular value of the     |
%   |                 | system to truncate the improper part, if 0 no     |
%   |                 | improper balanced truncation is performed         |
%   |                 | (default log(n)*eps)                              |
%   +-----------------+---------------------------------------------------+
%   | Index           | nonnegative integer, index of the descriptor      |
%   |                 | system used to set an upper bound on the size of  |
%   |                 | the reduced improper part, Inf if unknown         |
%   |                 | (default Inf)                                     |
%   +-----------------+---------------------------------------------------+
%   | infdecopts      | structure, containing the optional parameters for |
%   |                 | the decomposition of the finite and infinite parts|
%   |                 | of the system using the disk function and subspace|
%   |                 | extraction method, see ml_disk and ml_getqz       |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | lyapopts        | structure, containing the optional parameters for |
%   |                 | the computation of the continuous-time algebraic  |
%   |                 | Lyapunov equation, see ml_lyap_sgn_fac            |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | Method          | character array, determining algorithm for the    |
%   | {!}             | computation of the reduced-order model            |
%   |                 |  'sr'   - square-root method                      |
%   |                 |  'bfsr' - balancing-free square-root method       |
%   |                 | (default 'sr')                                    |
%   +-----------------+---------------------------------------------------+
%   | Order           | positive integer, order of the resulting          |
%   | {!}             | reduced-order model chosen by the user if         |
%   |                 | 'order' is set for OrderComputation               |
%   |                 | (default min(10,length(Hsvp)) + Nu + Ni)          |
%   +-----------------+---------------------------------------------------+
%   | OrderComputation| character array, determining the method for the   |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 |  'order'     - take explicit order                |
%   |                 |  'tolerance' - using relative error bound         |
%   |                 | (default 'tolerance')                             |
%   +-----------------+---------------------------------------------------+
%   | RiccatiSolver   | character array, determining the solver for the   |
%   |                 | dual Riccati equations                            |
%   |                 |  'newton' - Newton iteration                      |
%   |                 |  'sign'   - dual sign function method             |
%   |                 | (default 'sign')                                  |
%   +-----------------+---------------------------------------------------+
%   | stabdecopts     | structure, containing the optional parameters for |
%   |                 | the decomposition of the stable and unstable parts|
%   |                 | of the system using the sign function and subspace|
%   |                 | extraction method, see ml_signm and ml_getqz      |
%   |                 | (default struct())                                |
%   +-----------------+---------------------------------------------------+
%   | StoreGramians   | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed low-rank Gramian factors                 |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | StoreProjection | {0, 1}, used to disable/enable storing of the     |
%   |                 | computed projection matrices W and V              |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | Tolerance       | nonnegative scalar, tolerance used for the        |
%   | {!}             | computation of the size of the reduced-order model|
%   |                 | by an absolute error bound if 'tolerance' is set  |
%   |                 | for OrderComputation                              |
%   |                 | (default 1.0e-02)                                 |
%   +-----------------+---------------------------------------------------+
%
%   Note: Parameters marked with {!} may also be cell arrays containing
%         multiple arguments. In this case, a cell array of the same size
%         is returned with one entry computed for each input argument and
%         the marked fields of the info struct are cells as well.
%         When multiple arguments are given as cells, they are expected to
%         have the same length.
%
% OUTPUTS:
%   rom  - structure, containing the reduced-order descriptor system:
%   {!}
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   |        A        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   |        B        | matrix from (3) with dimensions r x m             |
%   +-----------------+---------------------------------------------------+
%   |        C        | matrix from (4) with dimensions p x r             |
%   +-----------------+---------------------------------------------------+
%   |        D        | matrix from (4) with dimensions p x m             |
%   +-----------------+---------------------------------------------------+
%   |        E        | matrix from (3) with dimensions r x r             |
%   +-----------------+---------------------------------------------------+
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | GramFacC        | low-rank factor of the controllability Gramian, if|
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | GramFacO        | low-rank factor of the observability Gramian, if  |
%   |                 | opts.StoreGramians == 1                           |
%   +-----------------+---------------------------------------------------+
%   | Hsvi            | a vector, containing the computed Hankel singular |
%   |                 | values of the improper part of the system         |
%   +-----------------+---------------------------------------------------+
%   | Hsvp            | a vector, containing the computed characteristic  |
%   |                 | stochastic singular values of the proper part of  |
%   |                 | the system                                        |
%   +-----------------+---------------------------------------------------+
%   | IGramFacC       | low-rank factor of the improper controllability   |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | IGramFacO       | low-rank factor of the improper observability     |
%   |                 | Gramian, if opts.StoreGramians == 1               |
%   +-----------------+---------------------------------------------------+
%   | infoCARE        | structure, containing information about the       |
%   |                 | sign function solver for the dual Riccati         |
%   |                 | equations, see ml_caredl_sgn_fac                  |
%   +-----------------+---------------------------------------------------+
%   | infoADTF        | structure, containing information about the       |
%   |                 | additive decomposition of the system into its     |
%   |                 | infinite, finite stable and finite anti-stable    |
%   |                 | parts, see ml_ct_dss_adtf                         |
%   +-----------------+---------------------------------------------------+
%   | infoGDLYAP      | structure, containing information about the       |
%   |                 | generalized discrete-time Lyapunov equation solver|
%   |                 | for the improper Gramians,                        |
%   |                 | see ml_gdlyapdl_smith_fac                         |
%   +-----------------+---------------------------------------------------+
%   | infoLYAP        | structure, containing information about the       |
%   |                 | continuous-time Lyapunov equation sovler for the  |
%   |                 | controllability Gramian, ml_lyap_sgn_fac          |
%   +-----------------+---------------------------------------------------+
%   | infoPCARE       | structure, containing information about the       |
%   |                 | continuous-time algebraic positive Riccati        |
%   |                 | equation for the observability Gramian, see       |
%   |                 | ml_pcare_nwt_fac                                  |
%   +-----------------+---------------------------------------------------+
%   | Ni              | Dimension of the improper part in the reduced-    |
%   | {!}             | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | Np              | Dimension of the proper part in the reduced-order |
%   | {!}             | model                                             |
%   +-----------------+---------------------------------------------------+
%   | Nu              | Dimension of the unstable part in the reduced-    |
%   |                 | order model                                       |
%   +-----------------+---------------------------------------------------+
%   | RelErrBound     | computed error bound for the relative error of the|
%   | {!}             | of the reduced-order model in H-infinity norm     |
%   +-----------------+---------------------------------------------------+
%   | V               | projection matrix used as right state-space       |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%   | W               | projection matrix used as left state-space        |
%   | {!}             | transformation to obtain the resulting block      |
%   |                 | system, if opts.StoreProjection == 1              |
%   +-----------------+---------------------------------------------------+
%
%
% REFERENCE:
%   P. Benner, T. Stykel, Model order reduction for differential-algebraic
%   equations: A survey, in: A. Ilchmann, T. Reis (Eds.), Surveys in
%   Differential-Algebraic Equations IV, Differential-Algebraic Equations
%   Forum, Springer International Publishing, Cham, 2017, pp. 107--160.
%   https://doi.org/10.1007/978-3-319-46618-7_3
%
% See also ml_ct_d_ss_bst, ml_morlabopts.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

narginchk(1, 2);

if (nargin < 2) || isempty(opts)
    opts = struct();
end

% Check that struct and system type are correct.
[sys, opts, ~] = ml_decide_system_type('ct', sys, opts);

% Check system type and fill-in matrices.
switch lower(sys.SystemType)
    case 'ct_d_ss'
        warning('MORLAB:data', ...
            ['No descriptor matrix E found, set to identity E = I_n.' ...
            ' Use ml_ct_d_ss_bst to handle standard systems.']);

        sys.E = eye(size(sys.A));

    case 'ct_s_ss_default'
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['No descriptor matrix E found, set to identity.', ...
                ' System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_bst to handle sparse systems.']);
            sys.E = eye(size(sys.A));
        else
            error('MORLAB:data', ...
                ['Large-scale sparse standard system detected.' ...
                ' Use ml_ct_s_foss_bst to handle such systems.']);
        end

    case 'ct_d_dss'
        % No extra action in main supported case.

    case {'ct_s_dss_default', 'ct_s_dss_dae_1', 'ct_s_dss_dae_2'}
        if size(sys.A, 1) <= 5000
            warning('MORLAB:data', ...
                ['System matrices were converted from sparse to' ...
                ' full. Use ml_ct_s_foss_bst to handle sparse systems.']);
        else
            error('MORLAB:data', ...
                ['Large-scale sparse descriptor system detected.' ...
                ' Use ml_ct_s_foss_bst to handle such systems.']);
        end

    otherwise
        error('MORLAB:data', ...
            ['This function is not suited to handle the given' ...
            ' system type.']);
end

sys.SystemType = 'ct_d_dss';
sys            = ml_prepare_system_data(sys);

% Check and assign optional parameters.
if ml_field_set_to_value(opts, 'careopts')
    assert(isa(opts.careopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.careopts has to be a struct!');
else
    opts.careopts = struct();
end

if ml_field_set_to_value(opts, 'Epsilon')
    ml_assert_posscalar(opts.Epsilon, 'opts.Epsilon');
else
    opts.Epsilon = 1.0e-03;
end

if ml_field_set_to_value(opts, 'gdlyapopts')
    assert(isa(opts.gdlyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.gdlyapopts has to be a struct!');
else
    opts.gdlyapopts = struct();
end

if ml_field_set_to_value(opts, 'lyapopts')
    assert(isa(opts.lyapopts, 'struct'), ...
        'MORLAB:data', ...
        'The parameter opts.lyapopts has to be a struct!');
else
    opts.lyapopts = struct();
end

opts = ml_check_cell_param(opts, 'OrderComputation', ...
    @ml_assert_char, 'tolerance');

numOrderComp = length(opts.OrderComputation);
rselect      = cell(1, numOrderComp);
for k = 1:numOrderComp
    if strcmpi(opts.OrderComputation{k}, 'order')
        rselect{k} = 0;
    elseif strcmpi(opts.OrderComputation{k}, 'tolerance')
        rselect{k} = 4;
    else
        error('MORLAB:data', ...
            'The desired order computation method is not implemented!');
    end
end

if ml_field_set_to_value(opts, 'RiccatiSolver')
    assert(strcmpi(opts.RiccatiSolver, 'newton') ...
        || strcmpi(opts.RiccatiSolver, 'sign'), ...
        'MORLAB:data', ...
        'The desired Riccati equation solver is not implemented!');
else
    opts.RiccatiSolver = 'sign';
end

if ml_field_set_to_value(opts, 'StoreGramians')
    ml_assert_boolean(opts.StoreGramians, 'opts.StoreGramians');
else
    opts.StoreGramians = false;
end

if ml_field_set_to_value(opts, 'StoreProjection')
    ml_assert_boolean(opts.StoreProjection, 'opts.StoreProjection');
else
    opts.StoreProjection = false;
end

% Initial info structure.
info = struct();


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% DECOMPOSITION OF TRANSFER FUNCTION.                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Decomposition of the system into unstable, finite and infinite parts.
[sys, infoADTF] = ml_ct_d_dss_adtf(sys, opts);

ninf = size(sys.Ainf, 1);
m    = size(sys.B, 2);
p    = size(sys.C, 1);

% Assign information about additive decomposition.
info.infoADTF = rmfield(infoADTF, {'V'; 'W'});


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE MATRIX EQUATIONS.                                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Computation of the constant polynomial part.
if ninf > 0
    if isempty(sys.D)
        M = -sys.Cinf * (sys.Ainf \ sys.Binf);
    else
        M = sys.D - sys.Cinf * (sys.Ainf \ sys.Binf);
    end
else
    M = sys.D;
end

% Transpose system, if there are more outputs than inputs.
if m < p
    A = sys.A';
    B = sys.C';
    C = sys.B';
    M = M';
    E = sys.E';
else
    A = sys.A;
    B = sys.B;
    C = sys.C;
    E = sys.E;
end

% Computation of the full-rank factors of the proper Gramians.
if size(sys.A, 1) > 0
    if (m >= p) && ml_field_set_to_value(opts, 'GramFacC')
        R             = opts.GramFacC;
        info.infoLYAP = [];
    elseif (m < p) && ml_field_set_to_value(opts, 'GramFacO')
        R             = opts.GramFacO;
        info.infoLYAP = [];
    else
        [R, infoLYAP] = ml_lyap_sgn_fac(A, B, E, opts.lyapopts);
        info.infoLYAP = infoLYAP;
    end

    if (m >= p) && ml_field_set_to_value(opts, 'GramFacO')
        L             = opts.GramFacO;
        info.infoCARE = [];
    elseif (m < p) && ml_field_set_to_value(opts, 'GramFacC')
        L             = opts.GramFacC;
        info.infoCARE = [];
    else
        % Epsilon regularization.
        if (rank(M) < min(m, p)) || (norm(M, 'fro') < 1.0e+02 * eps)
            M  = opts.Epsilon * eye(min(m, p));
            Bw = B * opts.Epsilon * eye(min(m, p), max(m, p))' + ...
                E * (R * (R' * C'));
        else
            Bw     = B * M' + E * (R * (R' * C'));
            [~, M] = qr(M', 0);
        end

        A = A - Bw * ((M' * M) \ C);
        B = Bw / M;
        C = M' \ C;

        if strcmpi(opts.RiccatiSolver, 'sign')
            opts.careopts.EqnType = 'primal';
            [L, ~, infoCARE]      = ml_caredl_sgn_fac(A, B, C, E, 1, ...
                opts.careopts);
            info.infoCARE         = infoCARE;
        else
            [L, infoPCARE] = ml_pcare_nwt_fac(A, B, C, E, opts.careopts);
            info.infoPCARE = infoPCARE;
        end
    end
else
    [R, L] = deal([]);
end

% Swap Gramian factors if system was transposed.
if m < p
    [R, L] = deal(L, R);
end

% Computation of the full-rank factors of the improper Gramians.
if ninf > 0
    if ml_field_set_to_value(opts, 'IGramFacC') ...
            && ml_field_set_to_value(opts, 'IGramFacO')
        % Both precomputed Gramian factors.
        Rinf = opts.IGramFacC;
        Linf = opts.IGramFacO;
    elseif ml_field_set_to_value(opts, 'IGramFacC')
        % Precomputed controllability factor.
        [Linf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf', ...
            sys.Cinf', sys.Einf', opts.gdlyapopts);
        Rinf               = opts.IGramFacC;
        info.infoGDLYAP    = infoGDLYAP;
    elseif ml_field_set_to_value(opts, 'IGramFacO')
        % Precomputed observability factor.
        [Rinf, infoGDLYAP] = ml_gdlyap_smith_fac(sys.Ainf, ...
            sys.Binf, sys.Einf, opts.gdlyapopts);
        Linf               = opts.IGramFacO;
        info.infoGDLYAP    = infoGDLYAP;
    else
        % No Gramian factors precomputed.
        [Rinf, Linf, infoGDLYAP] = ml_gdlyapdl_smith_fac( ...
            sys.Ainf, sys.Binf, sys.Cinf, sys.Einf, opts.gdlyapopts);
        info.infoGDLYAP          = infoGDLYAP;
    end
else
    [Rinf, Linf] = deal([]);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTE REDUCED-ORDER MODEL.                                            %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute projection bases.
[Vinf, Winf, hsvi] = ml_balproj_improper(sys, Rinf, Linf, opts);

if not(isa(Vinf, 'cell')), Vinf = {Vinf}; end
if not(isa(Winf, 'cell')), Winf = {Winf}; end

[V, W, hsvp] = ml_balproj_proper(sys, R, L, ...
    size(sys.Au, 1) + size(Vinf{1}, 2), rselect, opts);

if not(isa(V, 'cell')), V = {V}; end
if not(isa(W, 'cell')), W = {W}; end

[V, W, Vinf, Winf] = ml_extend_cell(V, W, Vinf, Winf);

% Compute reduced-order model parts.
stabrom = ml_projtrunc_proper(sys, V, W);
infrom  = ml_projtrunc_improper(sys, Vinf, Winf);

% Construct final ROMs from stable, unstable and improper parts.
if not(isa(stabrom, 'cell')), stabrom = {stabrom}; end
if not(isa(infrom, 'cell')), infrom = {infrom}; end

numRoms = length(stabrom);
rom    = cell(1, numRoms);
for k = 1:numRoms
    rom{k}.A = blkdiag(stabrom{k}.A, sys.Au, infrom{k}.A);
    rom{k}.B = [stabrom{k}.B; sys.Bu; infrom{k}.B];
    rom{k}.C = [stabrom{k}.C, sys.Cu, infrom{k}.C];
    rom{k}.D = sys.D;
    rom{k}.E = blkdiag(stabrom{k}.E, sys.Eu, infrom{k}.E);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT.                                                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Assign output information.
nf          = cell(1, numRoms);
relErrBound = cell(1, numRoms);
for k = 1:numRoms
    nf{k}          = size(stabrom{k}.A, 1);
    relErrBound{k} = prod((1 + hsvp(nf{k}+1:end)) ...
        ./ (1 - hsvp(nf{k}+1:end))) - 1;
end

info.Hsvi        = hsvi;
info.Hsvp        = hsvp;
info.Ni          = size(infrom{1}.A, 1);
info.Np          = nf;
info.Nu          = size(sys.Au, 1);
info.RelErrBound = relErrBound;

% Store projection bases.
if opts.StoreProjection
    for k = 1:numRoms
        info.V{k} = infoADTF.V * blkdiag(V{k}, eye(info.Nu), Vinf{k});
        info.W{k} = infoADTF.W * blkdiag(W{k}, eye(info.Nu), Winf{k});
    end
else
    info.V = [];
    info.W = [];
end

% Store Gramian factors.
if opts.StoreGramians
    info.GramFacC  = R;
    info.GramFacO  = L;
    info.IGramFacC = Rinf;
    info.IGramFacO = Linf;
else
    info.GramFacC  = [];
    info.GramFacO  = [];
    info.IGramFacC = [];
    info.IGramFacO = [];
end

% Consistent global output formatting.
[rom, info] = ml_format_output(rom, 1, info);
