function [w, mag, tf, info] = ml_bodemag(varargin)
%ML_BODEMAG Bode magnitude plots of transfer functions.
%
% SYNTAX:
%   [w, mag, tf, info] = ML_BODEMAG(sys1, ..., sysN)
%   [w, mag, tf, info] = ML_BODEMAG(sys1, ..., sysN, opts)
%
%   [w, mag, tf, info] = ML_BODEMAG(sys, rom1, ..., romM)
%   [w, mag, tf, info] = ML_BODEMAG(sys, rom1, ..., romM, opts)
%
% DESCRIPTION:
%   This function can be used to compute and plot absolute magnitude values
%   of transfer functions in given intervals. It can be used to
%   compute/plot
%
%       abs(G1(jw)), ..., abs(GN(jw)),
%       abs(G(jw) - H1(jw)), ..., abs(G(jw) - HM(jw)),
%       abs(G(jw) - H1(jw)) / abs(G(jw)), ...,
%
%   where G1, ..., GN are the transfer functions of sys1, ..., sysN,
%   G the transfer function of sys and H1, ..., HM the transfer functions
%   of rom1, ..., romM.
%
%
% INPUTS:
%   sys1, ..., sysN - structure or state-space objects, containing
%   {!}               standard/descriptor/second-order systems
%   rom1, ..., romM - structure or state-space objects, containing
%   {!}               standard/descriptor/second-order systems
%   opts            - structure, containing the following optional entries:
%   +-----------------+---------------------------------------------------+
%   |    PARAMETER    |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AccuracyTol     | nonnegative scalar, tolerance used for the        |
%   |                 | accuracy of the adaptive sampling method          |
%   |                 | (default 0.1)                                     |
%   +-----------------+---------------------------------------------------+
%   | DiffMode        | character array, determining type of plot         |
%   |                 |  'off' - Bode magnitude plots of sys1, sys2, ...  |
%   |                 |  'abs' - absolute errors sys-rom1, sys-rom2, ...  |
%   |                 |  'rel' - relative errors sys-rom1, sys-rom2, ...  |
%   |                 |  'all' - all plot types are shown                 |
%   |                 | (default 'off')                                   |
%   +-----------------+---------------------------------------------------+
%   | FreqRange       | vector of length 2, exponents of the frequency    |
%   |                 | range, in which should be sampled                 |
%   |                 | (default [-4, 4])                                 |
%   +-----------------+---------------------------------------------------+
%   | FreqSample      | vector or character array, determining the        |
%   |                 | frequency points, where to sample                 |
%   |                 |  'adapt'   - adaptive sampling                    |
%   |                 |  'equal'   - logarithmically equidistant sampling |
%   |                 |  [numeric] - given frequency values are used      |
%   |                 | (default 'adapt')                                 |
%   +-----------------+---------------------------------------------------+
%   | FreqUnit        | character array, frequency unit of the sample     |
%   |                 | points and the plot                               |
%   |                 |  'Hz'      - frequency in Hertz                   |
%   |                 |  'rad/sec' - frequency in radian per second       |
%   |                 | (default 'rad/sec')                               |
%   +-----------------+---------------------------------------------------+
%   | Info            | {0, 1}, used to disable/enable display of         |
%   |                 | sampling steps                                    |
%   |                 | (default 0)                                       |
%   +-----------------+---------------------------------------------------+
%   | InitPoints      | positive integer >= 3 or vector of length >= 3,   |
%   |                 | initial number of equally distributed sampling    |
%   |                 | points or initial frequency vector for            |
%   |                 | opts.FreqSample == 'adapt'                        |
%   |                 | (default 3)                                       |
%   +-----------------+---------------------------------------------------+
%   | LineSpecs       | character array, determining the line             |
%   | {!}             | specifications for the plots                      |
%   |                 | (default '-')                                     |
%   +-----------------+---------------------------------------------------+
%   | MagScale        | character array, determines scaling of the        |
%   |                 | magnitude axis                                    |
%   |                 |  'log'    - creates loglog plot                   |
%   |                 |  'linear' - creat semilogx plot                   |
%   |                 | (default 'log')                                   |
%   +-----------------+---------------------------------------------------+
%   | MagUnit         | character array, magnitude unit in the plot       |
%   |                 |  'abs' - absolute values                          |
%   |                 |  'dB'  - deciBel                                  |
%   |                 | (default 'abs')                                   |
%   +-----------------+---------------------------------------------------+
%   | MaxPoints       | positive integer, maximum number of sampling      |
%   |                 | points, if DiffMode == 'equal' total number of    |
%   |                 | points                                            |
%   |                 | (default 500)                                     |
%   +-----------------+---------------------------------------------------+
%   | ShowPlot        | {0, 1}, used to disable/enable showing of plots   |
%   |                 | (default 1)                                       |
%   +-----------------+---------------------------------------------------+
%
%   Note: Arguments/Parameters marked with {!} may also be a cell array
%         containing multiple arguments.
%
% OUTPUTS:
%   w    - matrix of size (number samples) x (number systems),
%          conatining the frequency sampling points
%   mag  - 4-D array of size p x m x (number samples) x (number systems),
%          containing the absolute magnitude values
%   tf   - 4-D array of size p x m x (number samples) x (number systems),
%          containing the transfer function values
%   info - structure, containing the following information:
%   +-----------------+---------------------------------------------------+
%   |      ENTRY      |                     MEANING                       |
%   +-----------------+---------------------------------------------------+
%   | AbsErr          | if opts.DiffMode == 'off' empty, otherwise 4-D    |
%   |                 | array of size p x m x (number samples) x          |
%   |                 | (number systems - 1), containing the computed     |
%   |                 | absolute error values                             |
%   +-----------------+---------------------------------------------------+
%   | Accuracy        | vector, containing the computed accuracy values   |
%   |                 | during the adaptive sampling and initial zeros for|
%   |                 | non-adaptive sampling                             |
%   +-----------------+---------------------------------------------------+
%   | Npts            | positive integer, number of plotted points        |
%   +-----------------+---------------------------------------------------+
%   | RelErr          | if opts.DiffMode == 'off' empty, otherwise 4-D    |
%   |                 | array of size p x m x (number samples) x          |
%   |                 | (number systems - 1), containing the computed     |
%   |                 | relative error values                             |
%   +-----------------+---------------------------------------------------+
%
% See also ml_sigmaplot, ml_frobeniusplot.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHECK INPUTS.                                                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Check all input systems.
tfeval   = cell(1);
iosize   = cell(1);
tsamples = cell(1);
numSys   = 0;
hasOpts  = 0;

for k = 1:nargin()
    sysInput = varargin{k};

    if isa(sysInput, 'cell')
        assert(isvector(sysInput), ...
            'MORLAB:data', ...
            'The cell inputs must be a vector!');
        sys_length = length(sysInput);
    else
        sys_length = 1;
    end

    for j = 1:sys_length
        if isa(sysInput, 'cell')
            sys = sysInput{j};
        else
            sys = sysInput;
        end

        if isa(sys, 'struct') && isfield(sys, 'A')
            % First-order system case (struct).
            numSys = numSys + 1;

            ml_assert_stdsys(sys);

            if ml_field_set_to_value(sys, 'D')
                assert( ...
                    isequal(size(sys.D), [size(sys.C, 1), ...
                    size(sys.B, 2)]), ...
                    'MORLAB:data', ...
                    'The matrix D must have the dimensions %d x %d!', ...
                    size(sys.C, 1), size(sys.B, 2));
            else
                sys.D = zeros(size(sys.C, 1), size(sys.B, 2));
            end

            if ml_field_set_to_value(sys, 'E')
                assert(isequal(size(sys.E), size(sys.A)), ...
                    'MORLAB:data', ...
                    'The matrix E must have the same dimensions as A!');

                hasE = 1;
            else
                if issparse(sys.A)
                    sys.E = speye(size(sys.A));
                else
                    sys.E = eye(size(sys.A));
                end

                hasE = 0;
            end

            tsamples{numSys} = 0;
            if ml_field_set_to_value(sys, 'Ts')
                ml_assert_scalar(sys.Ts, 'sys.Ts');

                if not(sys.Ts == 0)
                    if sys.Ts > 0
                        tsamples{numSys} = sys.Ts;
                    else
                        tsamples{numSys} = 1;
                    end
                end
            end

            if not(exist('OCTAVE_VERSION', 'builtin')) ...
                    && not(issparse(sys.A))
                if hasE
                    [H, T, Q, Z] = hess(sys.A, sys.E);
                    CH           = sys.C * Z;
                    BH           = Q * sys.B;
                else
                    [Q, H] = hess(sys.A);
                    CH     = sys.C * Q;
                    BH     = Q' * sys.B;
                    T      = sys.E;
                end
                isHess = 1;
            else
                isHess = 0;
            end

            if not(tsamples{numSys})
                if size(sys.D, 2) <= size(sys.D, 1)
                    if isHess
                        tfeval{numSys} = @(w) CH * linsolve((1i * w) ...
                            * T - H, BH, struct('UHESS', true)) + sys.D;
                    else
                        tfeval{numSys} = @(w) sys.C * (((1i * w) ...
                            * sys.E - sys.A) \ sys.B) + sys.D;
                    end
                else
                    if isHess
                        tfeval{numSys} = @(w) linsolve((1i * w) ...
                            * T - H, CH', struct('UHESS', true, ...
                            'TRANSA', true))' * BH + sys.D;
                    else
                        tfeval{numSys} = @(w) (sys.C / ((1i * w) ...
                            * sys.E - sys.A)) * sys.B  + sys.D;
                    end
                end
            else
                if size(sys.D, 2) <= size(sys.D, 1)
                    if isHess
                        tfeval{numSys} = @(w) CH * linsolve(...
                            exp((1i * w) * tsamples{numSys}) ...
                            * T - H, BH, struct('UHESS', true)) + sys.D;
                    else
                        tfeval{numSys} = @(w) sys.C * ((exp((1i * w) ...
                            * tsamples{numSys}) * sys.E - sys.A) ...
                            \ sys.B) + sys.D;
                    end
                else
                    if isHess
                        tfeval{numSys} = @(w) linsolve(exp((1i * w) ...
                            * tsamples{numSys}) * T - H, CH', ...
                            struct('UHESS', true, 'TRANSA', true))' ...
                            * BH + sys.D;
                    else
                        tfeval{numSys} = @(w) (sys.C / (exp((1i * w) ...
                            * tsamples{numSys}) * sys.E - sys.A)) ...
                            * sys.B + sys.D;
                    end
                end
            end

            iosize{numSys} = size(sys.D);

        elseif isa(sys, 'ss')
            % First-order system case (state-space).
            numSys = numSys + 1;

            if isempty(sys.e)
                sys.e = eye(size(sys.a));
                hasE  = 0;
            else
                hasE = 1;
            end

            if sys.Ts < 0
                tsamples{numSys} = 1;
            else
                tsamples{numSys} = sys.Ts;
            end

            if not(exist('OCTAVE_VERSION', 'builtin'))
                if hasE
                    [H, T, Q, Z] = hess(sys.a, sys.e);
                    CH           = sys.c * Z;
                    BH           = Q * sys.b;
                else
                    [Q, H] = hess(sys.a);
                    CH     = sys.c * Q;
                    BH     = Q' * sys.b;
                    T      = sys.e;
                end
                isHess = 1;
            else
                isHess = 0;
            end

            if not(tsamples{numSys})
                if size(sys.d, 2) <= size(sys.d, 1)
                    if isHess
                        tfeval{numSys} = @(w) CH * linsolve((1i * w) ...
                            * T - H, BH, struct('UHESS', true)) + sys.d;
                    else
                        tfeval{numSys} = @(w) sys.c * (((1i * w) ...
                            * sys.e - sys.a) \ sys.b) + sys.d;
                    end
                else
                    if isHess
                        tfeval{numSys} = @(w) linsolve((1i * w) ...
                            * T - H, CH', struct('UHESS', true, ...
                            'TRANSA', true))' * BH + sys.d;
                    else
                        tfeval{numSys} = @(w) (sys.c / ((1i * w) ...
                            * sys.e - sys.a)) * sys.b  + sys.d;
                    end
                end
            else
                if size(sys.D, 2) <= size(sys.D, 1)
                    if isHess
                        tfeval{numSys} = @(w) CH * linsolve(...
                            exp((1i * w) * tsamples{numSys}) ...
                            * T - H, BH, struct('UHESS', true)) + sys.D;
                    else
                        tfeval{numSys} = @(w) sys.c * ((exp((1i * w) ...
                            * tsamples{numSys}) * sys.e - sys.a) ...
                            \ sys.b) + sys.d;
                    end
                else
                    if isHess
                        tfeval{numSys} = @(w) linsolve(exp((1i * w) ...
                            * tsamples{numSys}) * T - H, CH', ...
                            struct('UHESS', true, 'TRANSA', true))' ...
                            * BH + sys.d;
                    else
                        tfeval{numSys} = @(w) (sys.c / (exp((1i * w) ...
                            * tsamples{numSys}) * sys.e - sys.a)) ...
                            * sys.b + sys.d;
                    end
                end
            end

            iosize{numSys} = size(sys.d);

        elseif isa(sys, 'sparss')
            % Sparse state-space object.
            numSys = numSys + 1;

            if sys.Ts < 0
                tsamples{numSys} = 1;
            else
                tsamples{numSys} = sys.Ts;
            end

            if not(tsamples{numSys})
                if size(sys.D, 2) <= size(sys.D, 1)
                    tfeval{numSys} = @(w) sys.C * (((1i * w) ...
                        * sys.E - sys.A) \ sys.B) + sys.D;
                else
                    tfeval{numSys} = @(w) (sys.C / ((1i * w) ...
                        * sys.E - sys.A)) * sys.B  + sys.D;
                end
            else
                if size(sys.D, 2) <= size(sys.D, 1)
                    tfeval{numSys} = @(w) sys.C * ((exp((1i * w) ...
                        * tsamples{numSys}) * sys.E - sys.A) ...
                        \ sys.B) + sys.D;
                else
                    tfeval{numSys} = @(w) (sys.C / (exp((1i * w) ...
                        * tsamples{numSys}) * sys.E - sys.A)) ...
                        * sys.B + sys.D;
                end
            end

            iosize{numSys} = size(sys.D);

        elseif isa(sys, 'struct') && isfield(sys, 'K')
            % Second-order system case.
            numSys = numSys + 1;

            ml_assert_sosys(sys);

            tsamples{numSys} = 0;
            if ml_field_set_to_value(sys, 'Ts')
                ml_assert_scalar(sys.Ts, 'sys.Ts');

                if not(sys.Ts == 0)
                    if sys.Ts > 0
                        tsamples{numSys} = sys.Ts;
                    else
                        tsamples{numSys} = 1;
                    end
                end
            end

            if not(ml_field_set_to_value(sys, 'Cp'))
                sys.Cp = zeros(size(sys.Cv));
            end

            if not(ml_field_set_to_value(sys, 'Cv'))
                sys.Cv = zeros(size(sys.Cp));
            end

            if ml_field_set_to_value(sys, 'D')
                if ml_field_set_to_value(sys, 'Cp')
                    assert(isequal(size(sys.D), ...
                        [size(sys.Cp, 1), size(sys.Bu, 2)]), ...
                        'MORLAB:data', ...
                        ['The matrix D must have the dimensions ' ...
                        '%d x %d!'], ...
                        size(sys.Cp, 1), size(sys.Bu, 2));
                else
                    assert(isequal(size(sys.D), ...
                        [size(sys.Cv, 1), size(sys.Bu, 2)]), ...
                        'MORLAB:data', ...
                        ['The matrix D must have the dimensions ' ...
                        '%d x %d!'], ...
                        size(sys.Cv, 1), size(sys.Bu, 2));
                end
            else
                if ml_field_set_to_value(sys, 'Cp')
                    sys.D = zeros(size(sys.Cp, 1), size(sys.Bu, 2));
                else
                    sys.D = zeros(size(sys.Cv, 1), size(sys.Bu, 2));
                end
            end

            if not(tsamples{numSys})
                if size(sys.D, 2) <= size(sys.D, 1)
                    tfeval{numSys} = @(w) (sys.Cp + (1i * w) * sys.Cv) ...
                        * ((-w^2 * sys.M + 1i * w * sys.E + sys.K) ...
                        \ sys.Bu) + sys.D;
                else
                    tfeval{numSys} = @(w) ((sys.Cp + (1i * w) * sys.Cv) ...
                        / (-w^2 * sys.M + 1i * w * sys.E + sys.K)) ...
                        * sys.Bu + sys.D;
                end
            else
                if size(sys.D, 2) <= size(sys.D, 1)
                    tfeval{numSys} = @(w) (sys.Cp ...
                        + exp((1i * w) * tsamples{numSys}) * sys.Cv) ...
                        * ((exp((1i * w) * tsamples{numSys})^2 * sys.M ...
                        + exp((1i * w) * tsamples{numSys}) * sys.E ...
                        + sys.K) \ sys.Bu) + sys.D;
                else
                    tfeval{numSys} = @(w) ((sys.Cp ...
                        + exp((1i * w) * tsamples{numSys}) * sys.Cv) ...
                        / (exp((1i * w) * tsamples{numSys})^2 * sys.M ...
                        + exp((1i * w) * tsamples{numSys}) * sys.E ...
                        + sys.K)) ...
                        * sys.Bu + sys.D;
                end
            end

            tsamples{numSys} = 0;
            iosize{numSys}   = size(sys.D);

        elseif isa(sys, 'mechss')
            % Second-order system case.
            numSys = numSys + 1;

            if sys.Ts < 0
                tsamples{numSys} = 1;
            else
                tsamples{numSys} = sys.Ts;
            end

            if not(tsamples{numSys})
                if size(sys.D, 2) <= size(sys.D, 1)
                    tfeval{numSys} = @(w) (sys.F + (1i * w) * sys.G) ...
                        * ((-w^2 * sys.M + (1i * w) * sys.C + sys.K) ...
                        \ sys.B) + sys.D;
                else
                    tfeval{numSys} = @(w) ((sys.F + (1i * w) * sys.G) ...
                        / (-w^2 * sys.M + (1i * w) * sys.C + sys.K)) ...
                        * sys.B + sys.D;
                end
            else
                if size(sys.D, 2) <= size(sys.D, 1)
                    tfeval{numSys} = @(w) (sys.F ...
                        + exp((1i * w) * tsamples{numSys}) * sys.G) ...
                        * ((exp((1i * w) * tsamples{numSys})^2 * sys.M ...
                        + exp((1i * w) * tsamples{numSys}) * sys.C ...
                        + sys.K) \ sys.B) + sys.D;
                else
                    tfeval{numSys} = @(w) ((sys.F ...
                        + exp((1i * w) * tsamples{numSys}) * sys.G) ...
                        / (exp((1i * w) * tsamples{numSys})^2 * sys.M ...
                        + exp((1i * w) * tsamples{numSys}) * sys.C ...
                        + sys.K)) ...
                        * sys.B + sys.D;
                end
            end

            tsamples{numSys} = 0;
            iosize{numSys}   = size(sys.D);

        elseif k < nargin()
            % Unsupported input.
            if isa(sys, 'struct')
                error('MORLAB:data', ...
                    'The given system structure is not supported!');
            else
                error('MORLAB:data', ...
                    'The systems must be structs or ss/dss objects!');
            end
        else
            % Wrong format of option structure.
            assert(isa(sys, 'struct'), ...
                'MORLAB:data', ...
                'The input argument opts must be a structure!');

            hasOpts = 1;
        end
    end
end

% Check minimum number of given systems.
assert(numSys > 0, ...
    'MORLAB:data', ...
    'Give at least one system for plotting!');

% Check number of inputs and outputs.
assert(all(cellfun(@(s) isequal(iosize{1}, s), iosize)), ...
    'MORLAB:data', ...
    'Number of inputs and outputs have to be the same for all systems!');

% Get last input as possible option struct if not a system.
if hasOpts
    opts = varargin{end};
else
    opts = struct();
end

% Check and set optional parameters.
if ml_field_set_to_value(opts, 'AccuracyTol')
    ml_assert_nonnegscalar(opts.AccuracyTol, 'opts.AccuracyTol');
else
    opts.AccuracyTol = 0.1;
end

if ml_field_set_to_value(opts, 'DiffMode')
    ml_assert_char(opts.DiffMode, 'opts.DiffMode')
    assert(strcmpi(opts.DiffMode, 'off') ...
        || strcmpi(opts.DiffMode, 'abs') ...
        || strcmpi(opts.DiffMode, 'rel') ...
        || strcmpi(opts.DiffMode, 'all'), ...
        'MORLAB:data', ...
        'The requested difference mode is not implemented!');

    if (numSys == 1) && not(strcmpi(opts.DiffMode, 'off'))
        warning('MORLAB:data', ...
            ['The difference mode needs at least 2 systems.\n' ...
            'Difference mode is turned off!']);
        opts.DiffMode = 'off';
    end
else
    opts.DiffMode = 'off';
end

if ml_field_set_to_value(opts, 'FreqRange')
    ml_assert_vector(opts.FreqRange, 'opts.FreqRange');
    assert(length(opts.FreqRange) == 2, ...
        'MORLAB:data', ...
        'The parameter opts.FreqRange must be of length 2!');
else
    opts.FreqRange = [-4, 4];
end

if ml_field_set_to_value(opts, 'FreqSample')
    if isnumeric(opts.FreqSample)
        ml_assert_nonnegvector(opts.FreqSample, 'opts.FreqSample')
        assert(all(opts.FreqSample > 0), ...
            'MORLAB:data', ...
            'The parameter opts.FreqSample must be positive.');

    else
        ml_assert_char(opts.FreqSample, 'opts.FreqSample')
        assert(strcmpi(opts.FreqSample, 'equal') ...
            || strcmpi(opts.FreqSample, 'adapt'), ...
            'MORLAB:data', ...
            'The requested frequency sampling is not implemented!');
    end
else
    opts.FreqSample = 'adapt';
end

if ml_field_set_to_value(opts, 'FreqUnit')
    ml_assert_char(opts.FreqUnit, 'opts.FreqUnit')
    assert(strcmpi(opts.FreqUnit, 'rad/sec') ...
        || strcmpi(opts.FreqUnit, 'Hz'), ...
        'MORLAB:data', ...
        'The requested frequency unit is not implemented!');
else
    opts.FreqUnit = 'rad/sec';
end

if ml_field_set_to_value(opts, 'Info')
    ml_assert_boolean(opts.Info, 'opts.Info');
else
    opts.Info = false;
end

if ml_field_set_to_value(opts, 'InitPoints')
    if isscalar(opts.InitPoints)
        ml_assert_posinteger(opts.InitPoints, 'opts.InitPoints');
        assert(opts.InitPoints >= 3, ...
            'MORLAB:data', ...
            ['The parameter opts.InitPoints has to be larger or ' ...
            'equal to 3!']);
    else
        ml_assert_nonnegvector(opts.InitPoints, 'opts.InitPoints')
        assert(all(opts.InitPoints > 0), ...
            'MORLAB:data', ...
            'The parameter opts.InitPoints must be positive.');
        assert(length(opts.InitPoints) >= 3, ...
            'MORLAB:data', ...
            ['The parameter opts.InitPoints must be at least of ' ...
            'length 3.']);
    end
else
    opts.InitPoints = 3;
end

opts          = ml_check_cell_param(opts, 'LineSpecs', ...
    @ml_assert_char, '-');
numLineSpecs = length(opts.LineSpecs);

if ml_field_set_to_value(opts, 'MagScale')
    ml_assert_char(opts.MagScale, 'opts.MagScale')
    assert(strcmpi(opts.MagScale, 'linear') ...
        || strcmpi(opts.MagScale, 'log'), ...
        'MORLAB:data', ...
        'The requested magnitude scaling is not implemented!');
else
    opts.MagScale = 'log';
end

if ml_field_set_to_value(opts, 'MagUnit')
    ml_assert_char(opts.MagUnit, 'opts.MagUnit')
    assert(strcmpi(opts.MagUnit, 'abs') ...
        || strcmpi(opts.MagUnit, 'db'), ...
        'MORLAB:data', ...
        'The requested magnitude unit is not implemented!');
else
    opts.MagUnit = 'abs';
end

if ml_field_set_to_value(opts, 'MaxPoints')
    ml_assert_posinteger(opts.MaxPoints, 'opts.MaxPoints');
else
    opts.MaxPoints = 500;
end

if ml_field_set_to_value(opts, 'ShowPlot')
    ml_assert_boolean(opts.ShowPlot, 'opts.ShowPlot');
else
    opts.ShowPlot = true;
end

% Check sample times.
assert(all(cellfun(@(s) isequal(tsamples{1}, s), tsamples)) || ...
    strcmpi(opts.DiffMode, 'off'), ...
    'MORLAB:data', ...
    'Sample times need to be the same for error computation!');


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% COMPUTATION OF FUNCTION VALUES.                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

nyqFreq = zeros(1, numSys);

% Sample frequency points.
if isnumeric(opts.FreqSample)
    % Given frequency points.
    w              = log10(opts.FreqSample(:));
    opts.MaxPoints = length(w);
    iter           = length(w);

    % Copy initial frequencies for all given systems.
    w = repmat(w, 1, numSys);
elseif strcmpi(opts.FreqSample, 'adapt')
    % Adaptive sampling.
    w = zeros(opts.MaxPoints, numSys);

    for j = 1:numSys
        if isscalar(opts.InitPoints)
            iter = opts.InitPoints;

            if not(tsamples{j})
                maxFreq = opts.FreqRange(2);
            else
                if strcmpi(opts.FreqUnit, 'Hz')
                    nyqFreq(j) = 1 / (2 * tsamples{j});
                else
                    nyqFreq(j) = pi / tsamples{j};
                end
                maxFreq = min(opts.FreqRange(2), log10(nyqFreq(j)));
            end

            w(1:iter, j) = linspace(opts.FreqRange(1), maxFreq, iter);
        else
            iter         = length(opts.InitPoints);
            w(1:iter, j) = log10(opts.InitPoints(:));
        end
    end
else
    % Equidistant sampling.
    iter = opts.MaxPoints;
    w    = zeros(opts.MaxPoints, numSys);

    for j = 1:numSys
        if not(tsamples{j})
            maxFreq = opts.FreqRange(2);
        else
            if strcmpi(opts.FreqUnit, 'Hz')
                nyqFreq(j) = 1 / (2 * tsamples{j});
            else
                nyqFreq(j) = pi / tsamples{j};
            end
            maxFreq = min(opts.FreqRange(2), log10(nyqFreq(j)));
        end

        w(1:iter, j) = linspace(opts.FreqRange(1), maxFreq, iter)';
    end
end

% Shifting of frequencies for Hertz.
if strcmpi(opts.FreqUnit, 'Hz')
    w(1:iter, :) = w(1:iter, :) + log10(2 * pi);
end

% Vector for accuracy values.
accuracy = zeros(max([iter, opts.MaxPoints]), 1);

% Compute intial equally distributed points.
tf  = zeros(iosize{1}(1), iosize{1}(2), size(w, 1), numSys);
mag = zeros(iosize{1}(1), iosize{1}(2), size(w, 1), numSys);

if not(strcmpi(opts.DiffMode, 'off'))
    abserr = zeros(iosize{1}(1), iosize{1}(2), size(w, 1), numSys - 1);
    relerr = zeros(iosize{1}(1), iosize{1}(2), size(w, 1), numSys - 1);
end

if opts.Info
    fprintf(1, 'Equally distributed sampling:\n');
    fprintf(1, '=============================\n');
end

for j = 1:numSys
    for k = 1:iter
        if opts.Info
            if numSys > 1
                fprintf(1, 'System %2d, Step: %3d\n', j, k);
            else
                fprintf(1, 'BODEMAG Step: %3d\n', k);
            end
        end

        tf(:, :, k, j)  = tfeval{j}(10^w(k, j));
        mag(:, :, k, j) = abs(tf(:, :, k, j));

        if not(strcmpi(opts.DiffMode, 'off')) && (j >= 2)
            abserr(:, :, k, j-1) = abs(tf(:, :, k, 1) - tf(:, :, k, j));
            relerr(:, :, k, j-1) = abserr(:, :, k, j-1) ./ mag(:, :, k, 1);
        end
    end
end

% Compute adaptive set points.
if opts.Info && (iter < opts.MaxPoints)
    fprintf(1, '\nAdaptively distributed sampling:\n');
    fprintf(1, '================================\n');
end

idxAngles = zeros(1, numSys);
I         = repmat((1:iter)', 1, numSys);

while iter < opts.MaxPoints
    % Compute maximum change of consecutive scaled points.
    switch lower(opts.DiffMode)
        case 'off'
            [maxAngles, idxAngles] = get_max_angles(w, mag, I, iter, ...
                numSys, opts);
            maxAngles              = max(maxAngles);

        case 'abs'
            [maxAngles, I1] = get_max_angles(w, abserr, I, iter, ...
                numSys - 1, opts);
            [maxAngles, I2] = max(maxAngles);
            idxAngles(:)    = I1(I2);

        case 'rel'
            [maxAngles, I1] = get_max_angles(w, relerr, I, iter, ...
                numSys - 1, opts);
            [maxAngles, I2] = max(maxAngles);
            idxAngles(:)    = I1(I2);

        case 'all'
            [maxAngles1, I1] = get_max_angles(w, mag, I, iter, ...
                numSys, opts);
            [maxAngles2, I2] = get_max_angles(w, abserr, I, iter, ...
                numSys - 1, opts);
            [maxAngles3, I3] = get_max_angles(w, relerr, I, iter, ...
                numSys - 1, opts);
            maxAnglesFull    = [maxAngles1, maxAngles2, maxAngles3];
            Ifull            = [I1, I2, I3];
            [maxAngles, I4]  = max(maxAnglesFull);
            idxAngles(:)     = Ifull(I4);

    end

    % If no index was found set to 1.
    idxAngles(idxAngles == 0) = 1;

    % Subdivide an interval.
    iter = iter + 1;
    for j = 1:numSys
        x = (w(I(idxAngles(j), j), j) + w(I(idxAngles(j)+1, j), j)) / 2;
        w(iter, j)        = x;
        tf(:, :, iter, j)  = tfeval{j}(10^x);
        mag(:, :, iter, j) = abs(tf(:, :, iter, j));

        if not(strcmpi(opts.DiffMode, 'off')) && (j >= 2)
            abserr(:, :, iter, j-1) = ...
                abs(tf(:, :, iter, 1) - tf(:, :, iter, j));
            relerr(:, :, iter, j-1) = ...
                abserr(:, :, iter, j-1) ./ mag(:, :, iter, 1);
        end
    end

    % Resort all new points.
    [~, I] = sort(w(1:iter, :), 1);

    % Termination criterion.
    accuracy(iter) = maxAngles / (2 * pi ...
        + 0.5 * abs(max(max(w(1:iter, :))) ...
        - min(min(w(1:iter, :))))) + 0.045;

    if opts.Info
        fprintf(1, 'BODEMAG Step: %3d | Accuracy: %1.3e\n', ...
            iter, accuracy(iter));
    end

    if accuracy(iter) < opts.AccuracyTol
        break;
    end
end

% Order computed values.
for j = 1: numSys
    w(1:iter, j)         = w(I(:, j), j);
    tf(:, :, 1:iter, j)  = tf(:, :, I(:, j), j);
    mag(:, :, 1:iter, j) = mag(:, :, I(:, j), j);

    if not(strcmpi(opts.DiffMode, 'off')) && (j >= 2)
        abserr(:, :, 1:iter, j-1) = abserr(:, :, I(:, j-1), j-1);
        relerr(:, :, 1:iter, j-1) = relerr(:, :, I(:, j-1), j-1);
    end
end

% Remove not computed values.
w        = w(1:iter, :);
tf       = tf(:, :, 1:iter, :);
mag      = mag(:, :, 1:iter, :);
accuracy = accuracy(1:iter);

if not(strcmpi(opts.DiffMode, 'off'))
    abserr = abserr(:, :, 1:iter, :);
    relerr = relerr(:, :, 1:iter, :);
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CONSTRUCTION OF PLOTS.                                                  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Shifting of frequencies for Hertz.
if strcmpi(opts.FreqUnit, 'Hz')
    w         = w - log10(2 * pi);
    freqLabel = 'Frequency \omega (Hz)';
else
    freqLabel = 'Frequency \omega (rad/sec)';
end

% Scaling of magnitudes for decibel.
if strcmpi(opts.MagUnit, 'dB')
    magPlot = 20 .* log10(mag);
    magUnit = ' (dB)';

    if not(strcmpi(opts.DiffMode, 'off'))
        absPlot = 20 .* log10(abserr);
        relPlot = 20 .* log10(relerr);
    end
else
    magPlot = mag;
    magUnit = '';

    if not(strcmpi(opts.DiffMode, 'off'))
        absPlot = abserr;
        relPlot = relerr;
    end
end

% Set magnitude labels.
magLabel = ['|G(j\omega)|' magUnit];
absLabel = ['|G(j\omega) - H(j\omega)|' magUnit];
relLabel = ['|G(j\omega) - H(j\omega)| / |G(j\omega)|' magUnit];

% Transform from exponents to frequency values.
w = 10.^w;

% Save hold status for figure window.
if opts.ShowPlot
    holdStatus = ishold();
else
    holdStatus = 1;
end

% Set line specifications for plots.
lineSpecs = cell(1, numSys);
for j = 1:numSys
    if numLineSpecs == 1
        lineSpecs{j} = opts.LineSpecs{1};
    elseif j <= numLineSpecs
        lineSpecs{j} = opts.LineSpecs{j};
    else
        lineSpecs{j} = '-';
    end
end

% Choose plot type.
if strcmpi(opts.MagScale, 'log')
    plotFunction = @(w, mag, spec) loglog(w, mag, spec);
else
    plotFunction = @(w, mag, spec) semilogx(w, mag, spec);
end

% Correct color choices for 'all' plots by setting first one black.
if strcmpi(opts.DiffMode, 'all')
    setColor = zeros(1, numSys);

    for colors = {'r', 'g', 'b', 'c', 'm', 'y', 'k', 'w'}
        setColor = setColor + cellfun(@(s) ismember(colors{1}, s), ...
            opts.LineSpecs);
    end

    if not(any(setColor))
        lineSpecs{1} = [lineSpecs{1} 'k'];
    end
end

% Plot transfer functions.
if opts.ShowPlot
    switch lower(opts.DiffMode)
        case 'off'
            for kp = 1:iosize{1}(1)
                for km = 1:iosize{1}(2)
                    subplot(iosize{1}(1), iosize{1}(2), ...
                        iosize{1}(2) * (kp - 1) + km);

                    for j = 1:numSys
                        plotFunction(w(:, j), ...
                            reshape(magPlot(kp, km, :, j), [], 1), ...
                            lineSpecs{j});
                        hold on;
                    end

                    wlimit = get(gca, 'xlim');
                    for j = 1:numSys
                        if any(nyqFreq(j)) && (nyqFreq(j) <= wlimit(2))
                            plotFunction([nyqFreq(j), nyqFreq(j)], ...
                                get(gca, 'ylim'), 'k');
                        end
                    end

                    if km == 1
                        if iosize{1}(1) > 1
                            ylabel(sprintf('%s Out(%d)', magLabel, kp));
                        else
                            ylabel(magLabel);
                        end
                    end

                    if kp == iosize{1}(1)
                        if iosize{1}(2) > 1
                            xlabel(sprintf('%s In(%d)', freqLabel, km));
                        else
                            xlabel(freqLabel);
                        end
                    end
                end
            end

        case 'abs'
            for kp = 1:iosize{1}(1)
                for km = 1:iosize{1}(2)
                    subplot(iosize{1}(1), iosize{1}(2), ...
                        iosize{1}(2) * (kp - 1) + km);

                    for j = 1:numSys-1
                        plotFunction(w(:, j), ...
                            reshape(absPlot(kp, km, :, j), [], 1), ...
                            lineSpecs{j});
                        hold on;
                    end

                    wlimit = get(gca, 'xlim');
                    if any(nyqFreq) && (nyqFreq(1) <= wlimit(2))
                        plotFunction([nyqFreq(1), nyqFreq(1)], ...
                            get(gca, 'ylim'), 'k');
                    end

                    if km == 1
                        if iosize{1}(1) > 1
                            ylabel(sprintf('%s Out(%d)', absLabel, kp));
                        else
                            ylabel(absLabel);
                        end
                    end

                    if kp == iosize{1}(1)
                        if iosize{1}(2) > 1
                            xlabel(sprintf('%s In(%d)', freqLabel, km));
                        else
                            xlabel(freqLabel);
                        end
                    end
                end
            end

        case 'rel'
            for kp = 1:iosize{1}(1)
                for km = 1:iosize{1}(2)
                    subplot(iosize{1}(1), iosize{1}(2), ...
                        iosize{1}(2) * (kp - 1) + km);

                    for j = 1:numSys-1
                        plotFunction(w(:, j), ...
                            reshape(relPlot(kp, km, :, j), [], 1), ...
                            lineSpecs{j});
                        hold on;
                    end

                    wlimit = get(gca, 'xlim');
                    if any(nyqFreq) && (nyqFreq(1) <= wlimit(2))
                        plotFunction([nyqFreq(1), nyqFreq(1)], ...
                            get(gca, 'ylim'), 'k');
                    end

                    if km == 1
                        if iosize{1}(1) > 1
                            ylabel(sprintf('%s Out(%d)', relLabel, kp));
                        else
                            ylabel(relLabel);
                        end
                    end

                    if kp == iosize{1}(1)
                        if iosize{1}(2) > 1
                            xlabel(sprintf('%s In(%d)', freqLabel, km));
                        else
                            xlabel(freqLabel);
                        end
                    end
                end
            end

        case 'all'
            for kp = 1:iosize{1}(1)
                for km = 1:iosize{1}(2)
                    subplot(iosize{1}(1), iosize{1}(2), ...
                        iosize{1}(2) * (kp - 1) + km);

                    for j = 1:numSys
                        plotFunction(w(:, j), ...
                            reshape(magPlot(kp, km, :, j), [], 1), ...
                            lineSpecs{j});
                        hold on;
                    end

                    wlimit = get(gca, 'xlim');
                    if any(nyqFreq) && (nyqFreq(1) <= wlimit(2))
                        plotFunction([nyqFreq(1), nyqFreq(1)], ...
                            get(gca, 'ylim'), 'k');
                    end

                    if km == 1
                        if iosize{1}(1) > 1
                            ylabel(sprintf('%s Out(%d)', magLabel, kp));
                        else
                            ylabel(magLabel);
                        end
                    end

                    if kp == iosize{1}(1)
                        if iosize{1}(2) > 1
                            xlabel(sprintf('%s In(%d)', freqLabel, km));
                        else
                            xlabel(freqLabel);
                        end
                    end
                end
            end

            figure();
            for kp = 1:iosize{1}(1)
                for km = 1:iosize{1}(2)
                    subplot(iosize{1}(1), iosize{1}(2), ...
                        iosize{1}(2) * (kp - 1) + km);

                    for j = 1:numSys-1
                        plotFunction(w(:, j), ...
                            reshape(absPlot(kp, km, :, j), [], 1), ...
                            lineSpecs{j+1});
                        hold on;
                    end

                    if any(nyqFreq) && (nyqFreq(1) <= wlimit(2))
                        plotFunction([nyqFreq(1), nyqFreq(1)], ...
                            get(gca, 'ylim'), 'k');
                    end

                    if km == 1
                        if iosize{1}(1) > 1
                            ylabel(sprintf('%s Out(%d)', absLabel, kp));
                        else
                            ylabel(absLabel);
                        end
                    end

                    if kp == iosize{1}(1)
                        if iosize{1}(2) > 1
                            xlabel(sprintf('%s In(%d)', freqLabel, km));
                        else
                            xlabel(freqLabel);
                        end
                    end
                end
            end

            figure();
            for kp = 1:iosize{1}(1)
                for km = 1:iosize{1}(2)
                    subplot(iosize{1}(1), iosize{1}(2), ...
                        iosize{1}(2) * (kp - 1) + km);

                    for j = 1:numSys-1
                        plotFunction(w(:, j), ...
                            reshape(relPlot(kp, km, :, j), [], 1), ...
                            lineSpecs{j+1});
                        hold on;
                    end

                    if any(nyqFreq) && (nyqFreq(1) <= wlimit(2))
                        plotFunction([nyqFreq(1), nyqFreq(1)], ...
                            get(gca, 'ylim'), 'k');
                    end

                    if km == 1
                        if iosize{1}(1) > 1
                            ylabel(sprintf('%s Out(%d)', relLabel, kp));
                        else
                            ylabel(relLabel);
                        end
                    end

                    if kp == iosize{1}(1)
                        if iosize{1}(2) > 1
                            xlabel(sprintf('%s In(%d)', freqLabel, km));
                        else
                            xlabel(freqLabel);
                        end
                    end
                end
            end
    end
end

% Turn hold status to normal.
if not(holdStatus)
    hold off;
end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ASSIGN OUTPUT INFORMATION.                                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if strcmpi(opts.DiffMode, 'off')
    [abserr, relerr] = deal([]);
end

info = struct( ...
    'AbsErr'  , abserr, ...
    'Accuracy', accuracy, ...
    'Npts'    , iter, ...
    'RelErr'  , relerr);

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% HELPER FUNCTIONS.                                                       %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function angle = get_angle(u, v)
%GET_ANGLE Comptues angles between vectors and adds their lengths.
%
% SYNTAX:
%   angles = GET_ANGLES(u, v)
%
% DESCRIPTION:
%   This function computes the smallest angle between two vectors
%   and adds half of their lengths.
%
% INPUTS:
%   u - first vector
%   v - second vector
%
% OUTPUTS:
%   angle - resulting angle with lengths

x = dot(u, v) / (norm(v, 2) * norm(u, 2));

if x > 1
    x = 1;
elseif x < -1
    x = -1;
end

angle = pi - acos(x);

end


function [maxAngles, idxAngles] = get_max_angles(w, data, idx, m, n, opts)
%GET_MAX_ANGLES Get maximum change of sampled transfer functions.
%
% DESCRIPTION:
%   This function computes the maximum change of transfer function samples
%   to determine the next point to sample. It uses as criteria the angles
%   of the points enclosing the intervals, the length of the intervals and
%   the distance in the frequency points.
%
% INPUTS:
%   w    - matrix of dimensions (m x n), containing the frequency points'
%          exponents
%   data - matrix of dimensions (m x n), containing the absolute values of
%          the transfer function
%   idx  - index array of dimensions (m x n), for sorting w and data
%   m    - positive integer, number of already computed data points
%   n    - positive integer, number of underlying systems
%   opts - struct, containing the nonnegative scalar opts.AccuracyTol
%
% OUTPUTS:
%   maxAngles - vector of length n, containing the maximum angles of the
%               systems
%   idxAngles - vector of length n, indices corresponding to the maxAngles
%               vector

wmin      = min(min(w(1:m, 1:n)));
wmax      = max(max(w(1:m, 1:n)));
idxAngles = zeros(1, n);
maxAngles = zeros(1, n);

mindat = min(min(min(data(:, :, 1:m, 1:n), [], 1), [], 2), [], 3);
maxdat = max(max(max(data(:, :, 1:m, 1:n), [], 1), [], 2), [], 3);

for j = 1:n
    for kp = 1:size(data, 1)
        for km = 1:size(data, 2)
            % Determinine special data cases and corresponding scaling.
            if maxdat(j) == 0
                data(kp, km, data(kp, km, :, j) == 0, j) = eps;
                maxdat(j)                                = eps;
                mindat(j)                                = eps;

                scale = abs(wmax - wmin) ...
                    ./ abs(log10(maxdat(j)));
            elseif mindat(j) == 0
                data(kp, km, data(kp, km, :, j) == 0, j) = maxdat(j) * eps;
                mindat(j)                                = maxdat(j) * eps;

                scale = abs(wmax - wmin) ...
                    ./ abs(log10(maxdat(j)) - log10(mindat(j)));
            elseif abs(log10(maxdat(j)) - log10(mindat(j))) < 1.0e-12
                if abs(log10(maxdat(j))) < 1.e-12
                    scale = abs(wmax - wmin);
                else
                    scale = abs(wmax - wmin) ...
                        ./ abs(log10(maxdat(j)));
                end
            else
                scale = abs(wmax - wmin) ...
                    ./ abs(log10(maxdat(j)) - log10(mindat(j)));
            end

            % Scale data and find new points.
            for k = 1:m - 2
                v = [w(idx(k, j), j); ...
                    scale * log10(data(kp, km, idx(k, j), j))] ...
                    - [w(idx(k+1, j), j); ...
                    scale * log10(data(kp, km, idx(k+1, j), j))];
                u = [w(idx(k+2, j), j); ...
                    scale * log10(data(kp, km, idx(k+2, j), j))] ...
                    - [w(idx(k+1, j), j); ...
                    scale * log10(data(kp, km, idx(k+1, j), j))];

                % Determine new angles if regions are not too close.
                regTol = 5.0e-03 * opts.AccuracyTol;
                if (abs(v(1)) <  regTol) ...
                        || (abs(u(1)) <  regTol) ...
                        || (norm(v, 2) < regTol) ...
                        || (norm(u, 2) < regTol)
                    angleNew = 0;
                else
                    angleNew = get_angle(u, v);
                end

                % Determine new maximum interval.
                if k == 1
                    [maxAngles(j), idxMax] = max([maxAngles(j), ...
                        angleNew + norm(v, 2) + abs(v(1))]);
                    if idxMax == 2
                        idxAngles(j) = k;
                    end
                elseif k < m - 2
                    [maxAngles(j), idxMax] = max([maxAngles(j), ...
                        angleOld + angleNew + norm(v, 2) + abs(v(1))]);
                    if idxMax == 2
                        idxAngles(j) = k;
                    end
                elseif k == m - 2
                    [maxAngles(j), idxMax] = max([maxAngles(j), ...
                        angleNew + norm(u, 2) + abs(u(1))]);
                    if idxMax == 2
                        idxAngles(j) = k + 1;
                    end
                end
                angleOld = angleNew;
            end
        end
    end
end

end
