%% MORLAB Demo: Time Evaluation and Visualization
% This demo script contains the handling of the time evaluation and
% visualization functions (simulations).
%
% See also ml_ct_dss_simulate_ie, ml_ct_dss_simulate_ss11,
% ml_ct_dss_simulate_ss21, ml_ct_soss_simulate_ie,
% ml_ct_soss_simulate_ss22, ml_ct_ss_simulate_ie, ml_ct_ss_simulate_ss11,
% ml_ct_ss_simulate_ss21, ml_dt_dss_simulate, ml_dt_ss_simulate.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%% General Handling
% For each supported system type there are time simulation routines in
% MORLAB:
%
% * continuous-time standard systems: implicit Euler
%   |ml_ct_ss_simulate_ie|, SS11 |ml_ct_ss_simulate_ss11|, SS21
%   |ml_ct_ss_simulate_ss21|
% * continuous-time descriptor systems: implicit Euler
%   |ml_ct_dss_simulate_ie|, SS11 |ml_ct_dss_simulate_ss11|, SS21
%   |ml_ct_dss_simulate_ss21|
% * continous-time second-order systems: implicit Euler
%   |ml_ct_soss_simulate_ie|, SS22 |ml_ct_soss_simulate_ss22|
% * discrete-time standard systems: |ml_dt_ss_simulate|
% * discrete-time descriptor systems: |ml_dt_dss_simulate|
%
% For a better demonstration, we load a prepared datafile containing
% matrices of a standard contractive state-space system:

if exist('OCTAVE_VERSION', 'builtin')
    orig_warn = warning('off', 'Octave:data-file-in-path');
    load morlab_data_std_br.mat;
    warning(orig_warn);
else
    load morlab_data_std_br.mat;
end

%%
% For using the plot functions, we need to put the system matrices either
% into a struct or, if the System Control Toolbox (control package) is
% installed, an ss object. For simplicity we decide here for the struct
% version.

sys = struct( ...
    'A', A, ...
    'B', B, ...
    'C', C, ...
    'D', D);

%%
% We start with the most simplest way of using the visualization by just
% calling the implicit Euler function.

ml_ct_ss_simulate_ie(sys);

%%
% This produces by default a unit step response plot of the system over the
% time range [0, 10]. The time range, number of time points and further
% things can be changed by optional parameters. In general, the simulation
% functions can be used to simulate the system for arbitrary input signals.
% In the standard case, we have a system of the form
%
%   x'(t) = A*x(t) + B*u(t),
%    y(t) = C*x(t) + D*u(t).
%
% To change the input signal u(t) of the system, we need to give a function
% handle as optional parameter describing a function depending on time that
% is giving the appropriate number of input signals. We decide on the
% following signal

u = @(t) sin(t) * ones(3, 1) - 0.5;

%%
% Due to using subplots in case of several system outputs, generated plots
% cannot be overwritten easily. Therefore, we close the previous created
% plot.
close;

%%
% Now we can simulate the system with the new input signal over the time
% range [0, 30]:
opts = struct('InputFcn', u, 'TimeRange', 30);
ml_ct_ss_simulate_ie(sys, opts);

%%
% An important feature of the time evaluation functions are the output
% arguments. For the moment, we suppress the creation of the simulation
% plot and consider just the output arguments:

[t, y, info] = ml_sigmaplot(sys, struct('ShowPlot', 0));

%%
% We shortly explain the meaning of those outputs:
%
% * t contains the time points over which the system was simulated
% * y contains the outputs of the system generated by the simulated states,
% * info is a struct with the number of computed time points, the internal
%   states x(t) and possibel error values between given systems, if this
%   option was used.

%%
% Another sepcial feature to consider is the input range of the simulation.
% By this it is possible to determine a time range in which the input
% signal is applied and outside it is set to 0, e.g., we apply the new
% input signal only on the interval [5, 15] and simulate the system again
% until the time 30.

close;
opts.InputRange = [5 15];
ml_ct_ss_simulate_ie(sys, opts);

%%
% It shall be mentioned that the initial value of the internal states is by
% default 0 but can be changed also by the optional parameters.


%% Multiple Systems and Error Plots
% Another feature of the MORLAB time evaluation functions is the ability to
% accept multiple systems. The given systems need to be of the same type
% and have the same number of inputs and outputs.
% We construct a second system

sys2 = struct( ...
    'A', -1, ...
    'B', ones(1, 3), ...
    'C', ones(2, 1));

%%
% Now the function will create to graphs per plot, for each system one.

close;
ml_ct_ss_simulate_ie(sys, sys2);

%%
% If multiple systems are given, the evaluation functions support
% difference modi. Those allow the computation of absolute and relative
% error outputs based on the given systems.
% In this case, the first given system is taken as the primary one and all
% following systems will be substracted from the first one to create the
% results. Let's compute as an example the relative error plot between
% sys and sys2.

close;
opts = struct('DiffMode', 'rel');
ml_ct_ss_simulate_ie(sys, sys2, opts);

%%
% The magnitude values of the error systems are then given in the info
% output argument.

%%
% Multiple systems can be given either as demonstrated above or in an
% one-dimensional cell array, where each entry is a system.

close;
ml_ct_ss_simulate_ie({sys, sys2});


%% Remarks
%
% * All frequency evaluation functions are basically equivalent in use but
%   differ in the computed values and results. Therefore, all examples here
%   work for all the functions in the same way.
% * The functions support sparse system matrices by using structs as input
%   arguments.
% * The simulator functions in general store only the outputs of the system
%   in the argument y. By setting the optional parameter StoreStates, also
%   the internal states can be stored during the simulation and afterwards
%   be found in the info struct.
