%% MORLAB Demo: Model Reduction for Second-Order Systems
% This demo script shows the application of unstructured and
% structure-preserving model reduction methods second-order systems.
%
% See also ml_ct_brbt, ml_ct_bst, ml_ct_bt, ml_ct_flbt, ml_ct_hinfbt,
% ml_ct_tlbt, ml_ct_hna, ml_ct_krylov, ml_ct_lqgbt, ml_ct_mt, ml_ct_prbt
% ml_ct_tlbt, ml_ct_twostep_mor.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%% General Handling
% The MORLAB toolbox implements unstructured as well as
% structure-preserving model reduction methods for continuous-time
% second-order systems, i.e., for systems of the form
%
%  M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t),
%      y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t).
%
% In practice, those systems are described by a large number of
% differential equations, which makes the usage in optimization and
% controller design difficult.
% The goal is to construct an approximation given by either
%
%  Er*xr'(t) = Ar*xr(t) + Br*u(t),
%      yr(t) = Cr*xr(t) + Dr*u(t),
%
% or if the underlying system structure is preserved of the form
%
%  Mr*xr''(t) = -Kr*xr(t) -  Er*xr'(t) + Bur*u(t),
%       yr(t) = Cpr*xr(t) + Cvr*xr'(t) +  Dr*u(t),
%
% which are described by less differential equations than the original
% system. The new system has to approximate the input-output behavior of
% the original system, i.e., the same inputs u(t) the outputs are close in
% a certain sense ||y(t) - yr(t)|| < tol.


%%
% To get started with the model reduction in MORLAB, we load a prepared
% data file containing a general stable second-order system.

if exist('OCTAVE_VERSION', 'builtin')
    orig_warn = warning('off', 'Octave:data-file-in-path');
    load morlab_data_so_stab.mat;
    warning(orig_warn);
else
    load morlab_data_so_stab.mat;
end

%%
% For the application of the model reduction methods, we just need to give
% the system to the model reduction function we want to use. This can be
% done in different ways. First, MORLAB supports the single matrices given
% to the function. As example we want to apply the balanced truncation
% method to the system matrices. Since the default behavior of MORLAB
% routines is non-structure-preserving, we expect the output model to have
% first-order form.

[Ar, Br, Cr, Dr, Er, info] = ml_ct_bt(M, E, K, Bu, Cp, Cv, D);

%%
% The output matrices Ar, Br, Cr, Dr, Er are now the system matrices of the
% approximation. The info struct contains general information about the
% underlying used methods.

disp(info);

%%
% A more advanced way of using the model reduction function is to construct
% first the system as a struct. This is recommended, since most of the
% evaluation and analysis tools in MORLAB assume the system to be in a
% struct.

sys = struct( ...
    'M' , M, ...
    'E' , E, ...
    'K' , K, ...
    'Bu', Bu, ...
    'Cp', Cp, ...
    'Cv', Cv, ...
    'D' , D);

%%
% Now we want to apply the balanced truncation method to this system again
% but in structure-preserving form such that the resulting reduced-order
% model is also in second-order form. Therefore, we need to give the option
% OutputModel = 'so' to the model reduction function.

rom = ml_ct_bt(sys, struct('OutputModel', 'so'));

%%
% The reduced-order model is now given in the rom struct with the same
% naming of the matrices as in sys.

disp(rom);

%%
% To take a closer look on the result, we can use the analysis tools of
% MORLAB, e.g., let's compute the transfer functions of the original system
% and the reduction in the frequency domain by the sigma plot function.
% Note that it is not necessary to compute the first-order realization of
% the system, since the methods in MORLAB can handle the second-order
% structure.

ml_sigmaplot(sys, rom);

%%
% Or we can compare the two systems in a time simulation.

ml_ct_soss_simulate_ss22(sys, rom);

%%
% All model reduction methods allow customization by optional parameters.
% See also the |morlab_demo_morlabopts| for a starting guide with the
% option contructor. Here we will just call the function for simplicity and
% adjust the parameters as we want to.

opts = ml_morlabopts('ml_ct_bt');

%%
% Now opts is an empty option struct with all the fields that the balanced
% truncation routine accepts. Note that MORLAB functions doesn't need the
% full option struct in general, non-existent or empty fields are replaced
% by default values.

disp(opts);

%%
% As a simple start, we want to compute a reduced-order model of order 5.
% Therefor, we are arranging the opts struct in the following way.

opts.Order            = 3;
opts.OrderComputation = 'Order';
opts.OutputModel      = 'so';

%%
% Now we can start the model reduction method again but with opts as
% additional input.

rom = ml_ct_bt(sys, opts);

disp(rom);

%%
% And we see that rom now contains the matrices of a second-order system
% of order 3.

%% Remarks
%
% * All model reduction routines are more or less equivalent to each
%   other, which means that all usage examples here also apply to the other
%   routines.
% * Not all model reduction methods are structure-preserving but all
%   continuous-time methods accept second-order systems as input.
% * The shown functionality also applies to sparse second-order systems.
