%% MORLAB Demo: Frequency Evaluation and Visualization
% This demo script contains the handling of the frequency evaluation and
% visualization functions |ml_bodemag|, |ml_frobeniusplot| and
% |ml_sigmaplot|.
%
% See also ml_bodemag, ml_frobeniusplot, ml_sigmaplot.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%


%% General Handling
% MORLAB provides three different functions for frequency evaluation:
%
% * |ml_sigmaplot| plotting the largest singular value of the transfer
%   function over the imaginary axis
% * |ml_frobeniusplot| plotting the Frobenius norm of the transfer
%   function over the imaginary axis
% * |ml_bodemag| plotting the absolute values of the single transfer
%   function entries over the imaginary axis
%
% For a better demonstration, we load a prepared datafile containing
% matrices of a standard unstable state-space system:

if exist('OCTAVE_VERSION', 'builtin')
    orig_warn = warning('off', 'Octave:data-file-in-path');
    load morlab_data_std_unstab.mat;
    warning(orig_warn);
else
    load morlab_data_std_unstab.mat;
end

%%
% For using the plot functions, we need to put the system matrices either
% into a struct or, if the System Control Toolbox (control package) is
% installed, an ss object. For simplicity we decide here for the struct
% version.

sys = struct( ...
    'A', A, ...
    'B', B, ...
    'C', C, ...
    'D', D);

%%
% We start with the most simplest way of using the visualization by just
% calling the sigma plot function.

ml_sigmaplot(sys);

%%
% This produces a sigma plot of the function in the default frequency range
% of [1e-4, 1e+4] rad/sec. This range as well as the units of frequency and
% magnitude scale can be changed by optional parameters given in an option
% struct. For example, we want to have Hz as frequency unit and a linear
% scale for the magnitude. We creat the option struct

opts = struct( ...
    'FreqUnit', 'Hz', ...
    'MagScale', 'linear');

%%
% Now we call again the plot function but with the changed parameters:

ml_sigmaplot(sys, opts);

%%
% An important feature of the frequency evaluation functions are the output
% arguments. For the moment, we suppress the creation of the frequency plot
% and consider just the output arguments:

[w, mag, tf, info] = ml_sigmaplot(sys, struct('ShowPlot', 0));

%%
% We shortly explain the meaning of those outputs:
%
% * w contains the frequency points in rad/sec or Hz where the transfer
%   function was evaluated,
% * mag contains the computed magnitudes, the results differ for the
%   different plot functions,
% * tf contains the computed transfer function evaluations,
% * info is a struct with the number of computed frequency points and
%   possibel error values between given systems, if this option was used.


%% Ways of Frequency Sampling
% An important feature of the frequency evaluation functions is the way of
% determining the points for sampling. Here MORLAB supports different ways
% of sampling.
% First, the default way of sampling is the adaptive one. This can also be
% activated by setting the parameter FreqSample

opts = struct('FreqSample', 'adapt');

%%
% By default, the adaptive mode uses 3 initial equidistant points and
% refines the grid then adaptively to reach a certain tolerance given by
% the parameter AccuracyTol. To make the adaptive sampling visible, we
% change the line style of the plot to points by

opts.LineSpecs = '.';
ml_sigmaplot(sys, opts);

%%
% One can see that in regions with more system behavior and difficult
% curves, the method is automatically placing more points. The initial
% points can be controlled by setting the parameter InitPoints to either an
% integer for the number of equidistant initial points or a vector of
% initial frequencies.

%%
% To change the sampling to an equidistant one set

opts.FreqSample = 'equal';

%%
% Now the parameter MaxPoints, which sets an upper bound for the number of
% frequency points in the adaptive setting, is used as exact number of
% equidistant points.

opts.MaxPoints = 200;
ml_sigmaplot(sys, opts);

%%
% A third way of sampling is just giving a vector of frequency points to
% the FreqSample parameter:

opts.FreqSample = logspace(-2, 2, 50);
ml_sigmaplot(sys, opts);


%% Multiple Systems and Error Plots
% Another feature of the MORLAB frequency evaluation functions is the
% ability to accept multiple systems of different types. The limitations of
% the systems is the restriction of having the same inputs and outputs.
% We construct a second system

sys2 = struct( ...
    'A', -1, ...
    'B', ones(1, 3), ...
    'C', ones(2, 1));

%%
% Now the function will create to graphs in the plot, for each system one.

ml_sigmaplot(sys, sys2);

%%
% If multiple systems are given, the evaluation functions support
% difference modi. Those allow the computation of absolute and relative
% error transfer functions based on the given systems.
% In this case, the first given system is taken as the primary one and all
% following systems will be substracted from the first one to create the
% results. Let's compute as an example the relative error plot between
% sys and sys2.

opts = struct('DiffMode', 'rel');
ml_sigmaplot(sys, sys2, opts);

%%
% The magnitude values of the error systems are then given in the info
% output argument.

%%
% Multiple systems can be given either as demonstrated above or in an
% one-dimensional cell array, where each entry is a system.

ml_sigmaplot({sys, sys2});


%% Remarks
%
% * All frequency evaluation functions are basically equivalent in use but
%   differ in the computed values and results. Therefore, all examples here
%   work for all the functions in the same way.
% * The functions support sparse system matrices by using structs as input
%   arguments.
% * The functions support all implemented system types of MORLAB. The
%   underlying system type is automatically determined by the matrices
%   in the ss object or the struct.
