% Data script for the construction of a stable second-order system.
%
% DESCRIPTION:
%   This data script generates the matrices of a stable second-order LTI
%   system of the form
%
%	M*x''(t) = -K*x(t) -  E*x'(t) + Bu*u(t),
% 	    y(t) = Cp*x(t) + Cv*x'(t) +  D*u(t),
%
%   with M, E, K as (n x n) matrices, Bu an (n x m) matrix, Cp, Cv as
%   (p x n) matrix, and D a (p x m) matrix.
%
%   The construction has the following dimensions:
%       n = 100,
%       m = 3,
%       p = 2,
%   and the matrices M, E, K is constructed such that all eigenvalues of
%   the quadratic pencil lie in the open left half-plane.
%   In the datainfo struct the following numbers are saved:
%       - n, number of states.
%
% NOTES:
%   * The data, generated under Matlab 9.1.0.441655 (R2016b) GLNXA64, is
%     saved in the same-named .MAT file.
%   * This script cannot be used under Octave.
%
% See also morlab_data_std_unstab.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% System dimensions.
n = 100;
m = 3;
p = 2;

% Random seed.
s = 2.0;

% System generation.
rng(s);
M  = randn(n);
E  = randn(n);
K  = randn(n);
Bu = rand(n, m);
Cp = rand(p, n);
Cv = rand(p, n);
D  = rand(p, m);

% Shift of the matrices to get the desired eigenvalue structure.
E = E + 9.6 * M;
K = K + 4.8 * E + 23 * M;

% Info about the generated data structure.
datainfo = struct('n', n);

% Clear unwanted variables.
clear n m p s;
