function opts = ml_check_cell_vector(opts, name, numSys, orders, default)
%ML_CHECK_CELL_VECTOR Checks input vectors and sets default values.
%
% SYNTAX:
%   opts = ML_CHECK_CELL_VECTOR(opts, name, numSys, orders, default)
%
% DESCRIPTION:
%   Checks for the field name of a struct to be a vector or one dimensional
%   cell array of vectors and checks those vectors to fit with the
%   dimensions in orders. If there are less vectors than numSys, the rest
%   is filled up by using the default function.
%
% INPUTS:
%   opts    - struct, having its field checked
%   name    - character array, name of the field to be checked
%   numSys  - positive integer, number of underlying systems
%   orders  - cell array, containing the orders of the underlying systems
%   default - function handle, accepting two input arguments, both
%             nonnegative integers
%
% OUTPUTS:
%   opts    - structure, copy of the input with the field set and put into
%             a cell array
%
% See also ml_check_cell_param.

%
% This file is part of the MORLAB toolbox
% (https://www.mpi-magdeburg.mpg.de/projects/morlab).
% Copyright (C) 2006-2023 Peter Benner, Jens Saak, and Steffen W. R. Werner
% All rights reserved.
% License: BSD 2-Clause License (see COPYING)
%

% Check the inputs.
narginchk(5, 5);

assert(isa(opts, 'struct'), ...
    'MORLAB:data', ...
    'The opts input needs to be a struct!');

assert(isa(name, 'char'), ...
    'MORLAB:data', ...
    'The name input needs to be a character array!');

ml_assert_posinteger(numSys);

assert(isa(orders, 'cell') ...
    && isvector(orders) ...
    && (length(orders) == numSys), ...
    'MORLAB:data', ...
    ['The orders needs to be given in a one dimensional cell ' ...
    'array of length %d!'], numSys);

assert(isa(default, 'function_handle'), ...
    'MORLAB:data', ...
    'The default value needs to be a function handle!');

% Perform struct checks and compute default values.
if ml_field_set_to_value(opts, name)
    tmp         = opts.(name);
    opts.(name) = cell(1, numSys);

    if isa(tmp, 'cell')
        assert(isvector(tmp), ...
            'MORLAB:data', ...
            'The initial states must be given in a cell vector!');

        lVec = length(tmp);
    else
        tmp = {tmp};
        lVec = 1;
    end

    for k = 1:numSys
        if k <= lVec
            ml_assert_vector(tmp{k}, 'Xp0');

            assert(isequal(size(tmp{k}), [orders{k}, 1]), ...
                'MORLAB:data', ...
                'The parameter opts.%s must be a vector of length %d!', ...
                name, orders{k});

            opts.(name){k} = tmp{k};
        else
            opts.(name){k} = default(orders{k}, 1);
        end
    end
else
    opts.(name) = cell(1, numSys);
    for k = 1:numSys
        opts.(name){k} = default(orders{k}, 1);
    end
end
