# Amphipod husbandry optimization experiments
# Ver. 1.2
# 2022-09-19
# Authors: Roman Alther (roman.alther@eawag.ch), Eawag, Duebendorf, Switzerland & Andrin Kraehenbuehl (andrin_kraehenbuehl94@hotmail.ch)
# Works in R ver. 4.1.1 (Windows 10, version 21H1) and 4.1.3 (macOS 11.6), and 4.2.0 (Ubuntu 22.04)
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

# 1) Preparation                                                           ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

## 1.1) Reset R, clean environment, set WD                                 ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
rm(list=ls()) # start from scratch
if(!is.null(dev.list())){dev.off()} # close any open graphics device
opar <- par() # save current plotting parameters for resetting if needed
# set working directory
WD <- getwd()
setwd("..")

## 1.2) Load necessary packages                                            ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# check if packages httr and sf are already installed
ulib=NULL # define user library
toload <- c("survival", # version 3.2-13 works
            "survminer", # version 0.4.9 works
            # "ggplot2", # version 3.3.5 works
            # "tidyverse", # version 1.3.1 works
            "vioplot") # version 0.3.7 works
toinstall <- setdiff(toload, rownames(installed.packages()))
if (length(toinstall) > 0) {
  cat(paste0("The ",ifelse(length(toinstall)==1,"dependency '","dependencies '"),paste(setdiff(toinstall, rownames(installed.packages())),collapse="', '"),"' will be installed.\n"))
  install.packages(toinstall, lib=ulib)
}
# invisible(lapply(toload, library, character.only = T)) # load packages

## 1.3) Helper function                                                    ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# Function to add transparency to HEX color codes
#https://gist.github.com/lopspower/03fb1cc0ac9f32ef38f4
hex_alpha <- function(hex, alpha, format="RRGGBB"){
  trns <- data.frame(trns=seq(100,0,-1),hex=c("FF","FC","FA","F7","F5","F2","F0","ED","EB","E8","E6","E3","E0","DE","DB","D9","D6","D4","D1","CF","CC","C9","C7","C4","C2","BF","BD","BA","B8","B5","B3","B0","AD","AB","A8","A6","A3","A1","9E","9C","99","96","94","91","8F","8C","8A","87","85","82","80","7D","7A","78","75","73","70","6E","6B","69","66","63","61","5E","5C","59","57","54","52","4F","4D","4A","47","45","42","40","3D","3B","38","36","33","30","2E","2B","29","26","24","21","1F","1C","1A","17","14","12","0F","0D","0A","08","05","03","00")) #https://gist.github.com/lopspower/03fb1cc0ac9f32ef38f4
  if(format=="RRGGBB"){
    return(paste0(hex,trns$hex[match(alpha,trns$trns)]))
  }else{
    return(paste0("#",trns$hex[match(alpha,trns$trns)],gsub("\\#","",hex)))
  }
}

## 1.4) Set figure parameters                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# options for shelter & protein experiment
lwd_mean <- 2
lwd_ci <- 1
alpha_surv <- 20
# col_pal <- colorRampPalette(c("steelblue","white","red"))(4)
# col_pal <- MetBrewer::met.brewer("Isfahan1", n=4) # MetBrewer::colorblind_palettes
col_pal <- c("steelblue","lightblue","orange","red")
col_pal <- rgb(col2rgb(col_pal, F)[1,], col2rgb(col_pal, F)[2,], col2rgb(col_pal, F)[3,], maxColorValue = 255) # if colors are given with name

# options for daylength experiment
col_und <- "gray"
col_fix <- col_pal[1]
col_var <- col_pal[4]
col_adu <- col_pal[3]
col_juv <- col_pal[2]
col_alg <- c("white","lightgreen","darkgreen") # colors for algae data

# Labels for daylength experiment
lab_mon <- "Time [months]"
lab_day <- "Time [d]"
lab_abun <- "Total abundance"
lab_repr <- "Reproductive activity"
lab_repr_rel <- "Relative reproductive activity"
lab_fix <- "Fixed day length"
lab_var <- "Variable day length"
lab_daytreat <- c(lab_fix,lab_var)
lab_treat <- "Treatment"
lab_cond <- c("Control","Shelter","Feed enrichment","Feed enrichment plus shelter")
lab_alg <- c("Low algal growth",
             "Medium algal growth",
             "High algal growth")
lab_daylength <- "Day length [h]"
lab_surv <- "Survival rate"
lab_juv <- "Proportion of juveniles"

## 1.5) Experimental settings                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# shelter & protein experiment
n_treat=4 # four different treatments
n_rep=6 # six replicates per treatment
n_tanks=n_treat*n_rep # twentyfour tanks in total
n_ind=20 # twenty individuals per tank

## 1.6) Create output folders                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
dir.create(paste0(WD,"/Results"), showWarnings=F)
dir.create(paste0(WD,"/Results/Supplement"), showWarnings=F)

## ___________________________####
# 2) Load and prepare data                                                 ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 2.1) Data shelter & protein experiment                                  ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

#### 2.1.1) Read data                                                      ####
# read corrected adult count data set
data_KM<-read.table(paste0(WD,"/Nutrition_Exp_KaplanMeier_v1_input.txt"),header=T,sep="\t") # data from shelter&protein experiment for Kaplan Meier survival analysis
data_KM$CountingDate <- as.Date(data_KM$CountingDate, format="%d.%m.%Y") # transform counting date to date format

#### 2.1.2) Checks                                                         ####
if(length(unique(data_KM$Tank.ID))!=n_tanks){stop("Something went wrong prior to code line 113. Try 'traceback()'.")} # 24 tanks?
if(!all(data_KM$No.of.ind.Start==n_ind)){stop("Something went wrong prior to code line 114. Try 'traceback()'.")} # 20 individuals?
if(!all(c(min(data_KM$Tank.ID):max(data_KM$Tank.ID))==unique(data_KM$Tank.ID))){stop("Something went wrong prior to code line 115. Try 'traceback()'.")} # are the Tank IDs consecutive?

#### 2.1.3) Add days between countings                                     ####
data_KM$Days <- data_KM$CountingDate-min(data_KM$CountingDate) # what is the time difference between countings?
data_KM$Days[data_KM$Days==42] <- 41 # adjust one counting date that was one day later
counting_days <- sort(unique(data_KM$Days))

#### 2.1.4) Reshape data for survival analysis                             ####
# create empty vector for survival time
d<-vector()
# fill with survival times (days) for each tank(TankID 44:67) and time point(1:9)
for(i in min(data_KM$Tank.ID):max(data_KM$Tank.ID)){
  for(c in min(data_KM$Count.No):max(data_KM$Count.No)){
    if(c<max(data_KM$Count.No)){ # add the number of days as many times as there were deceased in that time (if three individuals died prior to first counting, add three times a zero to the vector)
      d<-c(d,rep(counting_days[c+1],data_KM$Adults[data_KM$Count.No==c&data_KM$Tank.ID==i]-data_KM$Adults[data_KM$Count.No==(c+1)&data_KM$Tank.ID==i]))
    }else{
      d<-c(d,rep(counting_days[c+1],data_KM$Adults[data_KM$Count.No==c&data_KM$Tank.ID==i]))
    }
  }
}

#check if everything worked properly
if(length(d)!=n_tanks*n_ind){stop("Something went wrong prior to code line 137. Try 'traceback()'.")} # lenght of vector should be 24 tanks * 20 individuals per tank observed = 480 individual survival times

# create empty vector for information on censored observations
e<-vector()
# fill with survival times (days) for each tank(TankID 44:67) and time point(1:9)
for(i in min(data_KM$Tank.ID):max(data_KM$Tank.ID)){
  for(c in min(data_KM$Count.No):max(data_KM$Count.No)){
    if(c<max(data_KM$Count.No)){ # add the number of days as many times as there were deceased in that time (if three individuals died prior to first counting, add three times a zero to the vector)
      e<-c(e,rep(1,data_KM$Adults[data_KM$Count.No==c&data_KM$Tank.ID==i]-data_KM$Adults[data_KM$Count.No==(c+1)&data_KM$Tank.ID==i]))
    }else{
      e<-c(e,rep(0,data_KM$Adults[data_KM$Count.No==c&data_KM$Tank.ID==i]))
    }
  }
}

#check if everything worked properly
if(length(e)!=n_tanks*n_ind){stop("Something went wrong prior to code line 153. Try 'traceback()'.")} # length of vector should be 480 again

# create vector rx with information on treatment
rx <- vector()
rx <- c(rep("Control",n_rep*n_ind),rep("Protein",n_rep*n_ind),rep("Shelter&Protein",n_rep*n_ind),rep("Shelter",n_rep*n_ind))

# create data frame with data vectors
df <- data.frame(d, e, rx)

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 2.2) Data daylength experiment                                          ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

#### 2.2.1) Read data                                                      ####
data <- read.table(paste0(WD,"/LabdataAK_v6_210616_Daylength_input.txt"), sep = "\t", header = T)
# remove NAs
if(any(is.na(data$Tank.ID))){
  data <- data[-which(is.na(data$Tank.ID)),]
}
data <- Filter(function(x)!all(is.na(x)), data) # https://stackoverflow.com/a/12614723/6380381 by https://stackoverflow.com/users/1385941/mnel
data$CountingDate <- as.Date(data$CountingDate, format="%d.%m.%Y") # transform counting date to date format
data$CountingDate[data$CountingDate=="0202-01-06"] <- "2021-01-06" # correct wrong date

#### 2.2.2) Experimental parameters                                        ####
nrep <- length(unique(data$Tank.ID)) # number of replicates
ntim <- length(which(data$Tank.ID==data$Tank.ID[1])) # number of measurements per replicate
nrep_fixed <- length(unique(data$Tank.ID[data$Treatment=="14:10 L:D ratio"])) # number of replicates per treatment
nrep_vary <- length(unique(data$Tank.ID[data$Treatment=="varying L:D ratio"]))

#### 2.2.3) Checks                                                         ####
if(max(data$Count.No[data$Tank.ID==105& !is.na(data$CountDay)])!=ntim-1){stop("Something went wrong prior to code line 183. Try 'traceback()'.")}
if(!all(data$Adults+data$Juveniles==data$Total)){stop("Something went wrong prior to code line 184. Try 'traceback()'.")}

#### 2.2.4) Add days between countings                                     ####
data$Days <- data$CountingDate-min(data$CountingDate) # what is the time difference between countings?
# data$Days[data$Days==342] <- xx # adjust one counting date that was six days late

#### 2.2.5) Prepare count matrices                                         ####
# Total count matrix
DaylengthTotal<- matrix(nrow = nrep, ncol = ntim)
for (i in 1:nrep){
  for (j in 1:ntim){
    DaylengthTotal[i+(ntim-ntim-1+j)*nrep]<- data$Total[data$Count.No==as.character(j-1)][i]
  }
}

# Adult count matrix
DaylengthAdult<- matrix(nrow = nrep, ncol = ntim)
for (i in 1:nrep){
  for (j in 1:ntim){
    DaylengthAdult[i+(ntim-ntim-1+j)*nrep]<- data$Adult[data$Count.No==as.character(j-1)][i]
  }
}

# Juvenile count matrix
DaylengthJuvenile<- matrix(nrow = nrep, ncol = ntim)
for (i in 1:nrep){
  for (j in 1:ntim){
    DaylengthJuvenile[i+(ntim-ntim-1+j)*nrep]<- data$Juvenile[data$Count.No==as.character(j-1)][i]
  }
}

# Plot of data for supplement
# png(paste0(WD,"/Results/Supplement/FigA10_alt.png"), width=6, height=5, units="in", res=300)
# plot(DaylengthTotal[1,], xlim=c(1,13.6), ylim=c(0,450), type="l", xlab=lab_mon, ylab=lab_abun, col=col_und, xaxt="n")
# axis(1, at=1:13, labels=0:12)
# for (i in 1:nrow(DaylengthTotal)-1){
#   lines(DaylengthTotal[i+1,], lwd=2, col=col_und)
# }
# for (i in 1:nrow(DaylengthAdult)){
#   lines(DaylengthAdult[i,], lwd=2, col=col_adu)
# }
# for (i in 1:nrow(DaylengthJuvenile)){
#   lines(DaylengthJuvenile[i,], lwd=2, col=col_juv)
# }
# legend("topright", c("Total","Adults","Juveniles"), lty=1, lwd=2, col=c(col_und,col_adu,col_juv), bty="n")
# dev.off()

#### 2.2.6) Prepare treatment count matrices                               ####
# make separate matrices for treatments and adults/juveniles
# Adult count matrix fixed
DaylengthFixedAdults<- matrix(nrow=nrep_fixed, ncol = ntim)
for (i in 1:nrep_fixed){
  for (j in 1:ntim){
    DaylengthFixedAdults[i+(ntim-ntim-1+j)*nrep_fixed]<- data$Adult[data$Treatment=="14:10 L:D ratio"& data$Count.No==as.character(j-1)][i]
  }
}

# Adult count matrix variable
DaylengthVariableAdults<- matrix(nrow=nrep_vary, ncol = ntim)
for (i in 1:nrep_vary){
  for (j in 1:ntim){
    DaylengthVariableAdults[i+(ntim-ntim-1+j)*nrep_vary]<- data$Adult[data$Treatment=="varying L:D ratio"& data$Count.No==as.character(j-1)][i]
  }
}

# Juvenile count matrix fixed
DaylengthVariableJuveniles<- matrix(nrow=nrep_vary, ncol = ntim)
for (i in 1:nrep_vary){
  for (j in 1:ntim){
    DaylengthVariableJuveniles[i+(ntim-ntim-1+j)*nrep_vary]<- data$Juvenile[data$Treatment=="varying L:D ratio"& data$Count.No==as.character(j-1)][i]
  }
}

# Juvenile count matrix variable
DaylengthFixedJuveniles<- matrix(nrow=nrep_fixed, ncol = ntim)
for (i in 1:nrep_fixed){
  for (j in 1:ntim){
    DaylengthFixedJuveniles[i+(ntim-ntim-1+j)*nrep_fixed]<- data$Juvenile[data$Treatment=="14:10 L:D ratio"& data$Count.No==as.character(j-1)][i]
  }
}

# Plot of data for supplement
# png(paste0(WD,"/Results/Supplement/FigA10_alt2.png"), width=12, height=5, units="in", res=300)
# par(mfrow=c(1,2)) # two plots per plotting window
# # plot for juveniles
# plot(DaylengthFixedJuveniles[1,], xlim=c(1,13.6), ylim=c(0,420), type="l", ylab=lab_abun, xlab=lab_mon, col=col_fix, xaxt="n")
# axis(1, at=1:13, labels=0:12)
# for (i in 1:nrow(DaylengthFixedJuveniles)-1){
#   lines(DaylengthFixedJuveniles[i+1,], lwd=2, col=col_fix)
# }
# for (i in 1:nrow(DaylengthVariableJuveniles)){
#   lines(DaylengthVariableJuveniles[i,], lwd=2, col=col_var)
# }
# title("Juveniles")
# # plot for adults
# plot(DaylengthFixedAdults[1,], xlim=c(1,13.6), ylim=c(0,420), type="l", ylab=lab_abun, xlab=lab_mon, col=col_fix, xaxt="n")
# axis(1, at=1:13, labels=0:12)
# for (i in 1:nrow(DaylengthFixedAdults)-1){
#   lines(DaylengthFixedAdults[i+1,], lwd=2, col=col_fix)
# }
# for (i in 1:nrow(DaylengthVariableAdults)){
#   lines(DaylengthVariableAdults[i,], lwd=2, col=col_var)
# }
# legend("topright", c("Fixed","Variable"), title="Daylength", lty=1, lwd=2, col=c(col_fix,col_var), bty="n")
# title("Adults")
# dev.off()

###### 2.2.6.1) Supp. Fig A10                                                ####
# Plot of data for supplement
scaling=1.6
same_yaxt=F
ylab_right=F
png(paste0(WD,"/Results/Supplement/FigA10.png"), width=12, height=10, units="in", res=300)
par(mfrow=c(2,2), # two plots per plotting window
    mar=c(4,4,2,0)+0.1)
# Juveniles, fixed
plot(DaylengthFixedJuveniles[1,], xlim=c(1,13.6), ylim=c(0,420), type="b", pch=19, ylab=lab_abun, xlab=lab_mon, col=col_fix, xaxt="n", main="Juveniles", cex.axis=scaling, cex.lab=scaling, cex.main=1.2*scaling)
axis(1, at=1:13, labels=0:12, cex.axis=scaling)
for (i in 1:nrow(DaylengthFixedJuveniles)-1){
  lines(DaylengthFixedJuveniles[i+1,], lwd=2, type="b", pch=19, col=col_fix)
}
# Adults, fixed
plot(DaylengthFixedAdults[1,], xlim=c(1,13.6), ylim=c(0,ifelse(same_yaxt,420,70)), type="b", pch=19, ylab=ifelse(ylab_right,lab_abun,""), xlab=lab_mon, col=col_fix, xaxt="n", main="Adults", cex.axis=scaling, cex.lab=scaling, cex.main=1.2*scaling)
axis(1, at=1:13, labels=0:12, cex.axis=scaling)
for (i in 1:nrow(DaylengthFixedAdults)-1){
  lines(DaylengthFixedAdults[i+1,], lwd=2, type="b", pch=19, col=col_fix)
}
legend("topright",lab_daytreat, lty=1, lwd=2, col=c(col_fix,col_var),box.lty=0, inset=0.005, cex=1.3*scaling)
# Juveniles, variable
plot(DaylengthVariableJuveniles[1,], xlim=c(1,13.6), ylim=c(0,420), type="b", pch=19, ylab=lab_abun, xlab=lab_mon, col=col_var, xaxt="n", cex.axis=scaling, cex.lab=scaling)
axis(1, at=1:13, labels=0:12, cex.axis=scaling)
for (i in 1:nrow(DaylengthVariableJuveniles)-1){
  lines(DaylengthVariableJuveniles[i+1,], lwd=2, type="b", pch=19, col=col_var)
}
# Adults, variable
plot(DaylengthVariableAdults[1,], xlim=c(1,13.6), ylim=c(0,ifelse(same_yaxt,420,70)), type="b", pch=19, ylab=ifelse(ylab_right,lab_abun,""), xlab=lab_mon, col=col_var, xaxt="n", cex.axis=scaling, cex.lab=scaling)
axis(1, at=1:13, labels=0:12, cex.axis=scaling)
for (i in 1:nrow(DaylengthVariableAdults)-1){
  lines(DaylengthVariableAdults[i+1,], lwd=2, type="b", pch=19, col=col_var)
}
dev.off()

#### 2.2.7) Prepare reproductive activity data                             ####
# Needed: vectors for time (in days) and reproductive activity (no of events, gravid females & praecopula pairs)

# Reproductive activity is the sum of gravid females and praecopula pairs
repr.act<-data$PF+data$PC
repr.act.rel <- repr.act/data$Adults # relative reproductive activity is the absolute value divided by the number of adult individuals

# Transform the data to a matrix, where each row is a replicate and each column is the number of measurements per replicate
repr.act.mat<-matrix(repr.act, nrow=nrep, ncol=ntim,byrow = T)
repr.act.rel.mat <- matrix(repr.act.rel, nrow=nrep, ncol=ntim,byrow = T)

# Separate matrices of the above matrix (absolute)
repr.act.pc<-matrix(data$PC, nrow=nrep, ncol=ntim,byrow = T) # reproductive activity precopulating pairs
repr.act.gf<-matrix(data$PF, nrow=nrep, ncol=ntim,byrow = T) # reproductive activity gravid females

#### 2.2.8) Time vector                                                    ####
# Create a vector with the 'planned' counting days
Time<-rep(1, ntim)
for (i in 1: ntim){
  Time[i]<-((length(Time[1:i])-1)*28)
  Time[1]<-1
}
counting_days_dl <- sort(unique(data$Days)) # not all countings could be done in one day, hence there is some deviation

#### 2.2.9) Daylength table                                                ####
# Create vectors with the daylengths (both, fixed and variable)
Daylengthvariable<-rep(c(14,12,10,10,12,14,16,14,12,10,10,12,14,16),100)     # the year has 52 weeks, by using 4 week intervals i receive start+13=14 data steps...
Daylengthvariable<-Daylengthvariable[1:ntim] # Daylength just indicated monthly
Daylengthstable<-rep(14,ntim) # Daylength just indicated monthly

# Make it a table
Daylengths<-matrix(ncol=ntim, nrow=nrep)
rownames(Daylengths)<-c(rep("fix",(nrep/2)), rep("var",(nrep/2)))
Daylengths[1:(nrep/2),1:ntim]<-Daylengthstable
Daylengths[(nrep/2+1):nrep,1:ntim]<-Daylengthvariable

#### 2.2.10) Age structure                                                 ####
# Agestructure as the percentage of juveniles
agestructure <- data$Juveniles/data$Total  # juveniles as percentage of total abundance
agestructure <- matrix(agestructure, nrow=nrep, ncol=ntim, byrow = T) # turn it into matrix

#### 2.2.11) Algae                                                         ####
algae<-matrix(data$Algae, nrow=nrep, ncol=ntim,byrow = T)

#### 2.2.12) Overall data matrix                                           ####
data.matrix<-data.frame(abundance=as.vector(DaylengthTotal[,1:ntim]),
                        adult=as.vector(DaylengthAdult[,1:ntim]),
                        daylength=as.vector(Daylengths),
                        treatment=rep(rownames(Daylengths),ntim),
                        days=rep(Time, each=nrep),
                        repr.activity=as.vector(repr.act.mat),
                        repr.act.rel=as.vector(repr.act.rel.mat),
                        agestructure=as.vector(agestructure),
                        algae=as.vector(algae),
                        id=rep(105:116,ntim),
                        precopulation=as.vector(repr.act.pc),
                        gravid_females=as.vector(repr.act.gf))
data.matrix[,1]<-as.numeric(data.matrix[,1])
data.matrix[,2]<-as.numeric(data.matrix[,2])
data.matrix[,5]<-as.numeric(data.matrix[,5])
data.matrix[,6]<-as.numeric(data.matrix[,6])

###### 2.2.12.1) Supp. Fig A9                                              ####
# relative reproductive activity (divided by #adults/time point/tank)
png(paste0(WD,"/Results/Supplement/FigA9.png"), width=6, height=5, units="in", res=300)
vioplot::vioplot(data.matrix$abundance~as.factor(data.matrix$treatment),
                 col = c(col_fix,col_var),
                 xlab=lab_treat,
                 names=lab_daytreat,
                 ylab=lab_abun,
                 # border = c(col_fix,col_var),
                 ylim=c(0,max(data.matrix$abundance, na.rm = T)))
# boxplot(data.matrix$abundance~as.factor(data.matrix$treatment),
#         col=c(col_fix,col_var),
#         xlab=lab_treat,
#         names=lab_daytreat,
#         ylab=lab_abun,
#         ylim=c(0,max(data.matrix$abundance, na.rm = T)))
dev.off()

## ___________________________####
# 3) Analysis                                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 3.1) Analysis shelter & protein experiment                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# data about tanks
dimension_tank_1 <- 268 # base dimension 1 of tank in mm
dimension_tank_2 <- 215 # base dimension 2 of tank in mm
base_area_tank <- ((dimension_tank_1*dimension_tank_2)/(1000^2)) # product of base dimensions divided by square-root of 1000 to trnasform mm to m
base_area_tank <- 0.037 # since they are smaller at base, take manufacturers data
20/base_area_tank # starting density

# create survival object
surv_object <- survival::Surv(time = d, event = e)
# fit
surv <- survival::survfit(surv_object ~ rx)
summary(surv)

# log-rank test
surv_diff <- survival::survdiff(survival::Surv(d, e) ~ rx, data = df)
surv_diff

# pairwise comparisons
survminer::pairwise_survdiff(survival::Surv(d, e) ~ rx, data = df)

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 3.2) Analysis daylength experiment                                      ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# data about trays
dimension_tray_1 <- 750 # base dimension 1 of tray in mm
dimension_tray_2 <- 445 # base dimension 2 of tray in mm
base_area_tray <- ((dimension_tray_1*dimension_tray_2)/(1000^2)) # product of base dimensions divided by square-root of 1000 to trnasform mm to m
((dimension_tray_1*dimension_tray_2)/(10^2)) # base area in square cm
60/base_area_tray # starting density

# normality test
shapiro.test(data.matrix$repr.act.rel) # p>0.05 means normal distribution
shapiro.test(data.matrix$abundance)
shapiro.test(data.matrix$algae)

# QQ plots
# qqnorm(data.matrix$repr.act.rel)
# qqnorm(data.matrix$abundance)

# data for treatments
mean(data.matrix$abundance[data.matrix$treatment=="fix"]) # mean total abundance for fixed day length
range(data.matrix$abundance[data.matrix$treatment=="fix"]) # range of total abundance for fixed day length
mean(data.matrix$abundance[data.matrix$treatment=="fix"])/base_area_tray # mean density fixed day length

mean(data.matrix$abundance[data.matrix$treatment=="var"]) # mean total abundance for variable day length
range(data.matrix$abundance[data.matrix$treatment=="var"]) # range of total abundance for variable day length
mean(data.matrix$abundance[data.matrix$treatment=="var"])/base_area_tray # mean density for variable day length

mean(data.matrix$repr.act.rel[data.matrix$treatment=="fix"], na.rm=T) # mean relative reproductive activity for fixed day length
mean(data.matrix$repr.act.rel[data.matrix$treatment=="var"], na.rm=T) # mean relative reproductive activity for variable day length
median(data.matrix$repr.act.rel[data.matrix$treatment=="fix"], na.rm=T) # mean relative reproductive activity for fixed day length
median(data.matrix$repr.act.rel[data.matrix$treatment=="var"], na.rm=T) # mean relative reproductive activity for variable day length

# comparison of treatments
# Wilcoxon test (unpaired two sample non-parametric test)
# Amphipod densities
wilcox.test(data.matrix$abundance~data.matrix$treatment, conf.int=T)
# Relative reproductive activity
wilcox.test(data.matrix$repr.act.rel~data.matrix$treatment, conf.int=T)
cor.test(data.matrix$repr.act.rel[!is.na(data.matrix$repr.act.rel)],
         data.matrix$abundance[!is.na(data.matrix$repr.act.rel)],
         alternative="greater",
         method = "kendall")

# # ANOVA approach
# aov_rra <- aov(data.matrix$repr.act.rel~data.matrix$treatment+factor(data.matrix$days))
# summary(aov_rra)
# TukeyHSD(aov_rra)
# 
# # GLM approach
# glm_rra <- glm(data.matrix$repr.act.rel~data.matrix$treatment+data.matrix$days)
# summary(glm_rra)
# 
# library(nlme)
# # Model assuming the same variance for each time point
# data.matrix.na <- data.matrix[!is.na(data.matrix$repr.act.rel),]
# gls.fit <- 
#   gls(repr.act.rel ~ factor(days) + treatment, 
#       data = data.matrix.na,
#       corr = corSymm(form = ~ 1 | id),
#       control = glsControl(tolerance = 0.01, msTol = 0.01, 
#                            maxIter = 1000000, msMaxIter = 1000000))
# summary(gls.fit)
# 
# # Model allowing for different variance structure for each time point
# gls.fit.diff.var <- 
#   gls(repr.act.rel ~ factor(days) + treatment, 
#       data = data.matrix.na,
#       corr = corSymm(form = ~ 1 | id),
#       weights = varIdent(form = ~ 1 | factor(days)),
#       control = glsControl(tolerance = 0.01, msTol = 0.01, 
#                            maxIter = 1000000, msMaxIter = 1000000, msVerbose=FALSE))
# summary(gls.fit.diff.var)

# Algae
wilcox.test(data.matrix$algae~data.matrix$treatment, conf.int=T) # algae comparison as numeric values
cor.test(data.matrix$algae[!is.na(data.matrix$algae)],
         data.matrix$abundance[!is.na(data.matrix$algae)],
         alternative="greater",
         method = "kendall")
cor.test(data.matrix$algae[!is.na(data.matrix$algae)],
         data.matrix$repr.act.rel[!is.na(data.matrix$algae)],
         alternative="greater",
         method = "kendall")
# GLM approach
glm_alg <- glm(data.matrix$algae~data.matrix$treatment+data.matrix$days)
summary(glm_alg)

# Juveniles
wilcox.test(data.matrix$agestructure~data.matrix$treatment, conf.int=T)
# GLM approach
glm_juv <- glm(data.matrix$agestructure~data.matrix$treatment+data.matrix$days)
summary(glm_juv)

# effect size estimate  # 0.1+ weak, 0.3+ intermediate, 0.5+ strong
# can to my knowledge not be calculated for wilcoxon rank test. Differences in locations for groups can be used to decide if differences matter or not.
# There is a lack in size measurements for non-parametric tests. (eg. Leech & Onwuegbuzie, 2002)

## ___________________________####
# 4) Figures                                                               ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 4.1) Fig 1                                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
#### 4.1.1) Prepare data for plotting with baseplot                        ####
# the resulting objects should be the same as values returned by summary(surv, time=counting_days)$surv
control <- matrix(NA,nrow=10,ncol=6)
control_id <- unique(data_KM$Tank.ID[data_KM$Treatment=="Control"])
for (i in 1:length(control_id)){
  control[,i] <- data_KM$Adults[data_KM$Tank.ID==control_id[i]]
}
control_surv <- apply(control,1,mean)/n_ind
control_lwr <- summary(surv, time=counting_days)$lower[summary(surv, time=counting_days)$strata=="rx=Control"]
control_upr <- summary(surv, time=counting_days)$upper[summary(surv, time=counting_days)$strata=="rx=Control"]

shelter <- matrix(NA,nrow=10,ncol=6)
shelter_id <- unique(data_KM$Tank.ID[data_KM$Treatment=="Shelter"])
for (i in 1:length(shelter_id)){
  shelter[,i] <- data_KM$Adults[data_KM$Tank.ID==shelter_id[i]]
}
shelter_surv <- apply(shelter,1,mean)/n_ind
shelter_lwr <- summary(surv, time=counting_days)$lower[summary(surv, time=counting_days)$strata=="rx=Shelter"]
shelter_upr <- summary(surv, time=counting_days)$upper[summary(surv, time=counting_days)$strata=="rx=Shelter"]

protein <- matrix(NA,nrow=10,ncol=6)
protein_id <- unique(data_KM$Tank.ID[data_KM$Treatment=="Protein"])
for (i in 1:length(protein_id)){
  protein[,i] <- data_KM$Adults[data_KM$Tank.ID==protein_id[i]]
}
protein_surv <- apply(protein,1,mean)/n_ind
protein_lwr <- summary(surv, time=counting_days)$lower[summary(surv, time=counting_days)$strata=="rx=Protein"]
protein_upr <- summary(surv, time=counting_days)$upper[summary(surv, time=counting_days)$strata=="rx=Protein"]

shelter_protein <- matrix(NA,nrow=10,ncol=6)
shelter_protein_id <- unique(data_KM$Tank.ID[data_KM$Treatment=="SP"])
for (i in 1:length(shelter_protein_id)){
  shelter_protein[,i] <- data_KM$Adults[data_KM$Tank.ID==shelter_protein_id[i]]
}
shelter_protein_surv <- apply(shelter_protein,1,mean)/n_ind
shelter_protein_lwr <- summary(surv, time=counting_days)$lower[summary(surv, time=counting_days)$strata=="rx=Shelter&Protein"]
shelter_protein_upr <- summary(surv, time=counting_days)$upper[summary(surv, time=counting_days)$strata=="rx=Shelter&Protein"]

# some numbers
survival_df <- data.frame(days=counting_days, mean=NA, min=NA, max=NA, sd=NA)
for (t in 1:10){
  survival_df[t,2] <- mean(c(shelter_protein[t,],protein[t,],shelter[t,],control[t,])/20)
  survival_df[t,c(3:4)] <- range(c(shelter_protein[t,],protein[t,],shelter[t,],control[t,])/20)
  survival_df[t,5] <- sd(c(shelter_protein[t,],protein[t,],shelter[t,],control[t,])/20)
}

#### 4.1.2) Using baseplot                                                 ####
png(paste0(WD,"/Results/Fig1.png"), width=6, height=5, units="in", res=300)
par(oma=c(0,0,0,0))
par(mar=c(4,4,0,0) + 0.1)
plot(counting_days,control_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xlab=lab_day,ylab=lab_surv)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(control_lwr[c(i,i)]), control_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[1], alpha=alpha_surv), border = NA)
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(shelter_lwr[c(i,i)]), shelter_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[2], alpha=alpha_surv), border = NA)
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(protein_lwr[c(i,i)]), protein_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[3], alpha=alpha_surv), border = NA)
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(shelter_protein_lwr[c(i,i)]), shelter_protein_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[4], alpha=alpha_surv), border = NA)
  # CI intervals as broken line
  segments(counting_days[i],control_upr[i],counting_days[i+1],control_upr[i], col=col_pal[1], lwd=lwd_ci, lty=2)
  segments(counting_days[i],control_lwr[i],counting_days[i+1],control_lwr[i], col=col_pal[1], lwd=lwd_ci, lty=2)
  segments(counting_days[i],shelter_upr[i],counting_days[i+1],shelter_upr[i], col=col_pal[2], lwd=lwd_ci, lty=2)
  segments(counting_days[i],shelter_lwr[i],counting_days[i+1],shelter_lwr[i], col=col_pal[2], lwd=lwd_ci, lty=2)
  segments(counting_days[i],protein_upr[i],counting_days[i+1],protein_upr[i], col=col_pal[3], lwd=lwd_ci, lty=2)
  segments(counting_days[i],protein_lwr[i],counting_days[i+1],protein_lwr[i], col=col_pal[3], lwd=lwd_ci, lty=2)
  segments(counting_days[i],shelter_protein_upr[i],counting_days[i+1],shelter_protein_upr[i], col=col_pal[4], lwd=lwd_ci, lty=2)
  segments(counting_days[i],shelter_protein_lwr[i],counting_days[i+1],shelter_protein_lwr[i], col=col_pal[4], lwd=lwd_ci, lty=2)
  # Fits as lines (staircase plot, drop first: unlike using option type='s')
  segments(counting_days[i],control_surv[i],counting_days[i],control_surv[i+1], col=col_pal[1], lwd=lwd_mean)
  segments(counting_days[i],control_surv[i+1],counting_days[i+1],control_surv[i+1], col=col_pal[1], lwd=lwd_mean)
  segments(counting_days[i],shelter_surv[i],counting_days[i],shelter_surv[i+1], col=col_pal[2], lwd=lwd_mean)
  segments(counting_days[i],shelter_surv[i+1],counting_days[i+1],shelter_surv[i+1], col=col_pal[2], lwd=lwd_mean)
  segments(counting_days[i],protein_surv[i],counting_days[i],protein_surv[i+1], col=col_pal[3], lwd=lwd_mean)
  segments(counting_days[i],protein_surv[i+1],counting_days[i+1],protein_surv[i+1], col=col_pal[3], lwd=lwd_mean)
  segments(counting_days[i],shelter_protein_surv[i],counting_days[i],shelter_protein_surv[i+1], col=col_pal[4], lwd=lwd_mean)
  segments(counting_days[i],shelter_protein_surv[i+1],counting_days[i+1],shelter_protein_surv[i+1], col=col_pal[4], lwd=lwd_mean)
  
}
legend("topright", lab_cond, lty=1, lwd=lwd_mean, col=col_pal, bty="n")
text(min(counting_days),0, survminer::surv_pvalue(surv, data=df)$pval.txt, adj=0)
dev.off()

###### 4.1.2.1) Fig 1 for graphical abstract                               ####
png(paste0(WD,"/Results/Supplement/Fig1_abstract.png"), width=6, height=5, units="in", res=300)
lwd_abstract <- 4
par(oma=c(0,0,0,0))
par(mar=c(4,4.1,0,0) + 0.1)
plot(counting_days,control_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xaxt="n", yaxt="n",
     xlab="Days since experimental start",ylab=expression("Mean survival rate of "~italic(G.~fossarum)),
     cex.lab=1.5)
axis(1, cex.axis=1.5)
axis(2, cex.axis=1.5)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(control_lwr[c(i,i)]), control_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[1], alpha=alpha_surv), border = NA)
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(shelter_lwr[c(i,i)]), shelter_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[2], alpha=alpha_surv), border = NA)
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(protein_lwr[c(i,i)]), protein_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[3], alpha=alpha_surv), border = NA)
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(shelter_protein_lwr[c(i,i)]), shelter_protein_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[4], alpha=alpha_surv), border = NA)
  # CI intervals as broken line
  # segments(counting_days[i],control_upr[i],counting_days[i+1],control_upr[i], col=col_pal[1], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],control_lwr[i],counting_days[i+1],control_lwr[i], col=col_pal[1], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_upr[i],counting_days[i+1],shelter_upr[i], col=col_pal[2], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_lwr[i],counting_days[i+1],shelter_lwr[i], col=col_pal[2], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],protein_upr[i],counting_days[i+1],protein_upr[i], col=col_pal[3], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],protein_lwr[i],counting_days[i+1],protein_lwr[i], col=col_pal[3], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_protein_upr[i],counting_days[i+1],shelter_protein_upr[i], col=col_pal[4], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_protein_lwr[i],counting_days[i+1],shelter_protein_lwr[i], col=col_pal[4], lwd=lwd_ci, lty=2)
  # Fits as lines (staircase plot, drop first: unlike using option type='s')
  # segments(counting_days[i],control_surv[i],counting_days[i],control_surv[i+1], col=col_pal[1], lwd=lwd_abstract)
  # segments(counting_days[i],control_surv[i+1],counting_days[i+1],control_surv[i+1], col=col_pal[1], lwd=lwd_abstract)
  # segments(counting_days[i],shelter_surv[i],counting_days[i],shelter_surv[i+1], col=col_pal[2], lwd=lwd_abstract)
  # segments(counting_days[i],shelter_surv[i+1],counting_days[i+1],shelter_surv[i+1], col=col_pal[2], lwd=lwd_abstract)
  # segments(counting_days[i],protein_surv[i],counting_days[i],protein_surv[i+1], col=col_pal[3], lwd=lwd_abstract)
  # segments(counting_days[i],protein_surv[i+1],counting_days[i+1],protein_surv[i+1], col=col_pal[3], lwd=lwd_abstract)
  # segments(counting_days[i],shelter_protein_surv[i],counting_days[i],shelter_protein_surv[i+1], col=col_pal[4], lwd=lwd_abstract)
  # segments(counting_days[i],shelter_protein_surv[i+1],counting_days[i+1],shelter_protein_surv[i+1], col=col_pal[4], lwd=lwd_abstract)
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],control_surv, col=col_pal[1], lwd=lwd_abstract, lty=1)
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],shelter_surv, col=col_pal[2], lwd=lwd_abstract, lty=1)
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],protein_surv, col=col_pal[3], lwd=lwd_abstract, lty=1)
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],shelter_protein_surv, col=col_pal[4], lwd=lwd_abstract, lty=1)
}
legend("topright", lab_cond, lty=1, lwd=lwd_abstract, col=col_pal, bty="n", cex=1.5)
# text(min(counting_days),0, survminer::surv_pvalue(surv, data=df)$pval.txt, adj=0)
dev.off()

###### 4.1.2.2) Supp. Fig A7a                                              ####
# Plot of data for supplement
png(paste0(WD,"/Results/Supplement/FigA7.png"), width=6, height=6, units="in", res=300)
par(oma=c(0,0,0,0))
par(mar=c(4,4,0,0) + 0.1)
plot(counting_days,control_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xlab=lab_day,
     # xaxt="n",
     ylab=lab_surv)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(control_lwr[c(i,i)]), control_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[1], alpha=alpha_surv), border = NA)
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(shelter_lwr[c(i,i)]), shelter_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[2], alpha=alpha_surv), border = NA)
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(protein_lwr[c(i,i)]), protein_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[3], alpha=alpha_surv), border = NA)
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(shelter_protein_lwr[c(i,i)]), shelter_protein_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[4], alpha=alpha_surv), border = NA)
  # CI intervals as broken line
  segments(counting_days[i],control_upr[i],counting_days[i+1],control_upr[i], col=col_pal[1], lwd=0.5, lty=3)
  segments(counting_days[i],control_lwr[i],counting_days[i+1],control_lwr[i], col=col_pal[1], lwd=0.5, lty=3)
  segments(counting_days[i],shelter_upr[i],counting_days[i+1],shelter_upr[i], col=col_pal[2], lwd=0.5, lty=3)
  segments(counting_days[i],shelter_lwr[i],counting_days[i+1],shelter_lwr[i], col=col_pal[2], lwd=0.5, lty=3)
  segments(counting_days[i],protein_upr[i],counting_days[i+1],protein_upr[i], col=col_pal[3], lwd=0.5, lty=3)
  segments(counting_days[i],protein_lwr[i],counting_days[i+1],protein_lwr[i], col=col_pal[3], lwd=0.5, lty=3)
  segments(counting_days[i],shelter_protein_upr[i],counting_days[i+1],shelter_protein_upr[i], col=col_pal[4], lwd=0.5, lty=3)
  segments(counting_days[i],shelter_protein_lwr[i],counting_days[i+1],shelter_protein_lwr[i], col=col_pal[4], lwd=0.5, lty=3)
  # Fits as lines (staircase plot, drop first: unlike using option type='s')
  segments(counting_days[i],control_surv[i],counting_days[i],control_surv[i+1], col=col_pal[1], lwd=2, lty=2)
  segments(counting_days[i],control_surv[i+1],counting_days[i+1],control_surv[i+1], col=col_pal[1], lwd=2, lty=2)
  segments(counting_days[i],shelter_surv[i],counting_days[i],shelter_surv[i+1], col=col_pal[2], lwd=2, lty=2)
  segments(counting_days[i],shelter_surv[i+1],counting_days[i+1],shelter_surv[i+1], col=col_pal[2], lwd=2, lty=2)
  segments(counting_days[i],protein_surv[i],counting_days[i],protein_surv[i+1], col=col_pal[3], lwd=2, lty=2)
  segments(counting_days[i],protein_surv[i+1],counting_days[i+1],protein_surv[i+1], col=col_pal[3], lwd=2, lty=2)
  segments(counting_days[i],shelter_protein_surv[i],counting_days[i],shelter_protein_surv[i+1], col=col_pal[4], lwd=2, lty=2)
  segments(counting_days[i],shelter_protein_surv[i+1],counting_days[i+1],shelter_protein_surv[i+1], col=col_pal[4], lwd=2, lty=2)
}
for (j in 1:ncol(control)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],control[,j]/n_ind, col=col_pal[1], lwd=2, lty=1)
}
for (j in 1:ncol(shelter)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],shelter[,j]/n_ind, col=col_pal[2], lwd=2, lty=1)
}
for (j in 1:ncol(protein)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],protein[,j]/n_ind, col=col_pal[3], lwd=2, lty=1)
}
for (j in 1:ncol(shelter_protein)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],shelter_protein[,j]/n_ind, col=col_pal[4], lwd=2, lty=1)
}

legend("topright", lab_cond, lty=1, lwd=lwd_mean, col=col_pal, bty="n", title = paste0("Replicates = ",ncol(control)," each"))
text(min(counting_days),0, survminer::surv_pvalue(surv, data=df)$pval.txt, adj=0)
dev.off()

###### 4.1.2.3) Supp. Fig A7b                                              ####
# Plot of data for supplement
png(paste0(WD,"/Results/Supplement/FigA7_stacked.png"), width=5, height=8, units="in", res=300)
par(mfrow=c(4,1))
par(oma=c(4,4,0,0))
par(mar=c(0.5,0.5,0,0) + 0.1)
plot(counting_days,control_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xlab="",
     xaxt="n",
     ylab="",
     yaxt="n")
axis(1, labels=F)
axis(2, cex.axis=1.5, las=2)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(control_lwr[c(i,i)]), control_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[1], alpha=alpha_surv), border = NA)
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(shelter_lwr[c(i,i)]), shelter_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[2], alpha=alpha_surv), border = NA)
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(protein_lwr[c(i,i)]), protein_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[3], alpha=alpha_surv), border = NA)
  # polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
  #         c(rev(shelter_protein_lwr[c(i,i)]), shelter_protein_upr[c(i,i)]),
  #         col = hex_alpha(hex=col_pal[4], alpha=alpha_surv), border = NA)
  # # CI intervals as broken line
  segments(counting_days[i],control_upr[i],counting_days[i+1],control_upr[i], col=col_pal[1], lwd=0.5, lty=3)
  segments(counting_days[i],control_lwr[i],counting_days[i+1],control_lwr[i], col=col_pal[1], lwd=0.5, lty=3)
  # segments(counting_days[i],shelter_upr[i],counting_days[i+1],shelter_upr[i], col=col_pal[2], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_lwr[i],counting_days[i+1],shelter_lwr[i], col=col_pal[2], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],protein_upr[i],counting_days[i+1],protein_upr[i], col=col_pal[3], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],protein_lwr[i],counting_days[i+1],protein_lwr[i], col=col_pal[3], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_protein_upr[i],counting_days[i+1],shelter_protein_upr[i], col=col_pal[4], lwd=lwd_ci, lty=2)
  # segments(counting_days[i],shelter_protein_lwr[i],counting_days[i+1],shelter_protein_lwr[i], col=col_pal[4], lwd=lwd_ci, lty=2)
  # # Fits as lines (staircase plot, drop first: unlike using option type='s')
  segments(counting_days[i],control_surv[i],counting_days[i],control_surv[i+1], col=col_pal[1], lwd=1, lty=2)
  segments(counting_days[i],control_surv[i+1],counting_days[i+1],control_surv[i+1], col=col_pal[1], lwd=1, lty=2)
  # segments(counting_days[i],shelter_surv[i],counting_days[i],shelter_surv[i+1], col=col_pal[2], lwd=lwd_mean)
  # segments(counting_days[i],shelter_surv[i+1],counting_days[i+1],shelter_surv[i+1], col=col_pal[2], lwd=lwd_mean)
  # segments(counting_days[i],protein_surv[i],counting_days[i],protein_surv[i+1], col=col_pal[3], lwd=lwd_mean)
  # segments(counting_days[i],protein_surv[i+1],counting_days[i+1],protein_surv[i+1], col=col_pal[3], lwd=lwd_mean)
  # segments(counting_days[i],shelter_protein_surv[i],counting_days[i],shelter_protein_surv[i+1], col=col_pal[4], lwd=lwd_mean)
  # segments(counting_days[i],shelter_protein_surv[i+1],counting_days[i+1],shelter_protein_surv[i+1], col=col_pal[4], lwd=lwd_mean)
  # 
 
}
# axis(1,at=counting_days)
for (j in 1:ncol(control)){
  # for (i in 1:(length(control_surv)-1)){
  #   segments(counting_days[i],control[i,j]/n_ind,counting_days[i],control[i+1,j]/n_ind, col=col_pal[1], lwd=1, lty=3)
  #   segments(counting_days[i],control[i+1,j]/n_ind,counting_days[i+1],control[i+1,j]/n_ind, col=col_pal[1], lwd=1, lty=3)
  # }
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],control[,j]/n_ind, col=col_pal[1], lwd=2, lty=1)
}
legend("topright", lab_cond, lty=1, cex=1.5, lwd=lwd_mean, col=col_pal, bty="n", title = paste0("Replicates = ",ncol(control)," each"))

plot(counting_days,shelter_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xlab="",
     xaxt="n",
     ylab="",
     yaxt="n")
axis(1, labels=F)
axis(2, cex.axis=1.5, las=2)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(shelter_lwr[c(i,i)]), shelter_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[2], alpha=alpha_surv), border = NA)
  # # CI intervals as broken line
  segments(counting_days[i],shelter_upr[i],counting_days[i+1],shelter_upr[i], col=col_pal[2], lwd=0.5, lty=3)
  segments(counting_days[i],shelter_lwr[i],counting_days[i+1],shelter_lwr[i], col=col_pal[2], lwd=0.5, lty=3)
  # # Fits as lines (staircase plot, drop first: unlike using option type='s')
  segments(counting_days[i],shelter_surv[i],counting_days[i],shelter_surv[i+1], col=col_pal[2], lwd=1, lty=2)
  segments(counting_days[i],shelter_surv[i+1],counting_days[i+1],shelter_surv[i+1], col=col_pal[2], lwd=1, lty=2)
}
for (j in 1:ncol(shelter)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],shelter[,j]/n_ind, col=col_pal[2], lwd=2, lty=1)
}

plot(counting_days,control_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xlab="",
     xaxt="n",
     ylab="",
     yaxt="n")
axis(1, labels=F)
axis(2, cex.axis=1.5, las=2)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(protein_lwr[c(i,i)]), protein_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[3], alpha=alpha_surv), border = NA)
  # CI intervals as broken line
  segments(counting_days[i],protein_upr[i],counting_days[i+1],protein_upr[i], col=col_pal[3], lwd=0.5, lty=3)
  segments(counting_days[i],protein_lwr[i],counting_days[i+1],protein_lwr[i], col=col_pal[3], lwd=0.5, lty=3)
  # Fits as lines (staircase plot, drop first: unlike using option type='s')
  segments(counting_days[i],protein_surv[i],counting_days[i],protein_surv[i+1], col=col_pal[3], lwd=1, lty=2)
  segments(counting_days[i],protein_surv[i+1],counting_days[i+1],protein_surv[i+1], col=col_pal[3], lwd=1, lty=2)
}
for (j in 1:ncol(protein)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],protein[,j]/n_ind, col=col_pal[3], lwd=2, lty=1)
}

plot(counting_days,control_surv, type="n", xlim=c(0,max(counting_days)), ylim=c(0,1),
     xlab="",
     xaxt="n",
     ylab="",
     yaxt="n")
axis(1, labels=T, cex.axis=1.5)
axis(2, cex.axis=1.5, las=2)
for (i in 1:length(control_surv)){
  # CI intervals as shading
  polygon(c(rev(counting_days[c(i,i+1)]), counting_days[c(i,i+1)]),
          c(rev(shelter_protein_lwr[c(i,i)]), shelter_protein_upr[c(i,i)]),
          col = hex_alpha(hex=col_pal[4], alpha=alpha_surv), border = NA)
  # CI intervals as broken line
  segments(counting_days[i],shelter_protein_upr[i],counting_days[i+1],shelter_protein_upr[i], col=col_pal[4], lwd=0.5, lty=3)
  segments(counting_days[i],shelter_protein_lwr[i],counting_days[i+1],shelter_protein_lwr[i], col=col_pal[4], lwd=0.5, lty=3)
  # Fits as lines (staircase plot, drop first: unlike using option type='s')
  segments(counting_days[i],shelter_protein_surv[i],counting_days[i],shelter_protein_surv[i+1], col=col_pal[4], lwd=1, lty=2)
  segments(counting_days[i],shelter_protein_surv[i+1],counting_days[i+1],shelter_protein_surv[i+1], col=col_pal[4], lwd=1, lty=2)
}
for (j in 1:ncol(shelter_protein)){
  lines(c(0,counting_days + c(diff(counting_days)/2,0))[-10],shelter_protein[,j]/n_ind, col=col_pal[4], lwd=2, lty=1)
}
text(min(counting_days),0, survminer::surv_pvalue(surv, data=df)$pval.txt, adj=0, cex=1.5)
mtext(lab_day, 1, cex=1.5, outer = T, line=2.5)
mtext(lab_surv, 2, cex=1.5, outer = T, line=2.5)
dev.off()

#### 4.1.3) Same figure using ggplot                                       ####
# require(ggplot2)
# plot survival curves using ggplot
# ggsurvplot(surv, data=df, pval = TRUE, xlab=lab_day)

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 4.2) Fig 2                                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
#### 4.2.1) Prepare data for plotting with baseplot                        ####
data_abun <- data.frame(abundance=data.matrix$abundance,treatment=data.matrix$treatment,days=data.matrix$days)
data_temp <- data.frame(abundance=rep(NA,ntim),treatment=rep("a",ntim),days=rep(Time,each=nrep/2))
data_abun <- rbind(data_abun,data_temp)
rm(data_temp)

#### 4.2.2) Using baseplot                                                 ####
# boxplot(densities~treatment+days, data=data.matrix,col=c(col_fix,col_var), xlab="Days", names=c(as.vector(rbind(Time,rep("",length(Time))))), ylab="Population density")
png(paste0(WD,"/Results/Fig2.png"), width=6, height=5, units="in", res=300)
par(oma=c(0,0,0,0),
    mar=c(4,4,2,2) + 0.1)
boxplot(data_abun$abundance~as.factor(data_abun$treatment)+as.factor(data_abun$days),
        xlim=c(1,3*nlevels(factor(data_abun$days))),
        col=c("cyan",col_fix,col_var), # bright color for "break" rows such that errors are obvious
        xlab=lab_day,
        names=c(as.vector(rbind(rep("",length(Time)),Time,rep("",length(Time))))),
        ylab=lab_abun,
        ylim=c(0,max(data_abun$abundance, na.rm = T)+max(data_abun$abundance, na.rm = T)*.2),
        xaxt="n")
vec <- seq(1,3*ntim,by=3)
abline(v=vec, col=col_und, lty=2)
vec2 <- seq(1,3*ntim+3,by=3)
vec3 <- vec2[-length(vec2)] + diff(vec2)/2 # https://stackoverflow.com/a/33534084/6380381 by https://stackoverflow.com/users/986793/neal-fultz
axis(1,Time,at=vec3,
     # cex.axis=0.8,
     las=2)
axis(4, levels(as.factor(Daylengthvariable)), at=as.numeric(levels(as.factor(Daylengthvariable*33))), col=col_und)
mtext(lab_daylength, side=4, adj=0.3, line=1)
segments(min(seq(1,3*ntim,by=1)+.5), 14*33,max(seq(1,3*ntim,by=1)+.5), 14*33, col=col_fix, lwd=3, lty=2)
# abline(h=14*33, col=col_fix, lwd=3, lty=2)
lines(seq(1,3*ntim,by=1)+.5,rep(Daylengthvariable*33,each=3), col=col_var, lwd=3, lty=2)
box()
legend("topleft",lab_daytreat, fill=c(col_fix,col_var),box.lty=0, inset=0.005, cex=0.9)
dev.off()

###### 4.2.2.1) Fig 2 for graphical abstract                               ####
png(paste0(WD,"/Results/Supplement/Fig2_abstract.png"), width=6, height=5, units="in", res=300)
lwd_abstract <- 4
par(oma=c(0,0,0,0),
    mar=c(4,4.1,0,2) + 0.1)
boxplot(data_abun$abundance~as.factor(data_abun$treatment)+as.factor(data_abun$days),
        xlim=c(1,3*nlevels(factor(data_abun$days))),
        col=c("cyan",col_fix,col_var), # bright color for "break" rows such that errors are obvious
        xlab="Days since experimental start",
        names=c(as.vector(rbind(rep("",length(Time)),Time,rep("",length(Time))))),
        ylab=expression("Total abundance of "~italic(G.~fossarum)),
        ylim=c(0,max(data_abun$abundance, na.rm = T)+max(data_abun$abundance, na.rm = T)*.4),
        xaxt="n",
        yaxt="n",
        cex.lab=1.5)
vec <- seq(1,3*ntim,by=3)
abline(v=vec, col=col_und, lty=2)
vec2 <- seq(1,3*ntim+3,by=3)
vec3 <- vec2[-length(vec2)] + diff(vec2)/2 # https://stackoverflow.com/a/33534084/6380381 by https://stackoverflow.com/users/986793/neal-fultz
axis(1,Time,at=vec3,
     cex.axis=1.5,
     las=1)
axis(2, cex.axis=1.5)
axis(4, levels(as.factor(Daylengthvariable)), at=as.numeric(levels(as.factor(Daylengthvariable*33))), col=col_und,
     cex.axis=1.5)
# mtext("Days since experimental start", side=1, line=4, cex=1.5) # if axis1 las=2
mtext(lab_daylength, side=4, adj=0.1, line=0.95, cex=1.5)
segments(min(seq(1,3*ntim,by=1)+.5), 14*33,max(seq(1,3*ntim,by=1)+.5), 14*33, col=col_fix, lwd=lwd_abstract, lty=2)
lines(seq(1,3*ntim,by=1)+.5,rep(Daylengthvariable*33,each=3), col=col_var, lwd=lwd_abstract, lty=2)
legend("topleft",lab_daytreat, fill=c(col_fix,col_var),box.lty=0, inset=0, cex=1.25, horiz = T)
box()
dev.off()

#### 4.2.3) Same figure using ggplot                                       ####
# require(ggplot2)
# ggplot(data.matrix, aes(x=factor(days), y=abundance, fill=factor(treatment))) + 
#   ylab("Absolute densities") + xlab(lab_day) +
#   labs(fill = "Treatments") +                                         # legend title
#   scale_fill_manual(values = c(col_fix,col_var)) +               # color settings
#   geom_boxplot()

# Supplement: Separate the two treatments into two plots and stack them
# png(paste0(WD,"/Results/Supplement/Fig2_alt.png"), width=6, height=10, units="in", res=300)
# par(mfrow=c(2,1)) # two plots with same axis above each other
# par(mar=c(4.5, 4.5, 0.5, 0.5))
# boxplot(abundance[treatment=="fix"]~treatment[treatment=="fix"]+days[treatment=="fix"], data=data.matrix,col=c(col_fix), ylim=c(0,450), xlab=lab_mon, names=Time, ylab=lab_abun)
# legend("topright",lab_daytreat, fill=c(col_fix,col_var),box.lty=0, inset=0.005, cex=1.3)
# boxplot(abundance[treatment=="var"]~treatment[treatment=="var"]+days[treatment=="var"], data=data.matrix,col=c(col_var), ylim=c(0,450), xlab=lab_mon, names=Time, ylab=lab_abun)
# dev.off()

# Supplement: Boxplot for reproductive activity
# png(paste0(WD,"/Results/Supplement/SuppFigX.png"), width=6, height=5, units="in", res=300)
# boxplot(repr.activity~treatment+days, data=data.matrix,col=c(col_fix,col_var), xlab="Days", names=c(as.vector(rbind(Time[-c(1:2)],rep("",length(Time[-c(1:2)]))))), ylab=lab_repr)
# lines(rep(Daylengthvariable+4,each=2), col="darkgrey", lwd=3, lty=2)
# legend("topleft",lab_daytreat, fill=c(col_fix,col_var),box.lty=0, inset=0.005, cex=1.3)
# axis(side=4,at=c(14,16,18,20),labels=c("10","13","15","18"))
# mtext(lab_daylength, side = 4, adj = 0.5, line=1)
# dev.off()

# ggplot(data.matrix, aes(x=factor(days), y=repr.activity, fill=factor(treatment))) + 
#   ylab(lab_repr) + xlab(lab_day) +
#   labs(fill = "Treatments") +                                         # legend title
#   scale_fill_manual(values = c("#0000cc","#00FF00")) +               # color settings
#   geom_line(data = data.matrix, aes(y = daylength+4, group=1), size = 1, col="darkgray") +
#   scale_y_continuous(sec.axis = sec_axis(~ . - 4, name="Daylength (h)", breaks = seq(9,18)))+
#   theme(axis.title.y.right = element_text(color = "darkgray"),
#         axis.text.y.right=element_text(color="darkgray"))+
#   geom_boxplot()

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 4.3) Fig 3                                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# Look at reproductive activity in relation to abundance to correct for abundance differences

#### 4.3.1) Prepare data for plotting with baseplot                        ####
data_repr <- data.frame(repr.act.rel=data.matrix$repr.act.rel,treatment=data.matrix$treatment,days=data.matrix$days)
data_temp <- data.frame(repr.act.rel=rep(NA,ntim),treatment=rep("a",ntim),days=rep(Time,each=nrep/2))
data_repr <- rbind(data_repr,data_temp)
rm(data_temp)

#### 4.3.2) Using baseplot                                                 ####
scaling <- 1.3
png(paste0(WD,"/Results/Fig3.png"), width=6, height=10, units="in", res=300)
par(mfrow=c(2,1), # two plots with same axis above each other
    oma=c(0,0,0,0),
    mar=c(4,4,2,2) + 0.1) 
# boxplot(data.matrix$repr.act.rel~as.factor(data.matrix$treatment),
#         col=c(col_fix,col_var),
#         xlab=lab_treat,
#         names=lab_daytreat,
#         ylab=lab_repr_rel,
#         ylim=c(0,0.8))
vioplot::vioplot(data.matrix$repr.act.rel~as.factor(data.matrix$treatment),
                 col = c(col_fix,col_var),
                 xlab="",
                 names=lab_daytreat,
                 ylab="",
                 # border = c(col_fix,col_var),
                 ylim=c(0,0.8),
                 cex.axis=scaling)
title(xlab=lab_treat,ylab=lab_repr_rel,cex.lab=scaling) # since cex.lab seems broken in vioplot()
# text(par("usr")[1], par("usr")[4], "A)", cex=2, adj=c(0.5,1))
mtext("A)", side=3, line=0.5, adj=0.0, cex=1.5*scaling, outer=F)

# relative reproductive activity (divided by #adults/time point/tank)
boxplot(data_repr$repr.act.rel~as.factor(data_repr$treatment)+as.factor(data_repr$days),
        xlim=c(1,3*nlevels(factor(data_repr$days))),
        col=c("cyan",col_fix,col_var), # bright color for "break" rows such that errors are obvious
        xlab=lab_day,
        names=c(as.vector(rbind(rep("",length(Time)),Time,rep("",length(Time))))),
        ylab=lab_repr_rel,
        ylim=c(0,0.8),
        xaxt="n",
        cex.axis=scaling,
        cex.lab=scaling)
vec <- seq(1,3*ntim,by=3)
abline(v=vec, col=col_und, lty=2)
vec2 <- seq(1,3*ntim+3,by=3)
vec3 <- vec2[-length(vec2)] + diff(vec2)/2 # https://stackoverflow.com/a/33534084/6380381 by https://stackoverflow.com/users/986793/neal-fultz
axis(1,Time,at=vec3,
     cex.axis=scaling,
     las=2)
axis(4, levels(as.factor(Daylengthvariable)), at=as.numeric(levels(as.factor(Daylengthvariable/20))), col=col_und, cex.axis=scaling)
mtext(lab_daylength, side=4, adj=0.3, line=1, cex=scaling)
segments(min(seq(1,3*ntim,by=1)+.5), 14/20,max(seq(1,3*ntim,by=1)+.5), 14/20, col=col_fix, lwd=3, lty=2)
# abline(h=14/20, col=col_fix, lwd=3, lty=2)
lines(seq(1,3*ntim,by=1)+.5,rep(Daylengthvariable/20,each=3), col=col_var, lwd=3, lty=2)
box()
legend("topleft",lab_daytreat, fill=c(col_fix,col_var),box.lty=0, inset=0.005, cex=0.9*scaling, bg=hex_alpha(hex="#FFFFFF", alpha=70))
mtext("B)", side=3, line=0.5, adj=0.0, cex=1.5*scaling, outer=F)
dev.off()

#### 4.3.3) Using ggplot                                                  ####
# require(ggplot2)
# ggplot(data.matrix, aes(x=factor(days), y=repr.act.rel, fill=factor(treatment))) + 
#   ylab(lab_repr_rel) + xlab(lab_day) +
#   labs(fill = "Treatments") +                                         # legend title
#   scale_fill_manual(values = c("#0000cc","#00FF00")) +               # color settings
#   # geom_line(data = data.matrix, aes(y = daylength/20, group=1), size = 1, col="darkgray") +
#   # scale_y_continuous(sec.axis = sec_axis(~ . *20, name="Daylength (h)", breaks = seq(10,20)))+
#   theme(axis.title.y.right = element_text(color = "darkgray"),
#         axis.text.y.right=element_text(color="darkgray"))+
#   geom_boxplot()

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 4.4) Fig 4                                                              ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#

#### 4.4.1) Prepare data for plotting with baseplot                        ####
data_alga <- data.frame(abundance=data.matrix$abundance,algae=data.matrix$algae,days=data.matrix$days)
data_temp <- data.frame(abundance=rep(NA,ntim),algae=rep("a",ntim),days=rep(Time,each=nrep/2))
data_alga <- rbind(data_alga,data_temp)
rm(data_temp)

#### 4.4.2) Using baseplot                                                 ####
# boxplot(densities~treatment+days, data=data.matrix,col=c(col_fix,col_var), xlab="Days", names=c(as.vector(rbind(Time,rep("",length(Time))))), ylab="Population density")
png(paste0(WD,"/Results/Fig4.png"), width=6, height=5, units="in", res=300)
par(oma=c(0,0,0,0),
    mar=c(4,4,2,2) + 0.1)
boxplot(data_alga$abundance~as.factor(data_alga$algae)+as.factor(data_alga$days),
        xlim=c(1,4*nlevels(factor(data_alga$days))),
        col=c(col_alg,"cyan"), # bright color for "break" rows such that errors are obvious
        xlab=lab_day,
        names=c(as.vector(rbind(rep("",length(Time)),rep("",length(Time)),Time,rep("",length(Time))))),
        ylab=lab_abun,
        ylim=c(0,max(data_alga$abundance, na.rm = T)+max(data_alga$abundance, na.rm = T)*.2),
        xaxt="n")
vec <- seq(0,4*ntim,by=4)
abline(v=vec, col=col_und, lty=2)
vec2 <- seq(0,4*(ntim-1),by=4)
vec2 <- vec2 + unique(diff(vec2)/2) # https://stackoverflow.com/a/33534084/6380381 by https://stackoverflow.com/users/986793/neal-fultz
axis(1,Time,at=vec2,
     # cex.axis=0.8,
     las=2)
# axis(4, levels(as.factor(Daylengthvariable)), at=as.numeric(levels(as.factor(Daylengthvariable*33))), col=col_und)
# mtext(lab_daylength, side=4, adj=0.3, line=1)
# lines(seq(1,4*ntim,by=1)-.5,rep(Daylengthvariable*33,each=4), col="darkgrey", lwd=3, lty=2)
box()
legend("topleft",lab_alg, fill=col_alg,box.lty=0, inset=0.005, cex=0.9)
dev.off()

#### 4.4.3) Using ggplot                                                  ####
# require(ggplot2)
# p<-ggplot(data.matrix, aes(x=reorder(factor(days),as.numeric(days), na.rm=T), y=abundance, fill=factor(algae))) + 
#   ylab("Absolute densities") + xlab(lab_day) +
#   labs(fill = "Observation") +                                         # legend title
#   scale_fill_manual(values = c("aquamarine","chartreuse2","chartreuse4"), labels = c("low algae", "medium algae", "high algae")) +       # color settings + legend labels
#   geom_boxplot()
# 
# p +theme(panel.grid.major = element_blank(),
#          panel.grid.minor = element_blank(), panel.background = element_rect(fill = 'white')) +
#   coord_cartesian(xlim=c(6,14))+ geom_vline(xintercept = c(1.5:14.5), linetype="dashed", 
#                                             color = "lightgray", size=0.5)
# 
# p +theme_bw() +
#   coord_cartesian(xlim=c(6,14))+ geom_vline(xintercept = c(1.5:14.5), linetype="dashed", 
#                                             color = "lightgray", size=0.5) + 
#   theme(text = element_text(size = 20)) # increase all font sizes
# 
# png(paste0(WD,"/Results/Fig4a.png"), width=6, height=5, units="in", res=300)
# p
# dev.off()

## _ _ _ _ _ _ _ _ _ _ _ _ _ _ _ _####
## 4.5) Supp Fig A11                                                       ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
# boxplot for age structure

#### 4.5.1) Prepare data for plotting with baseplot                        ####
data_ages <- data.frame(agestructure=data.matrix$agestructure,treatment=data.matrix$treatment,days=data.matrix$days)
data_temp <- data.frame(agestructure=rep(NA,ntim),treatment=rep("a",ntim),days=rep(Time,each=nrep/2))
data_ages <- rbind(data_ages,data_temp)
rm(data_temp)

#### 4.5.2) Using baseplot                                                 ####
png(paste0(WD,"/Results/Supplement/FigA11.png"), width=6, height=5, units="in", res=300)
par(oma=c(0,0,0,0),
    mar=c(4,4,2,2) + 0.1)
boxplot(data_ages$agestructure~as.factor(data_ages$treatment)+as.factor(data_ages$days),
        xlim=c(1,3*nlevels(factor(data_ages$days))),
        col=c("cyan",col_fix,col_var), # bright color for "break" rows such that errors are obvious
        xlab=lab_day,
        names=c(as.vector(rbind(rep("",length(Time)),Time,rep("",length(Time))))),
        ylab=lab_juv,
        ylim=c(0,max(data_ages$agestructure, na.rm = T)+max(data_ages$agestructure, na.rm = T)*.6),
        xaxt="n",
        yaxt="n")
vec <- seq(1,3*ntim,by=3)
abline(v=vec, col=col_und, lty=2)
vec2 <- seq(1,3*ntim+3,by=3)
vec3 <- vec2[-length(vec2)] + diff(vec2)/2 # https://stackoverflow.com/a/33534084/6380381 by https://stackoverflow.com/users/986793/neal-fultz
axis(1,Time,at=vec3,
     # cex.axis=0.8,
     las=2)
axis(2, seq(0,1,by=0.2), at=seq(0,1,by=0.2))
axis(4, levels(as.factor(Daylengthvariable)), at=as.numeric(levels(as.factor(Daylengthvariable/10))), col=col_und)
mtext(lab_daylength, side=4, adj=0.3, line=1)
segments(min(seq(1,3*ntim,by=1)+.5), 14/10,max(seq(1,3*ntim,by=1)+.5), 14/10, col=col_fix, lwd=3, lty=2)
# abline(h=14/10, col=col_fix, lwd=3, lty=2)
lines(seq(1,3*ntim,by=1)+.5,rep(Daylengthvariable/10,each=3), col=col_var, lwd=3, lty=2)
box()
legend("topleft",lab_daytreat, fill=c(col_fix,col_var),box.lty=0, inset=0.005, cex=0.9)
dev.off()

## ___________________________####
# 5) End of script                                                         ####
#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#*#
if("beepr"%in%rownames(installed.packages())){beepr::beep(3)}
