#ifndef REACLIB_RATES_H
#define REACLIB_RATES_H

#include <AMReX.H>
#include <AMReX_Print.H>

#include <actual_network.H>

using namespace Rates;

struct rate_eval_t {
    Array1D<Real, 1, NumRates>  screened_rates;
    Array1D<Real, 1, NumRates>  dscreened_rates_dT;
    Array1D<Real, NrateReaclib+1, NrateReaclib+NrateTabular> add_energy_rate;
};

struct tf_t {
    Real T9;
    Real T9i;
    Real T943i;
    Real T923i;
    Real T913i;
    Real T913;
    Real T923;
    Real T953;
    Real lnT9;
};

AMREX_GPU_HOST_DEVICE AMREX_INLINE
tf_t evaluate_tfactors(const Real T)
{

    tf_t tf;
    tf.T9 = T / 1.e9_rt;
    tf.T9i = 1.0_rt / tf.T9;
    tf.T913 = std::cbrt(tf.T9);
    tf.T913i = 1.0_rt / tf.T913;
    tf.T923i = tf.T913i * tf.T913i;
    tf.T943i = tf.T9i * tf.T913i;
    tf.T923 = tf.T913 * tf.T913;
    tf.T953 = tf.T9 * tf.T923;
    tf.lnT9 = std::log(tf.T9);

    return tf;
}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_n13__c13__weak__wc12(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n13 --> c13

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // wc12w
    ln_set_rate =  -6.7601;

    dln_set_rate_dT9 = 0.0;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_o14__n14__weak__wc12(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // o14 --> n14

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // wc12w
    ln_set_rate =  -4.62354;

    dln_set_rate_dT9 = 0.0;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_o15__n15__weak__wc12(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // o15 --> n15

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // wc12w
    ln_set_rate =  -5.17053;

    dln_set_rate_dT9 = 0.0;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_f17__o17__weak__wc12(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // f17 --> o17

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // wc12w
    ln_set_rate =  -4.53318;

    dln_set_rate_dT9 = 0.0;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_c12__n13(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // c12 + p --> n13

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // ls09n
    ln_set_rate =  17.1482 + -13.692 * tfactors.T913i + -0.230881 * tfactors.T913
                         + 4.44362 * tfactors.T9 + -3.15898 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -13.692 * tfactors.T943i + (1.0/3.0) * -0.230881 * tfactors.T923i
                              + 4.44362 + (5.0/3.0) * -3.15898 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // ls09r
    ln_set_rate =  17.5428 + -3.77849 * tfactors.T9i + -5.10735 * tfactors.T913i + -2.24111 * tfactors.T913
                         + 0.148883 * tfactors.T9 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  3.77849 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * -5.10735 * tfactors.T943i + (1.0/3.0) * -2.24111 * tfactors.T923i
                              + 0.148883 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_he4_c12__o16(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // c12 + he4 --> o16

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // nac2 
    ln_set_rate =  254.634 + -1.84097 * tfactors.T9i + 103.411 * tfactors.T913i + -420.567 * tfactors.T913
                         + 64.0874 * tfactors.T9 + -12.4624 * tfactors.T953 + 137.303 * tfactors.lnT9;

    dln_set_rate_dT9 =  1.84097 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * 103.411 * tfactors.T943i + (1.0/3.0) * -420.567 * tfactors.T923i
                              + 64.0874 + (5.0/3.0) * -12.4624 * tfactors.T923 + 137.303 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // nac2 
    ln_set_rate =  69.6526 + -1.39254 * tfactors.T9i + 58.9128 * tfactors.T913i + -148.273 * tfactors.T913
                         + 9.08324 * tfactors.T9 + -0.541041 * tfactors.T953 + 70.3554 * tfactors.lnT9;

    dln_set_rate_dT9 =  1.39254 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * 58.9128 * tfactors.T943i + (1.0/3.0) * -148.273 * tfactors.T923i
                              + 9.08324 + (5.0/3.0) * -0.541041 * tfactors.T923 + 70.3554 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_c13__n14(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // c13 + p --> n14

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // nacrr
    ln_set_rate =  15.1825 + -13.5543 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  13.5543 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // nacrn
    ln_set_rate =  18.5155 + -13.72 * tfactors.T913i + -0.450018 * tfactors.T913
                         + 3.70823 * tfactors.T9 + -1.70545 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -13.72 * tfactors.T943i + (1.0/3.0) * -0.450018 * tfactors.T923i
                              + 3.70823 + (5.0/3.0) * -1.70545 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // nacrr
    ln_set_rate =  13.9637 + -5.78147 * tfactors.T9i + -0.196703 * tfactors.T913
                         + 0.142126 * tfactors.T9 + -0.0238912 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  5.78147 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * -0.196703 * tfactors.T923i
                              + 0.142126 + (5.0/3.0) * -0.0238912 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_n13__o14(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n13 + p --> o14

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // lg06r
    ln_set_rate =  10.9971 + -6.12602 * tfactors.T9i + 1.57122 * tfactors.T913i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  6.12602 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * 1.57122 * tfactors.T943i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // lg06n
    ln_set_rate =  18.1356 + -15.1676 * tfactors.T913i + 0.0955166 * tfactors.T913
                         + 3.0659 * tfactors.T9 + -0.507339 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -15.1676 * tfactors.T943i + (1.0/3.0) * 0.0955166 * tfactors.T923i
                              + 3.0659 + (5.0/3.0) * -0.507339 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_n14__o15(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n14 + p --> o15

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // im05n
    ln_set_rate =  17.01 + -15.193 * tfactors.T913i + -0.161954 * tfactors.T913
                         + -7.52123 * tfactors.T9 + -0.987565 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -15.193 * tfactors.T943i + (1.0/3.0) * -0.161954 * tfactors.T923i
                              + -7.52123 + (5.0/3.0) * -0.987565 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // im05r
    ln_set_rate =  6.73578 + -4.891 * tfactors.T9i
                         + 0.0682 * tfactors.lnT9;

    dln_set_rate_dT9 =  4.891 * tfactors.T9i * tfactors.T9i
                              + 0.0682 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // im05r
    ln_set_rate =  7.65444 + -2.998 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  2.998 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // im05n
    ln_set_rate =  20.1169 + -15.193 * tfactors.T913i + -4.63975 * tfactors.T913
                         + 9.73458 * tfactors.T9 + -9.55051 * tfactors.T953 + 0.333333 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -15.193 * tfactors.T943i + (1.0/3.0) * -4.63975 * tfactors.T923i
                              + 9.73458 + (5.0/3.0) * -9.55051 * tfactors.T923 + 0.333333 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_he4_n14__f18(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n14 + he4 --> f18

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // il10n
    ln_set_rate =  21.5339 + -36.2504 * tfactors.T913i
                         + -5.0 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -36.2504 * tfactors.T943i
                              + (5.0/3.0) * -5.0 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  13.8995 + -10.9656 * tfactors.T9i + -5.6227 * tfactors.T913i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  10.9656 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * -5.6227 * tfactors.T943i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  0.196838 + -5.16034 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  5.16034 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_n15__o16(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n15 + p --> o16

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // li10n
    ln_set_rate =  20.0176 + -15.24 * tfactors.T913i + 0.334926 * tfactors.T913
                         + 4.59088 * tfactors.T9 + -4.78468 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -15.24 * tfactors.T943i + (1.0/3.0) * 0.334926 * tfactors.T923i
                              + 4.59088 + (5.0/3.0) * -4.78468 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // li10r
    ln_set_rate =  14.5444 + -10.2295 * tfactors.T9i
                         + 0.0459037 * tfactors.T9 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  10.2295 * tfactors.T9i * tfactors.T9i
                              + 0.0459037 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // li10r
    ln_set_rate =  6.59056 + -2.92315 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  2.92315 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_o16__f17(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // o16 + p --> f17

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // ia08n
    ln_set_rate =  19.0904 + -16.696 * tfactors.T913i + -1.16252 * tfactors.T913
                         + 0.267703 * tfactors.T9 + -0.0338411 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -16.696 * tfactors.T943i + (1.0/3.0) * -1.16252 * tfactors.T923i
                              + 0.267703 + (5.0/3.0) * -0.0338411 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_o17__f18(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // o17 + p --> f18

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // il10n
    ln_set_rate =  15.8929 + -16.4035 * tfactors.T913i + 4.31885 * tfactors.T913
                         + -0.709921 * tfactors.T9 + -2.0 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -16.4035 * tfactors.T943i + (1.0/3.0) * 4.31885 * tfactors.T923i
                              + -0.709921 + (5.0/3.0) * -2.0 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  9.39048 + -6.22828 * tfactors.T9i + 2.31435 * tfactors.T913
                         + -0.302835 * tfactors.T9 + 0.020133 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  6.22828 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * 2.31435 * tfactors.T923i
                              + -0.302835 + (5.0/3.0) * 0.020133 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  -13.077 + -0.746296 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  0.746296 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_he4_n13__p_o16(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n13 + he4 --> p + o16

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // cf88n
    ln_set_rate =  40.4644 + -35.829 * tfactors.T913i + -0.530275 * tfactors.T913
                         + -0.982462 * tfactors.T9 + 0.0808059 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -35.829 * tfactors.T943i + (1.0/3.0) * -0.530275 * tfactors.T923i
                              + -0.982462 + (5.0/3.0) * 0.0808059 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_n15__he4_c12(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // n15 + p --> he4 + c12

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // nacrn
    ln_set_rate =  27.4764 + -15.253 * tfactors.T913i + 1.59318 * tfactors.T913
                         + 2.4479 * tfactors.T9 + -2.19708 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -15.253 * tfactors.T943i + (1.0/3.0) * 1.59318 * tfactors.T923i
                              + 2.4479 + (5.0/3.0) * -2.19708 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // nacrr
    ln_set_rate =  -6.57522 + -1.1638 * tfactors.T9i + 22.7105 * tfactors.T913
                         + -2.90707 * tfactors.T9 + 0.205754 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  1.1638 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * 22.7105 * tfactors.T923i
                              + -2.90707 + (5.0/3.0) * 0.205754 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // nacrr
    ln_set_rate =  20.8972 + -7.406 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  7.406 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // nacrr
    ln_set_rate =  -4.87347 + -2.02117 * tfactors.T9i + 30.8497 * tfactors.T913
                         + -8.50433 * tfactors.T9 + -1.54426 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  2.02117 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * 30.8497 * tfactors.T923i
                              + -8.50433 + (5.0/3.0) * -1.54426 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_he4_o14__p_f17(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // o14 + he4 --> p + f17

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // Ha96n
    ln_set_rate =  40.8358 + -39.388 * tfactors.T913i + -17.4673 * tfactors.T913
                         + 35.3029 * tfactors.T9 + -24.8162 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -39.388 * tfactors.T943i + (1.0/3.0) * -17.4673 * tfactors.T923i
                              + 35.3029 + (5.0/3.0) * -24.8162 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // Ha96r
    ln_set_rate =  16.3087 + -22.51 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  22.51 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // Ha96r
    ln_set_rate =  11.1184 + -13.6 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  13.6 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // Ha96r
    ln_set_rate =  -106.091 + -0.453036 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  0.453036 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // Ha96r
    ln_set_rate =  12.1289 + -12.0223 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  12.0223 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // Ha96r
    ln_set_rate =  18.6518 + -26.0 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  26.0 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_o17__he4_n14(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // o17 + p --> he4 + n14

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // il10r
    ln_set_rate =  5.5336 + -2.11477 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  2.11477 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  -7.20763 + -0.753395 * tfactors.T9i
                         + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  0.753395 * tfactors.T9i * tfactors.T9i
                              + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10n
    ln_set_rate =  19.579 + -16.9078 * tfactors.T913i
                         + -2.0 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -16.9078 * tfactors.T943i
                              + (5.0/3.0) * -2.0 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  10.174 + -4.95865 * tfactors.T9i + 5.10182 * tfactors.T913
                         + 0.379373 * tfactors.T9 + -0.0672515 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  4.95865 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * 5.10182 * tfactors.T923i
                              + 0.379373 + (5.0/3.0) * -0.0672515 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_p_f18__he4_o15(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // f18 + p --> he4 + o15

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // il10n
    ln_set_rate =  62.0058 + -21.4023 * tfactors.T913i + -80.8891 * tfactors.T913
                         + 134.6 * tfactors.T9 + -126.504 * tfactors.T953 + -0.666667 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -21.4023 * tfactors.T943i + (1.0/3.0) * -80.8891 * tfactors.T923i
                              + 134.6 + (5.0/3.0) * -126.504 * tfactors.T923 + -0.666667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  1.75704 + -3.01675 * tfactors.T9i + 13.3223 * tfactors.T913
                         + -1.36696 * tfactors.T9 + 0.0757363 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  3.01675 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * 13.3223 * tfactors.T923i
                              + -1.36696 + (5.0/3.0) * 0.0757363 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // il10r
    ln_set_rate =  -31.7388 + -0.376432 * tfactors.T9i + 61.738 * tfactors.T913
                         + -108.29 * tfactors.T9 + -34.2365 * tfactors.T953 + -1.5 * tfactors.lnT9;

    dln_set_rate_dT9 =  0.376432 * tfactors.T9i * tfactors.T9i + (1.0/3.0) * 61.738 * tfactors.T923i
                              + -108.29 + (5.0/3.0) * -34.2365 * tfactors.T923 + -1.5 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}

AMREX_GPU_HOST_DEVICE AMREX_INLINE
void rate_he4_he4_he4__c12(const tf_t& tfactors, Real& rate, Real& drate_dT) {

    // he4 + he4 + he4 --> c12

    rate = 0.0;
    drate_dT = 0.0;

    Real ln_set_rate{0.0};
    Real dln_set_rate_dT9{0.0};
    Real set_rate{0.0};

    // fy05r
    ln_set_rate =  -24.3505 + -4.12656 * tfactors.T9i + -13.49 * tfactors.T913i + 21.4259 * tfactors.T913
                         + -1.34769 * tfactors.T9 + 0.0879816 * tfactors.T953 + -13.1653 * tfactors.lnT9;

    dln_set_rate_dT9 =  4.12656 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * -13.49 * tfactors.T943i + (1.0/3.0) * 21.4259 * tfactors.T923i
                              + -1.34769 + (5.0/3.0) * 0.0879816 * tfactors.T923 + -13.1653 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // fy05r
    ln_set_rate =  -11.7884 + -1.02446 * tfactors.T9i + -23.57 * tfactors.T913i + 20.4886 * tfactors.T913
                         + -12.9882 * tfactors.T9 + -20.0 * tfactors.T953 + -2.16667 * tfactors.lnT9;

    dln_set_rate_dT9 =  1.02446 * tfactors.T9i * tfactors.T9i + -(1.0/3.0) * -23.57 * tfactors.T943i + (1.0/3.0) * 20.4886 * tfactors.T923i
                              + -12.9882 + (5.0/3.0) * -20.0 * tfactors.T923 + -2.16667 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

    // fy05n
    ln_set_rate =  -0.971052 + -37.06 * tfactors.T913i + 29.3493 * tfactors.T913
                         + -115.507 * tfactors.T9 + -10.0 * tfactors.T953 + -1.33333 * tfactors.lnT9;

    dln_set_rate_dT9 =  + -(1.0/3.0) * -37.06 * tfactors.T943i + (1.0/3.0) * 29.3493 * tfactors.T923i
                              + -115.507 + (5.0/3.0) * -10.0 * tfactors.T923 + -1.33333 * tfactors.T9i;

    // avoid underflows by zeroing rates in [0.0, 1.e-100]
    ln_set_rate = std::max(ln_set_rate, -230.0);
    set_rate = std::exp(ln_set_rate);
    rate += set_rate;
    drate_dT += set_rate * dln_set_rate_dT9 / 1.0e9;

}


AMREX_GPU_HOST_DEVICE AMREX_INLINE
void
fill_reaclib_rates(const tf_t& tfactors, rate_eval_t& rate_eval)
{

    Real rate;
    Real drate_dT;

    rate_n13__c13__weak__wc12(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_n13__c13__weak__wc12) = rate;
    rate_eval.dscreened_rates_dT(k_n13__c13__weak__wc12) = drate_dT;

    rate_o14__n14__weak__wc12(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_o14__n14__weak__wc12) = rate;
    rate_eval.dscreened_rates_dT(k_o14__n14__weak__wc12) = drate_dT;

    rate_o15__n15__weak__wc12(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_o15__n15__weak__wc12) = rate;
    rate_eval.dscreened_rates_dT(k_o15__n15__weak__wc12) = drate_dT;

    rate_f17__o17__weak__wc12(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_f17__o17__weak__wc12) = rate;
    rate_eval.dscreened_rates_dT(k_f17__o17__weak__wc12) = drate_dT;

    rate_p_c12__n13(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_c12__n13) = rate;
    rate_eval.dscreened_rates_dT(k_p_c12__n13) = drate_dT;

    rate_he4_c12__o16(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_he4_c12__o16) = rate;
    rate_eval.dscreened_rates_dT(k_he4_c12__o16) = drate_dT;

    rate_p_c13__n14(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_c13__n14) = rate;
    rate_eval.dscreened_rates_dT(k_p_c13__n14) = drate_dT;

    rate_p_n13__o14(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_n13__o14) = rate;
    rate_eval.dscreened_rates_dT(k_p_n13__o14) = drate_dT;

    rate_p_n14__o15(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_n14__o15) = rate;
    rate_eval.dscreened_rates_dT(k_p_n14__o15) = drate_dT;

    rate_he4_n14__f18(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_he4_n14__f18) = rate;
    rate_eval.dscreened_rates_dT(k_he4_n14__f18) = drate_dT;

    rate_p_n15__o16(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_n15__o16) = rate;
    rate_eval.dscreened_rates_dT(k_p_n15__o16) = drate_dT;

    rate_p_o16__f17(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_o16__f17) = rate;
    rate_eval.dscreened_rates_dT(k_p_o16__f17) = drate_dT;

    rate_p_o17__f18(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_o17__f18) = rate;
    rate_eval.dscreened_rates_dT(k_p_o17__f18) = drate_dT;

    rate_he4_n13__p_o16(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_he4_n13__p_o16) = rate;
    rate_eval.dscreened_rates_dT(k_he4_n13__p_o16) = drate_dT;

    rate_p_n15__he4_c12(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_n15__he4_c12) = rate;
    rate_eval.dscreened_rates_dT(k_p_n15__he4_c12) = drate_dT;

    rate_he4_o14__p_f17(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_he4_o14__p_f17) = rate;
    rate_eval.dscreened_rates_dT(k_he4_o14__p_f17) = drate_dT;

    rate_p_o17__he4_n14(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_o17__he4_n14) = rate;
    rate_eval.dscreened_rates_dT(k_p_o17__he4_n14) = drate_dT;

    rate_p_f18__he4_o15(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_p_f18__he4_o15) = rate;
    rate_eval.dscreened_rates_dT(k_p_f18__he4_o15) = drate_dT;

    rate_he4_he4_he4__c12(tfactors, rate, drate_dT);
    rate_eval.screened_rates(k_he4_he4_he4__c12) = rate;
    rate_eval.dscreened_rates_dT(k_he4_he4_he4__c12) = drate_dT;


}

#endif
