/*!
 *  \file source.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_sources_h
#define sequence_sources_h

#include <boost/iostreams/device/mapped_file.hpp>
#include <fstream>
#include <stdexcept>
#include <type_traits>

namespace seq
{
    template<typename Source>
    class source_traits
    {
        public:
            typedef Source                                source_type;
            typedef typename source_type::element_type    element_type;
            typedef typename source_type::size_type       size_type;

            static_assert (std::is_move_constructible<source_type>::value,
                           "A source must be move contructible!");
            static_assert (!std::is_copy_constructible<source_type>::value,
                           "A source cannot be copy constructible!");
            static_assert (std::is_move_assignable<source_type>::value,
                           "A source must be move assignable!");
            static_assert (!std::is_copy_assignable<source_type>::value,
                           "A source cannot be copy assignable!");
            static_assert (std::is_integral<size_type>::value,
                           "The size type must be an integral type!");

            static inline void open (source_type & src, const char * filename) {
                src.open (filename);
            }

            static inline bool good (source_type & src) {
                return src.good ();
            }

            static inline void read (source_type & src, element_type ** s, size_type count) {
                src.read (s, count);
            }

            static inline size_type gcount (source_type & src) {
                return src.gcount ();
            }
    };

    class source_error : public std::runtime_error
    {
        public:
            using std::runtime_error::runtime_error;
    };

    class stream_source
    {
        private:
            std::ifstream stream_;
            bool last_op_failed_ { false };

        public:
            typedef stream_source source_type;
            typedef char          element_type;
            typedef long          size_type;

            stream_source () = default;
            explicit stream_source (const char * filename) {
                open (filename);
            }
            stream_source (stream_source &&) = default;
            stream_source (const stream_source &) = delete;

            ~stream_source () = default;

            stream_source & operator = (stream_source &&) = default;
            stream_source & operator = (const stream_source &) = delete;

            void open (const char * filename) {
                stream_.exceptions (std::ios::badbit);
                stream_.open (filename, std::ios::in | std::ios::binary);
            }

            bool good () noexcept {
                return (stream_.peek () != -1) and !last_op_failed_;
            }

            void read (element_type ** s, size_type count) {
                if (stream_.read (*s, count).gcount () != count) {
                    last_op_failed_ = true;
                }
            }

            size_type gcount () const noexcept {
                return stream_.gcount ();
            }
    };

    class simple_mapped_source
    {
        public:
            typedef simple_mapped_source                  source_type;
            typedef char                                  element_type;
            typedef long                                  size_type;

        private:
            boost::iostreams::mapped_file_source mfs_;
            size_type processed_ { 0 };
            size_type gcount_ { 0 };

        public:
            simple_mapped_source () = default;
            explicit simple_mapped_source (const char * filename);
            simple_mapped_source (simple_mapped_source &&) = default;
            simple_mapped_source (const simple_mapped_source &) = delete;

            ~simple_mapped_source () = default;

            simple_mapped_source & operator = (simple_mapped_source &&) = default;
            simple_mapped_source & operator = (const simple_mapped_source &) = delete;

            void open (const char *);

            bool good () noexcept;

            void read (element_type **, size_type);

            size_type gcount () const noexcept;
    };

    class region_mapped_source
    {
        public:
            typedef region_mapped_source                  source_type;
            typedef char                                  element_type;
            typedef long                                  size_type;

        private:
            boost::iostreams::mapped_file_source old_mfs_;
            boost::iostreams::mapped_file_source mfs_;
            std::string filename_;
            size_type file_size_ { 0 };
            size_type processed_ { 0 };
            size_type gcount_ { 0 };

        public:
            region_mapped_source () = default;
            explicit region_mapped_source (const char * filename);
            region_mapped_source (region_mapped_source &&) = default;
            region_mapped_source (const region_mapped_source &) = delete;

            ~region_mapped_source () = default;

            region_mapped_source & operator = (region_mapped_source &&) = default;
            region_mapped_source & operator = (const region_mapped_source &) = delete;

            void open (const char *);

            bool good () noexcept;

            void read (element_type **, size_type);

            size_type gcount () const noexcept;
    };
}

#endif
