/*!
 *  \file sequence.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_sequence_h
#define sequence_sequence_h

#include <type_traits>
#include <string>
#include <iostream>
#include <tuple>

#include "common.h"

namespace seq
{
    
    /*!
     * Traits for a DNA sequence
     */
    template <class Sequence>
    class sequence_traits
    {
    public:
        
        //! \brief The container used for storage
        typedef typename Sequence::container_type container_type;
        
        //! \brief The container used for storage
        typedef typename container_type::value_type value_type;
        
        //! \brief The properties type
        typedef typename Sequence::properties_type properties_type;
        
        static_assert (std::is_move_constructible<Sequence>::value,
                       "A container value must be move contructible!");
        static_assert (std::is_move_assignable<Sequence>::value,
                       "A container value be move assignable!");
        
        //! \brief Returns the sequence of the read (the underlying container)
        template<class T>
        static container_type& get(T &s)
        {
            return s.get();
        }
        
        //! \brief Returns the const sequence of the read (the underlying container)
        template<class T>
        static const container_type& get(const T &s)
        {
            return s.get();
        }
        
        //! \brief Returns the sequence properties
        template<class T>
        static properties_type& properties(T &s)
        {
            return s.properties();
        }
        
        //! \brief Returns the const sequence properties
        template<class T>
        static const properties_type& properties(const T &s)
        {
            return s.properties();
        }
        
        //! \brief Returns a property associated to a read
        template<std::size_t idx, class T>
        static auto property(T &s) -> typename std::tuple_element<idx, typename T::properties_type>::type&
        {
            return std::get<idx>(s.properties());
        };
        
        //! \brief Returns a const property associated to a read
        template<std::size_t idx, class T>
        static auto property(const T &s) -> const typename std::tuple_element<idx, typename T::properties_type>::type&
        {
            return std::get<idx>(s.properties());
        };

        
        //! \brief Returns the count for the sequence
        template<class T>
        static std::size_t& count(T &s)
        {
            return s.count();
        }
        
        //! \brief Returns the const count for the sequence
        template<class T>
        static const std::size_t& count(const T &s)
        {
            return s.count();
        }
        
        //! \brief Returns the default DNA value for the sequence
        template<class T>
        static value_type default_value(const T &s)
        {
            return s.default_value();
        }
        
        //! \brief Returns the default DNA value for the sequence
        template<class T>
        static value_type default_quality(const T &s)
        {
            return s.default_quality();
        }
    };
    
    /*!
     * The general read class. A read is declared with a container, and optionally,
     * with a sequence of properties.
     *
     * Properties are stored in a tuple, and may be accessed with a <tt>get</tt> method.
     * Any type can be used (movable or non-movable).
     */
    template <class Container, class... Properties>
    class sequence
    {
    public:
        
        //! \brief General container type
        typedef Container container_type;
        
        //! \brief General value type
        typedef typename container_traits<Container>::value_type value_type;
        
        //! \brief General properties type
        typedef typename std::tuple<Properties...> properties_type;
        
        //! \brief Constructor
        sequence(Properties... args) : properties_(std::move(std::make_tuple(args...)))
        {
            //std::cout << "MOVE SEQ GENERAL " << typeid(*this).name() << std::endl;
        };
        
        sequence() : count_(0)
        {
            // NOP
        }
        
        //! \brief Returns the contained DNA sequence
        auto get() -> Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the const contained DNA sequence
        auto get() const -> const Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the properties as a whole
        auto properties() -> properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the const properties as a whole
        auto properties() const -> const properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns a property associated to a read
        template<std::size_t idx>
        auto property() -> typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns a const property associated to a read
        template<std::size_t idx>
        auto property() const -> const typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the count of the read
        std::size_t& count()
        {
            return count_;
        }
        
        //! \brief Returns the count of the read
        const std::size_t& count() const
        {
            return count_;
        }
        
        //! \brief The default value of a DNA sequence is unknown
        value_type default_value() const
        {
            return 'N';
        }
        
        //! \brief The default quality of a DNA sequence is 1/2
        value_type default_quality() const
        {
            return 'I';
        }
        
    private:
        
        //! \brief The container of the read (DNA sequence)
        Container sequence_;
        
        //! \brief Read properties
        properties_type properties_;
        
        //! \brief Count: this property may have different usages, for instance, counting reads
        std::size_t count_;
    };
    
    /*!
     * Partial specialization for one <b>movable</b> property: the header (as in FASTA).
     */
    template <class Container>
    class sequence<Container, Container>
    {
    public:
        
        //! \brief General container type
        typedef Container container_type;
        
        //! \brief General value type
        typedef typename container_traits<Container>::value_type value_type;
        
        //! \brief The header property
        typedef typename std::tuple<Container> properties_type;

        //! \brief Constructor
        sequence(Container&& s = Container(),
                 Container&& h = Container()) : sequence_(std::move(s)),
                                                properties_(std::move(std::make_tuple(std::move(h)))),
                                                count_(0)
        {
            Container q;
            
            container_traits<Container>::resize(q, container_traits<Container>::size(s), default_quality());
            
            quality_ = std::move(q);
        };

        //! \brief Returns the contained DNA sequence
        auto get() -> Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the const contained DNA sequence
        auto get() const -> const Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the properties as a whole
        auto properties() -> properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the const properties as a whole
        auto properties() const -> const properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the properties
        template<std::size_t idx>
        auto property() -> typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the const properties
        template<std::size_t idx>
        auto property() const -> const typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the header property
        auto header() -> Container&
        {
            return std::get<0>(properties_);
        }
        
        //! \brief Returns the const header property
        auto header() const -> const Container&
        {
            return std::get<0>(properties_);
        }
        
        //! \brief Returns the count of the read
        std::size_t& count()
        {
            return count_;
        }
        
        //! \brief Returns the const count of the read
        const std::size_t& count() const
        {
            return count_;
        }
        
        //! \brief Returns the header property
        auto quality() -> Container&
        {
            return quality_;
        }
        
        //! \brief Returns the const header property
        auto quality() const -> const Container&
        {
            return quality_;
        }
        
        //! \brief The default value of a DNA sequence is unknown
        value_type default_value() const
        {
            return 'N';
        }
        
        //! \brief The default quality of a DNA sequence is 1/2
        value_type default_quality() const
        {
            return 'I';
        }
        
    private:
        
        //! \brief The container of the read (DNA sequence)
        Container sequence_;

        //! \brief The container of the read's quality (dummy)
        Container quality_;
        
        //! \brief Read properties
        properties_type properties_;
        
        //! \brief The sequence count
        std::size_t count_;
    };
    
    /*!
     * Partial specialization for three <b>movable</b> properties: the header, optional, and quality (as in FASTQ)
     */
    template <class Container>
    class sequence<Container, Container, Container, Container>
    {
    public:
        
        //! \brief General container type
        typedef Container container_type;
        
        //! \brief General value type
        typedef typename container_traits<Container>::value_type value_type;
        
        //! \brief The properties type
        typedef typename std::tuple<Container, Container, Container> properties_type;
        
        //! \brief Constructor
        sequence(Container&& s = Container(), Container&& h = Container(),
                 Container&& o = Container(), Container&& q = Container()) : sequence_(std::move(s)),
                                                                             properties_(std::move(std::make_tuple(std::move(h), std::move(o), std::move(q)))),
                                                                             count_(0)
        {
            //std::cout << "MOVE READ THREE" << typeid(*this).name() << std::endl;
        };
        
        //! \brief Returns the contained DNA sequence
        auto get() -> Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the const contained DNA sequence
        auto get() const -> const Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the properties as a whole
        auto properties() -> properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the const properties as a whole
        auto properties() const -> const properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the properties
        template<std::size_t idx>
        auto property() -> typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the const properties
        template<std::size_t idx>
        auto property() const -> const typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the header property
        auto header() -> Container&
        {
            return std::get<0>(properties_);
        }
        
        //! \brief Returns the const header property
        auto header() const -> const Container&
        {
            return std::get<0>(properties_);
        }
        
        //! \brief Returns the optional field property
        auto optional() -> Container&
        {
            return std::get<1>(properties_);
        }
        
        //! \brief Returns the const optional field property
        auto optional() const -> const Container&
        {
            return std::get<1>(properties_);
        }
        
        //! \brief Returns the quality property
        auto quality() -> Container&
        {
            return std::get<2>(properties_);
        }
        
        //! \brief Returns the const quality property
        auto quality() const -> const Container&
        {
            return std::get<2>(properties_);
        }
        
        //! \brief Returns the count of the read
        std::size_t& count()
        {
            return count_;
        }
        
        //! \brief Returns the count of the read
        const std::size_t& count() const
        {
            return count_;
        }
        
        //! \brief The default value of a DNA sequence is unknown
        value_type default_value() const
        {
            return 'N';
        }
        
        //! \brief The default quality of a DNA sequence is 1/2
        value_type default_quality() const
        {
            return 'I';
        }
        
    private:
        
        //! \brief The container of the read (DNA sequence)
        Container sequence_;
        
        //! \brief Read properties
        properties_type properties_;
        
        //! \brief The sequence count
        std::size_t count_;
    };
    
    /*!
     * Partial specialization for forward-reverse matching and paired ends.
     */
    template <class Container, class Quality, class Reverse, class QualityFn>
    class sequence<Container, Quality, Reverse, QualityFn>
    {
    public:
        
        //! \brief General container type
        typedef Container container_type;
        
        //! \brief General value type
        typedef typename container_traits<Container>::value_type value_type;
        
        //! \brief The properties type
        typedef typename std::tuple<Container, Quality> properties_type;
        
        //! \brief The reverse function
        typedef Reverse reverse_function;
        
        //! \brief The quality function
        typedef QualityFn quality_function;

        //! \brief Constructor
        sequence(Container&& s = Container(), Quality&& q = Quality()) : sequence_(s),
                                                                         properties_(std::make_tuple(std::move(s), std::move(q))),
                                                                         count_(0)
        {
            Reverse fn;
            // Find the reverse
            std::get<0>(properties_) = fn(std::get<0>(properties_));
            
            // The representative of a sequence is the first lexicographically
            if (std::get<0>(properties_) < sequence_)
                std::swap(std::get<0>(properties_), sequence_);
        };
        
        //! \brief Returns the contained DNA sequence
        auto get() -> Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the const contained DNA sequence
        auto get() const -> const Container&
        {
            return sequence_;
        }
        
        //! \brief Returns the properties as a whole
        auto properties() -> properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the const properties as a whole
        auto properties() const -> const properties_type&
        {
            return properties_;
        }
        
        //! \brief Returns the properties
        template<std::size_t idx>
        auto property() -> typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the const properties
        template<std::size_t idx>
        auto property() const -> const typename std::tuple_element<idx, properties_type>::type&
        {
            return std::get<idx>(properties_);
        };
        
        //! \brief Returns the quality property
        auto quality() -> Quality&
        {
            return std::get<1>(properties_);
        }
        
        //! \brief Returns the const quality property
        auto quality() const -> const Quality&
        {
            return std::get<1>(properties_);
        }
        
        //! \brief Returns the quality property
        auto reverse() -> Container&
        {
            return std::get<0>(properties_);
        }
        
        //! \brief Returns the const quality property
        auto reverse() const -> const Container&
        {
            return std::get<0>(properties_);
        }

        //! \brief Returns the count of the read
        std::size_t& count()
        {
            return count_;
        }
        
        //! \brief Returns the count of the read
        const std::size_t& count() const
        {
            return count_;
        }
        
        //! \brief The default value of a DNA sequence is unknown
        value_type default_value() const
        {
            return 'N';
        }
        
        //! \brief The default quality of a DNA sequence is 1/2
        value_type default_quality() const
        {
            return 0.5;
        }
        
    private:
        
        //! \brief The container of the read (DNA sequence)
        Container sequence_;
        
        //! \brief Read properties
        properties_type properties_;
        
        //! \brief The sequence count
        std::size_t count_;
    };
    
    /*!
     * Handy alias for a FASTA read (one property) stored as a string.
     */
    using fasta_string = sequence<std::string, std::string>;
    
    /*!
     * Handy alias for a FASTQ read (three properties) stored as a string.
     */
    using fastq_string = sequence<std::string, std::string, std::string, std::string>;

    /*!
     * Handy alias for paired end and forward-reverse matching sequences
     */
    using paired_revfwd = sequence<std::string, float, details::reverse, details::quality_average>;
    
    /*!
     * Low memory sequences aliases, uses a raw bit_container type. USE WITH CAUTION.
     */
    using lmf_sequence10 = bit_container<10>;
    using lmf_sequence11 = bit_container<11>;
    using lmf_sequence12 = bit_container<12>;
    using lmf_sequence13 = bit_container<13>;
    using lmf_sequence14 = bit_container<14>;
    using lmf_sequence15 = bit_container<15>;
    using lmf_sequence16 = bit_container<16>;
    
    namespace details
    {
        
        /*!
         * Converts properties of sequences, copying tuples to their minimum length
         */
        template<class From, class To>
        void convert_property(From &from, To &to, std::size_t start, std::size_t length)
        {
            details::copy_tuple(from, to);
        }
        
        /*!
         * Converts properties from a FASTA to another FASTA, i.e., creates a copy
         */
        template<>
        void convert_property(fasta_string &from, fasta_string &to, std::size_t start, std::size_t length)
        {
            
            typedef typename sequence_traits<fasta_string>::properties_type property_type;
            
            // FASTA files have one proerty
            to.header() = from.header();
        }
        
        /*!
         * Converts properties from a FASTQ to another FASTQ, i.e., creates a copy with partial quality
         */
        template<>
        void convert_property(fastq_string &from, fastq_string &to, std::size_t start, std::size_t length)
        {
            typedef typename sequence_traits<fastq_string>::properties_type property_type;
            
            // Copy header and optional
            to.header()   = from.header();
            to.optional() = from.optional();
            
            container_traits<fasta_string>::resize(to.quality(), length);
            
            auto x = from.quality();
            auto y = to.quality();
            
            // Copy quality
            for(std::size_t i = 0; length > 0; length--, start++, i++)
            {
                to.quality().at(i) = from.quality().at(start);
            }
        }
        
        /*!
         * Converts properties from a FASTA to a FASTQ, i.e., creates a copy with dummy quality
         */
        template<>
        void convert_property(fasta_string &from, fastq_string &to, std::size_t start, std::size_t length)
        {
            typedef typename sequence_traits<fastq_string>::properties_type property_type;
            
            // Copy header and optional
            to.header() = from.header();
            
            container_traits<fasta_string>::resize(to.quality(), length, sequence_traits<fastq_string>::default_quality(to));
        }
        
        /*!
         * Converts properties from a FASTQ to a FASTA, i.e., discards all properties except header
         */
        template<>
        void convert_property(fastq_string &from, fasta_string &to, std::size_t start, std::size_t length)
        {
            typedef typename sequence_traits<fastq_string>::properties_type property_type;
            
            // Copy header and optional
            to.header()   = from.header();
        }
        
        /*!
         * Converts properties from a FASTA to a Forward-reverse Paired ended sequence
         */
        template<>
        void convert_property(fasta_string &from, paired_revfwd &to, std::size_t start, std::size_t length)
        {
            typedef typename sequence_traits<fasta_string>::properties_type property_type;
            
            // Copy header and optional
            to.quality() = to.default_quality();
        }

        /*!
         * Converts properties from a FASTQ to a Forward-reverse Paired ended sequence
         */
        template<>
        void convert_property(fastq_string &from, paired_revfwd &to, std::size_t start, std::size_t length)
        {
            typedef typename sequence_traits<fasta_string>::properties_type property_type;
            
            // Quality characters: ! is the min, ~ is the max, they are contiguous, similar to the following:
            // const std::string q = "!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~";
            //                indexes 012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345
            //                                 10        20        30        40        50        60        70        80        90

            typename paired_revfwd::container_type c;
            
            typename paired_revfwd::quality_function fn;
            
            container_traits<paired_revfwd>::resize(c, length);
            
            std::size_t l = 0;
            
            for (; length-- > 0; start++)
            {
                container_traits<paired_revfwd>::at(c, l++) = container_traits<paired_revfwd>::at(from.quality(), start);
            }
            to.quality() = fn(c);
        }

        /*!
         * Prints a property to a stream by recursion
         */
        template <class Stream, class Tuple, std::size_t N>
        class property_printer
        {
        public:
            
            static void print(Stream &s, const Tuple &t)
            {
                property_printer<Stream, Tuple, N - 1>::print(s, t);
                s << ", " << std::get<N - 1>(t);
            }
        };

        template <class Stream, class Tuple>
        class property_printer<Stream, Tuple, 1>
        {
        public:
            
            static void print(Stream &s, const Tuple &t)
            {
                s << std::get<0>(t);
            }
        };
        
        //! \brief This function is used for internal purposes. It prints all properties to a stream
        template <class Stream, class... Args>
        void dump_properties(Stream &s, const std::tuple<Args...> &t)
        {
            property_printer<Stream, decltype(t), sizeof...(Args)>::print(s, t);
        }

    }
}

#endif
