/*!
 *  \file reader.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_readers_h
#define sequence_readers_h

#include <boost/circular_buffer.hpp>
#include <array>
#include <cstring>
#include <functional>
#include <future>
#include <memory>
#include "source.h"
#include <stdexcept>
#include <type_traits>
#include <vector>

namespace seq
{
    template<typename Reader>
    class reader_traits
    {
        public:
            typedef Reader                               reader_type;
            typedef typename reader_type::value_type     value_type;
            typedef typename reader_type::position_type  position_type;
            typedef typename reader_type::allocator_type allocator_type;

            static_assert (std::is_move_constructible<reader_type>::value,
                           "A reader must be move contructible!");
            static_assert (!std::is_copy_constructible<reader_type>::value,
                           "A reader cannot be copy constructible!");
            static_assert (std::is_move_assignable<reader_type>::value,
                           "A reader must be move assignable!");
            static_assert (!std::is_copy_assignable<reader_type>::value,
                           "A reader cannot be copy assignable!");
            static_assert (std::is_destructible<reader_type>::value,
                           "A reader must be destructible!");
            static_assert (std::is_integral<position_type>::value,
                           "The position type must be an integral type!");

            static void open (reader_type & r, const char * filename) {
                r.open (filename);
            }

            static value_type get (reader_type & r, position_type pos) {
                return r.get (pos);
            }
    };

    class reader_error : public std::runtime_error
    {
        public:
            using std::runtime_error::runtime_error;
    };

    template<typename Allocator, typename Source>
    class basic_reader
    {
        public:
            typedef Source                     source_type;
            typedef source_traits<source_type> source_traits;
            typedef char                       value_type;
            typedef unsigned long              position_type;

            typedef Allocator                             allocator_type;
            typedef std::allocator_traits<allocator_type> allocator_traits;

        private:
            allocator_type allocator_;
            source_type source_;
            std::array<char, 4096> buffer_;

        public:
            void open (const char * filename) {
                source_traits::open (source_, filename);
            }

            value_type get (position_type pos) {
                auto real_pos = pos % buffer_.size ();

                if (real_pos == 0) {
                    if (source_traits::good (source_)) {
                        auto p_buffer = buffer_.data ();

                        source_traits::read (source_, &p_buffer,  4096);
                        auto gc = source_traits::gcount (source_);
                        if (gc != 4096) {
                            memset(&p_buffer[gc], '\0', 4096 - gc);
                        }
                    }
                }

                return buffer_[real_pos];
            }
    };

    template<typename Source>
    class basic_reader<void, Source>
    {
        public:
            typedef Source                     source_type;
            typedef source_traits<source_type> source_traits;
            typedef char                       value_type;
            typedef unsigned long              position_type;
            typedef void                       allocator_type;

        private:
            source_type source_;
            value_type * p_buffer_ = nullptr;
            typename source_type::size_type size_;
            position_type index_ { 0 };

        public:
            void open (const char * filename) {
                source_traits::open (source_, filename);
            }

            value_type get (position_type pos) {
                auto real_pos = pos % 4096;

                if (real_pos == 0) {
                    if (source_traits::good (source_)) {
                        source_traits::read (source_, &p_buffer_,  4096);
                        auto gc = source_traits::gcount (source_);
                        if (gc != 4096) {
                            memset(&p_buffer_[gc], '\0', 4096 - gc);
                        }
                    }
                }

                return p_buffer_[real_pos];
            }
    };

    template<typename Source>
    using allocating_reader = basic_reader<std::allocator<char>, Source>;

    template<typename Source>
    using non_allocating_reader = basic_reader<void, Source>;
}

#endif
