/*!
 *  \file common.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */


#ifndef sequence_common_h
#define sequence_common_h

#include <algorithm>
#include <iostream>
#include <bitset>
#include <stdexcept>
#include <unordered_map>

namespace seq
{
    //! \brief  Logging level: if the internal log is turned on, then functions will output
    static bool logging = true;
    
    /*!
     * \brief A type switch class, defines a type <tt>if_false</tt> if a condition is false
     */
    template <bool, class if_true, class if_false>
    struct type_switch
    {
        typedef if_false type;
    };
    
    /*!
     * \brief Template specialization defining <tt>if_true</tt> if a condition is true
     */
    template <class if_true, class if_false>
    struct type_switch<true, if_true, if_false>
    {
        typedef if_true type;
    };

    
    /*!
     * \brief A small memory footprint compile-time container
     */
    template <unsigned int max_size>
    class bit_container
    {
    public:
        
        //! \brief Storage type: for k-mer size <= 16, use a 32 bit unsigned integer, otherwise, a std::bitset
        typedef typename type_switch<(max_size <= 16), uint32_t, std::bitset<max_size * 2>>::type storage_type;
        
        //! \brief Constructor always sets bits to zero
        bit_container()
        {
            storage_ = 0;
        }
        
        //! \brief Returns a reference to the actual storage
        inline storage_type& get()
        {
            return storage_;
        }
        
        //! \brief Returns a reference to the actual storage
        inline const storage_type& get() const
        {
            return storage_;
        }
        
        //! \brief Returns the size of the kmer
        inline unsigned int size() const
        {
            return size_;
        }

        //! \brief Returns the size of the kmer
        inline void setsize(unsigned int newsize)
        {
            if (newsize > maxsize_) throw std::domain_error("This bit_container class cannot handle sizes exceeding " + std::to_string(maxsize_) + ".");;
            size_ = newsize;
        }

        //! \brief Returns the size of the (constant) kmer
        inline unsigned int maxsize() const
        {
            return maxsize_;
        }

        //! \brief Clean traliling bits
        inline void clean()
        {
            // Clean up trailing bits
            if (size_ < 16)
            {
                storage_type q = (~0) << (size() << 1);
                
                storage_ = storage_ & (~q);
            }
        }

        //! \brief A as a bit sequence
        static const char A = 0;

        //! \brief A as a bit sequence
        static const char T = 3;

        //! \brief A as a bit sequence
        static const char C = 1;

        //! \brief A as a bit sequence
        static const char G = 2;

    private:
        
        //! \brief Storage type, a 32 bit integer, or a bitset
        storage_type storage_;
        
        //! \brief Size of container, static
        static unsigned int size_;
        
        //! \brief Max size of the container
        static unsigned int maxsize_;
    };

    /*!
     * A container traits class: the containers used in the library definitions must be movable.
     */
    template <class Container>
    class container_traits
    {
    public:
        
        //! \brief Container value, valid for standard containers and strings.
        typedef typename Container::value_type value_type;
        
        static_assert (std::is_move_constructible<value_type>::value,
                       "A container value must be move contructible!");
        static_assert (std::is_move_assignable<value_type>::value,
                       "A container value be move assignable!");
        static_assert (std::is_move_constructible<Container>::value,
                       "A container must be move contructible!");
        static_assert (std::is_move_assignable<Container>::value,
                       "A container be move assignable!");
        
        //! \brief Returns the size of a container
        template <class T>
        static std::size_t size(const T& c)
        {
            return c.size();
        }
        
        //! \brief Access an element of a container
        template <class T>
        static value_type& at(T& c, std::size_t i)
        {
            return c.at(i);
        }

        //! \brief Access an element of a container
        template <class T>
        static const value_type& at(const T& c, std::size_t i)
        {
            return c.at(i);
        }

        //! \brief Access an element of a container
        template <class T>
        static value_type& at(T& c, value_type i)
        {
            return c.at(i);
        }

        //! \brief Reserve memory of a container
        template <class T>
        static void reserve(T& c, std::size_t size)
        {
            c.reserve(size);
        }
        
        //! \brief Resizes a container
        template <class T>
        static void resize(T& c, std::size_t size)
        {
            c.resize(size);
        }
        
        //! \brief Resizes a container with a constant value
        template <class T>
        static void resize(T& c, std::size_t size, value_type v)
        {
            c.resize(size, v);
        }
        
        //! \brief Sets the container's content from a string
        template <class T>
        static void fromString(T& c, const std::string &s, int start = -1, int length = -1)
        {
            c.resize(s.size());
            
            if (start < 0 || length < 0)
                std::copy(s.begin(), s.end(), c.begin());
            else
                std::copy(s.begin() + start, s.begin() + length, c.begin());
        }
        
        //! \brief Returns the container's content as a string
        template <class T>
        static std::string toString(T& c)
        {
            std::string s;
            
            s.resize(c.size());
            
            std::copy(c.begin(), c.end(), s.begin());
            
            return s;
        }
    };
    
    /*!
     * Partial specialization for bit container
     */
    template <int maxSize>
    class container_traits<bit_container<maxSize>>
    {
    public:
        
        //! \brief Returns the size of a container
        template <class T>
        static std::size_t size(const T& c)
        {
            return c.size();
        }
        
        //! \brief Sets the container's content from a string
        template <class T>
        static void fromString(T& c, const std::string &s, int start = -1, int length = -1)
        {
            typename bit_container<maxSize>::storage_type v;
            
            c.get() = 0;
            
            // Update start and length
            if (start < 0 || length < 0)
            {
                start =  0;
                length = s.size();
            }
            
            // Convert to binary
            for (int i = start; i < length; i++)
            {
                switch (s[i])
                {
                    // A
                    case 'A':
                        v = bit_container<maxSize>::A;
                        break;
                        
                    // T
                    case 'T':
                        v = bit_container<maxSize>::T;
                        break;
                        
                    // C
                    case 'C':
                        v = bit_container<maxSize>::C;
                        break;
                        
                    // G
                    case 'G':
                        v = bit_container<maxSize>::G;
                        break;
                        
                    default:
                        throw std::domain_error("The bit_container class cannot handle DNA with invalid or uncertain aminoacids.");
                }
                
                c.get() = c.get() | (v << (i << 1));
            }
            
            c.clean();
        }
        
        //! \brief Returns the container's content as a string
        template <class T>
        static std::string toString(T& c)
        {
            std::string b;
            
            typename bit_container<maxSize>::storage_type shl, shr, q;

            // Convert to binary
            for (int i = 0; i < c.size(); i++)
            {
                shl = (3 << (i << 1));
                shr = i << 1;
                q   = c.get() & shl;
                q   = q >> shr;
                
                unsigned char v = static_cast<char>(q);
                
                switch (v)
                {
                    // A
                    case bit_container<maxSize>::A:
                        b += 'A';
                        break;
                        
                    // T
                    case bit_container<maxSize>::T:
                        b += 'T';
                        break;
                        
                    // C
                    case bit_container<maxSize>::C:
                        b += 'C';
                        break;
                        
                    // G
                    case bit_container<maxSize>::G:
                        b += 'G';
                        break;
                        
                    default:
                        throw std::runtime_error("Something really bad happened in converting a bit_container to a string.");
                }
            }
            
            return b;
        }
    };
    
    /*!
     * Private namespace for unimportant details (from a user's point of view)
     */
    namespace details
    {
#if __cplusplus >= 201402L
        /*!
         * Private implementation of copying a tuple for C++14
         */
        template <std::size_t ...I, typename T1, typename T2>
        void copy_tuple_impl(T1 const & from, T2 & to, std::index_sequence<I...>)
        {
            int dummy[] = { (std::get<I>(to) = std::get<I>(from), 0)... };
            static_cast<void>(dummy);
        }
        
        template <typename T1, typename T2>
        void copy_tuple(T1 const & from, T2 & to)
        {
            copy_tuple_impl(
                            from, to,
                            std::make_index_sequence<std::tuple_size<T1>::value>());
        }
#else
        /*!
         * Private implementation of copying a tuple for C++11
         */
        template<std::size_t I = 0, typename ...From, typename ...To>
        auto copy_tuple(std::tuple<From...> const & from, std::tuple<To...> & to) -> typename std::enable_if<(I >= sizeof...(From) || I >= sizeof...(To))>::type
        {
            // NOP
        }
        
        template<std::size_t I = 0, typename ...From, typename ...To>
        auto copy_tuple(std::tuple<From...> const & from, std::tuple<To...> & to) -> typename std::enable_if<(I < sizeof...(From) && I < sizeof...(To))>::type
        {
            std::get<I>(to) = std::get<I>(from);
            copy_tuple<I + 1>(from,to);
        }
#endif
        
        //! \brief Mapping between a DNA aminoacid and its reverse
        static const std::unordered_map<char, char> fwd_rev_map
        {
            {'A', 'T'},
            {'C', 'G'},
            {'G', 'C'},
            {'T', 'A'},
            {'U', 'A'},
            {'R', 'Y'},
            {'Y', 'R'},
            {'K', 'M'},
            {'M', 'K'},
            {'S', 'S'},
            {'W', 'W'},
            {'B', 'V'},
            {'D', 'H'},
            {'H', 'D'},
            {'V', 'B'},
            {'N', 'N'}
        };
        
        /*!
         * This functor is for <b>tests only</b>. Do not use in production.
         */
        class id
        {
        public:
            
            template <class Container>
            Container operator()(const Container &s)
            {
                return Container(s);
            }
        };
        
        /*!
         * This functor returns the reverse sequence.
         */
        class reverse
        {
        public:
            
            template <class Container>
            Container operator()(const Container &s)
            {
                Container t;
                
                container_traits<Container>::resize(t, container_traits<Container>::size(s));
                
                std::copy(s.begin(), s.end(), t.begin());
                std::reverse(t.begin(), t.end());
//                std::transform(t.begin(), t.end(), t.begin(),
//                               [](auto p)
//                               {
//                                   std::cout << "KEY " << p << std::endl;
//                                   std::cout << "    -> " << fwd_rev_map.at(p) << std::endl;
//                                   return fwd_rev_map.at(p);
//                               });
                for(auto &p : t)
                {
                    std::cout << "KEY " << p << std::endl;
                    p = fwd_rev_map.at(p);
                }

                return t;
            }
            
            template <unsigned int maxSize>
            bit_container<maxSize> operator()(const bit_container<maxSize> &s)
            {
                bit_container<maxSize> t;
                
                typename bit_container<maxSize>::storage_type shl, shr, q;

                // Invert 2-bits blocks
                for (int i = 0; i < s.size(); i++)
                {
                    shl = (3 << (i << 1));
                    shr = i << 1;
                    q   = s.get() & shl;
                    q   = q >> shr;
                    
                    unsigned char c = static_cast<char>(q);
                    
                    shl = (s.size() - i - 1) << 1;
                    
                    t.get() = t.get() | (c << shl);
                }
                
                // Invert aminoacids
                t.get() = ~t.get();
                
                t.clean();

                return t;
            }
        };

        /*!
         * This functor returns the minimum of quality scores.
         */
        class quality_minimum
        {
        public:
            
            template <class Container>
            float operator()(const Container &s)
            {
                float q = 1.0;
                
                for (const auto &c : s)
                {
                    q = std::min(q, static_cast<float>(c - '!' + 1) / static_cast<float>('~' - '!' + 1));
                }
                return q;
            }
        };

        /*!
         * This functor returns the maximum of quality scores.
         */
        class quality_maximum
        {
        public:
            
            template <class Container>
            float operator()(const Container &s)
            {
                float q = 0.0;
                
                for (const auto &c : s)
                {
                    q = std::max(q, static_cast<float>(c - '!' + 1) / static_cast<float>('~' - '!' + 1));
                }
                return q;
            }
        };

        /*!
         * This functor returns the minimum of quality scores.
         */
        class quality_average
        {
        public:
            
            template <class Container>
            float operator()(const Container &s)
            {
                int   n = 0;
                float q = 0.0;
                
                for (const auto &c : s)
                {
                    q += static_cast<float>(c - '!') / static_cast<float>('~' - '!');
                    n++;
                }
                return static_cast<float>(q) / static_cast<float>(n);
            }
        };

    }
}

#endif
