/*!
 *  \file algorithm.h
 *
 *  \copyright Copyright (c) 2014 Franco "Sensei" Milicchio. All rights reserved.
 *
 *  \license BSD Licensed.
 */

#ifndef sequence_algorithm_h
#define sequence_algorithm_h

#include <tbb/pipeline.h>

#include "timer.h"
#include "parser.h"

namespace seq
{
    //! \brief This function applies a functor (concurrently) to all sequences in a file
    template <class Parser, class Fn, class... Args>
    void concurrent_for_each(Parser &p, std::string log, Fn f, Args... args)
    {
        typedef typename parser_traits<Parser>::sequence_type sequence_type;
        
        timer timer;
        
        if (logging) timer.start(log);
        
        tbb::parallel_pipeline(30,
                               tbb::make_filter<void, sequence_type>(tbb::filter::serial,
                                                                     [&](tbb::flow_control& fc)
                                                                     {
                                                                         sequence_type s = p.next();
                                                                         
                                                                         if (s.get().empty())
                                                                             fc.stop();
                                                                         return s;
                                                                     })
                               &
                               tbb::make_filter<sequence_type, void>(tbb::filter::parallel,
                                                                     [&](seq::fastq_string s)
                                                                     {
                                                                         f(s, args...);
                                                                     })
                               );
        
        if (logging) timer.stop();
    }
}

#endif
