//
//  unittests.cpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//

#define CATCH_CONFIG_RUNNER

#include <catch2/catch.hpp>

#include <iostream>
#include <string>
#include <array>
#include <unordered_map>
#include <limits>
#include <vector>
#include <fstream>

#include "timer.hpp"
#include "logger.hpp"
#include "path.hpp"
#include "fastq_mmap.hpp"
#include "precomputed.hpp"

#include "accelerator_string.hpp"
#include "accelerator_uint128.hpp"
#include "accelerator_sse.hpp"

#include "file_std.hpp"
#include "file_mmap.hpp"

#include <x86intrin.h>

struct listener : Catch::TestEventListenerBase
{
    using TestEventListenerBase::TestEventListenerBase;

    virtual void testCaseStarting(Catch::TestCaseInfo const& testInfo) override
    {
        std::cout << testInfo.tagsAsString() << " " << testInfo.name << std::endl;
    }
};
CATCH_REGISTER_LISTENER(listener)


/****************************************************************************
 * Test variables ***********************************************************
 ****************************************************************************/

std::string testfiles_dir;

/****************************************************************************
 * Basic classes ************************************************************
 ****************************************************************************/

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "File name consistency", "[Base]" )
{
    std::string  s("/bin/fake");
    libseq::path p(s);
    std::string  r = p.file_name();
    REQUIRE( r == "fake" );
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "File path consistency", "[Base]" )
{
    std::string  s("/bin/fake");
    libseq::path p(s);
    std::string  r = p.file_path();
    REQUIRE( r == "/bin" );
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "File path is a directory", "[Base]" )
{
    std::string  s("/");
    libseq::path p(s);
    REQUIRE( p.is_directory() == true );
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "File path is a file", "[Base]" )
{
    std::string  s("/bin/ls");
    libseq::path p(s);
    REQUIRE( p.is_file() == true );
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "File path append", "[Base]" )
{
    std::string  s("/usr/local/bin");
    libseq::path p(s);
    auto         q = p.append("fake");
    REQUIRE( q.absolute_path() == "/usr/local/bin/fake" );
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "Timer class, non-throwing", "[Base]" )
{
    // Temporary double, elapsed time doing less and more work
    double tmp = 0.0, time_less, time_more;
    
    // Start a timer
    libseq::timer t;
    
    // Do something
    for (double i = 1.0; i < 100.0; i += 1.0) tmp += i;

    time_less = t.reset<std::chrono::nanoseconds>();
    
    // Do something more
    for (double i = 1.0; i < 100000.0; i += 1.0) tmp += i;
    
    time_more = t.elapsed<std::chrono::nanoseconds>();

    // Should reach this line
    REQUIRE(true);
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "Logger class", "[Base]" )
{
    libseq::logger::init("./dumpster.log");
    
    libseq::logger::info("This should be logged everywhere.");
    
    // Temporary double, elapsed time
    double tmp = 0.0, time;
    
    // Start a timer
    libseq::timer t;
    
    // Do something
    for (double i = 1.0; i < 10000.0; i += 1.0) tmp += i;
    
    time = t.elapsed<std::chrono::nanoseconds>();

    libseq::logger::debug("This should be logged in the file only in release mode, useless loop took {} nanoseconds.", time);
    
    // Should reach this line
    REQUIRE(true);
}

/****************************************************************************
 * Reader classes ***********************************************************
 ****************************************************************************/

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "FASTQ mmaped iterator with a while loop ()", "[Readers]" )
{
    REQUIRE(!testfiles_dir.empty());
    
    libseq::path dir(testfiles_dir);
    
    // Test fastq files information
    const int   tests   = 3;
    const char* names[] = { "SRR000000.fastq", "SRR000001.fastq", "SRR000002.fastq" };
    const int   reads[] = {                 1,                 5,             10000 };
    
    int count = 0;
    
    for (int i = 0; i < tests; ++i)
    {
        std::string f = dir.append(names[i]), e = ".fastq";
        
        libseq::fastq_mmap fastq(f);
        
        // Iterators
        auto b = fastq.begin();
        auto q = fastq.end();
        
        count = 0;
        
        // Test with iterators (get the header property)
        while (b != q)
        {
            auto tmp = std::get<0>(b.properties());
            ++b; ++count;
            
            // Don't complain about unused variable
            tmp.size();
        }
        REQUIRE(count == reads[i]);
    }
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "FASTQ mmaped iterator with a for loop ()", "[Readers]" )
{
    REQUIRE(!testfiles_dir.empty());
    
    libseq::path dir(testfiles_dir);
    
    // Test fastq files information
    const int   tests   = 3;
    const char* names[] = { "SRR000000.fastq", "SRR000001.fastq", "SRR000002.fastq" };
    const int   reads[] = {                 1,                 5,             10000 };
    
    int count = 0;
    
    for (int i = 0; i < tests; ++i)
    {
        std::string f = dir.append(names[i]), e = ".fastq";

        libseq::fastq_mmap fastq(f);
                
        count = 0;
        
        // Test with range (just get the reads)
        for (auto &p : fastq)
        {
            count++;
            
            // Don't complain about unused variable
            p.size();
        }

        REQUIRE(count == reads[i]);
    }
}

/****************************************************************************
 * Intrinsics ***************************************************************
 ****************************************************************************/

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "Intel SSE", "[Intrinsics]" )
{
    // Test AVX
    __m128i a, b, c, d;
    
    const unsigned char s[] =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, //  64
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff  // 128
    };
    
    const unsigned char p[] =
    {
        0x11, 0x22, 0x33, 0x44, 0xA0, 0x00, 0x00, 0x0A, //  64
        0x03, 0x00, 0x00, 0x40, 0xB0, 0x00, 0x00, 0x0B  // 128
    };
    
    a = _mm_load_si128 ((__m128i*)(s));
    b = _mm_lddqu_si128((__m128i*)(p));
    
    c = _mm_slli_si128(a, 1);
    d = _mm_slli_si128(b, 1);
        
    REQUIRE(true);
}

/****************************************************************************
 * Accelerator classes ******************************************************
 ****************************************************************************/

////////////////////////////////////////////////////////////////////////////////
std::string random_string( size_t length )
{
    // I'm lazy for this, see: https://stackoverflow.com/questions/440133/how-do-i-create-a-random-alpha-numeric-string-in-c
    auto randchar = []() -> char
    {
        const char charset[] = "ATCG";
        const size_t max_index = (sizeof(charset) - 1);
        return charset[ rand() % max_index ];
    };
    std::string str(length,0);
    std::generate_n( str.begin(), length, randchar );
    return str;
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "String accelerator kmerizer", "[Kmerizers]" )
{
    std::size_t failed = 0, i, k;
    
    // Test string and substring
    std::string t = random_string(10000), s;
    
    //Accelerator
    libseq::accelerator_string acc;
    
    // Test every kmer length
    for (k = 33; k <= 64; k++)
    {
        std::size_t nsubstrings = t.length() - k + 1;
        
        // For every kmer
        for (i = 0; i < nsubstrings; i++)
        {
            auto kmer = std::string_view(t.data() + i, k);
            
            auto from = acc.to_forward(kmer);
            auto to   = acc.to_string(from, k);
            
            if (t.substr(i, k) != to)
                failed++;
        }
    }
    
    REQUIRE(failed == 0);
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "Integer accelerator kmerizer", "[Kmerizers]" )
{
    std::size_t failed = 0, i, k;
    
    // Test string and substring
    std::string t = random_string(10000), s;
    
    //Accelerator
    libseq::accelerator_uint128 acc;
    
    // Test every kmer length
    for (k = 33; k <= 64; k++)
    {
        std::size_t nsubstrings = t.length() - k + 1;
        
        // For every kmer
        for (i = 0; i < nsubstrings; i++)
        {
            auto kmer = std::string_view(t.data() + i, k);
            
            auto from = acc.to_forward(kmer);
            auto to   = acc.to_string(from, k);
            
            if (t.substr(i, k) != to)
                failed++;
        }
    }
    
    REQUIRE(failed == 0);
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "SSE accelerator kmerizer", "[Kmerizers]" )
{
    std::size_t failed = 0, i, k;
    
    // Test string and substring
    std::string t = random_string(10000), s;
    
    //Accelerator
    libseq::accelerator_sse acc;
    
    // Test every kmer length
    for (k = 33; k <= 64; k++)
    {
        std::size_t nsubstrings = t.length() - k + 1;
        
        // For every kmer
        for (i = 0; i < nsubstrings; i++)
        {
            auto kmer = std::string_view(t.data() + i, k);
            
            auto from = acc.to_forward(kmer);
            auto to   = acc.to_string(from, k);

            if (t.substr(i, k) != to)
                failed++;
        }
    }
    
    REQUIRE(failed == 0);
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE( "SSE accelerator revcomp", "[Kmerizers]" )
{
    std::size_t failed = 0, i, k;
    
    // Test string and substring
    std::string t = random_string(10000), s;
    
    //Accelerator
    libseq::accelerator_sse acc;
    
    // Test every kmer length
    for (k = 33; k <= 64; k++)
    {
        std::size_t nsubstrings = t.length() - k + 1;
        
        // For every kmer
        for (i = 0; i < nsubstrings; i++)
        {
            auto kmer = std::string_view(t.data() + i, k);
            
            auto from = acc.to_revcomp(kmer);
            auto to   = acc.to_string(from, k);
            
            std::string revcomp;
            
            for (char q : kmer)
            {
                switch (q)
                {
                    case 'A':
                        revcomp = std::string("T") + revcomp;
                        break;
                        
                    case 'T':
                        revcomp = std::string("A") + revcomp;
                        break;
                        
                    case 'C':
                        revcomp = std::string("G") + revcomp;
                        break;
                        
                    case 'G':
                        revcomp = std::string("C") + revcomp;
                        break;
                        
                    default:
                        throw std::domain_error("WAT");
                }
            }

            if (revcomp != to)
                failed++;
        }
    }
    
    REQUIRE(failed == 0);
}

////////////////////////////////////////////////////////////////////////////////
template <typename T>
void read_from_file(std::string path, std::vector<T>& data, std::size_t n)
{
    std::ifstream file(path, std::ios::binary);
    file.read((char*) data.data(), n * sizeof(T));
    
    file.close();
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE("Std file 1 write", "[File]")
{
    return;
    __m128i a;
    
    const unsigned char s[] =
    {
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, //  64
    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff  // 128
    };
    
    a = _mm_load_si128 ((__m128i*)(s));
    
    libseq::path path(testfiles_dir);
    path = path.append("file_std.bin");
    libseq::file_std file(path);
    
    file.append(a);
    file.close();
    
    std::vector<__m128i> data(1);
    read_from_file(path.absolute_path(), data, 1);
    
    REQUIRE(_mm_test_all_ones(_mm_cmpeq_epi8(a, data[0])));
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE("Mmap file 1 write", "[File]")
{
    return;
    __m128i a;
    
    const unsigned char s[] =
    {
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, //  64
    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff  // 128
    };
    
    a = _mm_load_si128 ((__m128i*)(s));
    
    libseq::path path(testfiles_dir);
    path = path.append("file_mmap.bin");
    libseq::file_mmap file(path);
    
    file.append(a);
    file.close();
    
    std::vector<__m128i> data(1);
    read_from_file(path.absolute_path(), data, 1);
    
    REQUIRE(_mm_test_all_ones(_mm_cmpeq_epi8(a, data[0])));
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE("Std file 2 write", "[File]")
{
    return;
    __m128i a, b;
    
    const unsigned char s[] =
    {
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, //  64
    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff  // 128
    };
    
    const unsigned char p[] =
    {
    0x11, 0x22, 0x33, 0x44, 0xA0, 0x00, 0x00, 0x0A, //  64
    0x03, 0x00, 0x00, 0x40, 0xB0, 0x00, 0x00, 0x0B  // 128
    };
    
    a = _mm_load_si128 ((__m128i*)(s));
    b = _mm_load_si128 ((__m128i*)(p));
    
    libseq::path path(testfiles_dir);
    path = path.append("file_std.bin");
    libseq::file_std file(path);
    
    file.append(a);
    file.append(b);
    file.close();
    
    std::vector<__m128i> data(2);
    read_from_file(path.absolute_path(), data, 2);
    
    REQUIRE(_mm_test_all_ones(_mm_cmpeq_epi8(a, data[0])));
    REQUIRE(_mm_test_all_ones(_mm_cmpeq_epi8(b, data[1])));
}

////////////////////////////////////////////////////////////////////////////////
TEST_CASE("Mmap file 2 write", "[File]")
{
    return;
    __m128i a, b;
    
    const unsigned char s[] =
    {
    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, //  64
    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff  // 128
    };
    
    const unsigned char p[] =
    {
    0x11, 0x22, 0x33, 0x44, 0xA0, 0x00, 0x00, 0x0A, //  64
    0x03, 0x00, 0x00, 0x40, 0xB0, 0x00, 0x00, 0x0B  // 128
    };
    
    a = _mm_load_si128 ((__m128i*)(s));
    b = _mm_load_si128 ((__m128i*)(p));
    
    libseq::path path(testfiles_dir);
    path = path.append("file_mmap.bin");
    libseq::file_mmap file(path);
    
    file.append(a);
    file.append(b);
    file.close();
    
    std::vector<__m128i> data(2);
    read_from_file(path.absolute_path(), data, 2);
    
    REQUIRE(_mm_test_all_ones(_mm_cmpeq_epi8(a, data[0])));
    REQUIRE(_mm_test_all_ones(_mm_cmpeq_epi8(b, data[1])));
}

/****************************************************************************
 * Run tests ****************************************************************
 ****************************************************************************/

int main( int argc, char* argv[] )
{
    Catch::Session session;
    
    using namespace Catch::clara;
    
    auto cli = session.cli() |
               Opt( testfiles_dir, "dir" ) ["--test-dir"] ("specify the directory containing the test dna sequences files");
    
    session.cli(cli);
    
    int returnCode = session.applyCommandLine(argc, argv);
    
    if( returnCode != 0 ) return returnCode;
    
    return session.run();
}


