//
//  veboas.hpp
//  Copyright © 2021 Franco Milicchio. All rights reserved.
//

#ifndef veboas_hpp
#define veboas_hpp

#include <vector>
#include <iostream>
#include "accelerator.hpp"
#include <x86intrin.h>


namespace libseq
{
    /// This is a helper class used to store Van Emde Boas indexes
    class vebindices
    {
    public:
        /// Pre-computed indices for all trees
        static const std::size_t indices_[8192];
        
        /// Pre-computed offsets for indices for accessing indexes given a height
        static const std::size_t offsets_[12];
    };
    
    /// This class implements a complete BST with Van Emde Boas tree layout
    template <typename Accelerator, typename FreqType = u_int32_t>
    class vebbsttree
    {
        /// Storage type for kmer and frequency
        class KmerCount
        {
        public:
            typename Accelerator::storage_type kmer;
            FreqType count;
        };
        
        void print128_num(__m128i var)
        {
            uint16_t val[8];
            memcpy(val, &var, sizeof(val));
            printf("%02x%02x%02x%02x%02x%02x%02x%02x",
                   val[0], val[1], val[2], val[3], val[4], val[5],
                   val[6], val[7]);
        }
    public:
        
        /// Creates a full complete binary tree with Van Emde Boas layout, with a given height
        vebbsttree(std::size_t height) : h_(height), maxindex_((1 << h_) - 1)
        {
            // Allocate sufficient elements for a complete binary tree
            data_ = new KmerCount[maxindex_];
        }
        
        /// Deallocates all memory
        ~vebbsttree()
        {
            delete[] data_;
        }
        
        std::size_t minindex() const
        {
            return 1;
        }
        
        std::size_t maxindex() const
        {
            return maxindex_;
        }
        
        /// Access elements by 1-based BFS indexes (i.e., only root has index 1)
        KmerCount& operator[](std::size_t idx)
        {
            if (idx > maxindex_ || idx == 0) throw std::domain_error("Index of VEB tree out of range.");
            
            return data_[bfs2veb(idx)];
        }
        
        /// Access const elements by 1-based BFS indexes (i.e., only root has index 1)
        const KmerCount& operator[](std::size_t idx) const
        {
            if (idx > maxindex_ || idx == 0) throw std::domain_error("Index of VEB tree out of range.");
            
            return data_[bfs2veb(idx)];
        }
        
        /// Return the left tree node index
        inline std::size_t index_left(std::size_t idx) const
        {
            return idx << 1;
        }

        /// Return the right tree node index
        inline std::size_t index_right(std::size_t idx) const
        {
            return (idx << 1) | 1;
        }
        
        /// Insert an element into the BST, returns false if the element cannot be inserted
        bool insert(const typename Accelerator::storage_type& e)
        {
            std::cout << "INSERTING ";
            print128_num(e);
            std::cout << std::endl;
            
            // Starting index in BST nodes (root = 1)
            std::size_t idx = 1;
            
            // Offset in the array
            std::size_t offset = 0;
            
            KmerCount *kmer;
            
            while (idx <= maxindex_)
            {
                offset = bfs2veb(idx);
                
                kmer = &data_[offset];
                
                std::cout << "> ";
                print128_num(kmer->kmer);
                std::cout << "(" << kmer->count << ") ";
                
                // New node
                if (kmer->count == 0)
                {
                    std::cout << "NEW" << std::endl;
                    kmer->kmer = e;
                    kmer->count = 1;
                    return true;
                }
                
                // If current kmer is == e, update count
                if (acc_.compare(kmer->kmer, e) == 0)
                {
                    std::cout << "EXISTING" << std::endl;
                    kmer->count++;
                    return true;
                }
                
                // If current kmer is < e, go left
                if (acc_.compare(kmer->kmer, e) < 0)
                {
                    std::cout << "LEFT" << std::endl;
                    idx = index_left(idx);
                }
                else
                {
                    std::cout << "RIGHT" << std::endl;
                    idx = index_right(idx);
                }
                
                // We reached the end of the tree, no insetion can be made
                if (idx > maxindex_) return false;
            }
            
            return true;
        }

    private:
        
        /// Computes the 0-based offset from a BFS 1-based indexing (i.e., only root has index 1)
        std::size_t bfs2veb(std::size_t idx)
        {
            if (idx > maxindex_ || idx == 0) throw std::domain_error("Index of VEB tree out of range.");
            
            // User precomputed indices if possible
            if (h_ <= 12)
            {
                return vebindices::indices_[vebindices::offsets_[h_ - 1] + (idx - 1)];
            }
            
            const std::size_t maxheight = 2;
                            
            // Initial height
            std::size_t temp = h_;
            
            // Offset from the beginning of the data_ array
            std::size_t offset = 0;
            
            while (temp > maxheight)
            {
                std::size_t ntop, midl, botl, nbot, hbot, nidx;
                
                // Middle level starting from 0
                midl = (temp >> 1);
                
                // Number of elements on top half
                ntop = (2 << midl) - 1;
                
                // Bottom half starting level (0 is root)
                botl = midl + 1;
                
                // Bottom chunks height
                hbot = temp - botl;
                
                // Number of sub-trees in the bottom half
                nbot = 2 << hbot;
                
                // Index is in the top half
                if (idx <= ntop)
                {
                    //offset += ntop;
                    temp    = midl + 1;
                }
                // Index is in the bottom half
                else
                {
                    // Is the node on the right child of its parent
                    bool rightchild = idx & 1;
                    
                    // Parent index
                    std::size_t parent = (idx & (~static_cast<std::size_t>(0) << 1)) >> 1;
                    
                    // New index, starting from the (putative) parent on top half, is 1
                    nidx = 1;
                    
                    // Find the parent in the top half
                    while (parent > ntop)
                    {
                        nidx       = (nidx << 1) + rightchild;
                        rightchild = parent & 1;
                        parent     = (parent & (~static_cast<std::size_t>(0) << 1)) >> 1;
                    }
                    
                    // Number of bottom blocks to skip
                    std::size_t nblocks = (parent - (1 << midl)) * 2 + rightchild;
                    
                    // Adjust index, offset, new index, and height
                    offset += ntop + nblocks * ((1 << hbot) - 1);
                    temp    = hbot;
                    idx     = nidx;
                } // End computing offset
            }
            
            // Remaining height is 2, return 0-based index plus offset
            return offset + (idx - 1);
        }
                    
        /// Height of the tree
        const std::size_t h_;
        
        /// Data in the tree
        KmerCount* data_;
        
        /// Total number of nodes (index-1)
        const std::size_t maxindex_;
        
        /// Accelerator
        Accelerator acc_;
    };
}

#endif /* veboas_hpp */
