//
//  filepath.cpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//

#include "path.hpp"

using namespace libseq;

/// Internal class for path_stat (incomplete, I know)
enum class stat_result
{
    incomplete_implementation,
    directory,
    file
};

// Forward declaration of platform-dependent code
std::string separator();
std::string to_absolute_path(const std::string &f);
std::string to_file_name(const std::string &f);
std::string to_file_path(const std::string &f);
stat_result path_stat(const std::string &f);

////////////////////////////////////////////////////////////////////////////////
path::path(const std::string &p)
{
    path_ = to_absolute_path(p);
}

////////////////////////////////////////////////////////////////////////////////
std::string path::absolute_path() const
{
    return to_absolute_path(path_);
}

////////////////////////////////////////////////////////////////////////////////
std::string path::file_name() const
{
    return to_file_name(path_);
}

////////////////////////////////////////////////////////////////////////////////
std::string path::file_path() const
{
    return to_file_path(path_);
}

////////////////////////////////////////////////////////////////////////////////
path path::append(const std::string &p) const
{
    return path_ + separator() + p;
}

////////////////////////////////////////////////////////////////////////////////
bool path::is_directory() const
{
    return path_stat(path_) == stat_result::directory;
}

////////////////////////////////////////////////////////////////////////////////
bool path::is_file() const
{
    return path_stat(path_) == stat_result::file;
}

////////////////////////////////////////////////////////////////////////////////
path::operator std::string()
{
    return path_;
}

////////////////////////////////////////////////////////////////////////////////
// PLATFORM DEPENDANT CODE /////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

#ifdef __APPLE__

#ifndef _POSIX_PATH_MAX
#define _POSIX_PATH_MAX  512
#endif

#include <stdlib.h>
#include <libgen.h>
#include <sys/stat.h>

std::string separator()
{
    return "/";
}

std::string to_absolute_path(const std::string &f)
{
    char buf[_POSIX_PATH_MAX] = {0};
    
    char *full_path = realpath(f.c_str(), buf);
    std::string s(buf);
    
    return s;
}

std::string to_file_name(const std::string &f)
{
    char *basec, *bname;
    
    basec = strdup(f.c_str());
    bname = basename(basec);
    std::string s(bname);
    
    return s;
}

std::string to_file_path(const std::string &f)
{
    char *dirc, *dname;
    
    dirc = strdup(f.c_str());
    dname = dirname(dirc);
    std::string s(dname);
    
    return s;
}

stat_result path_stat(const std::string &path)
{
    struct stat s;
    
    if (stat(path.c_str(), &s) == 0)
    {
        if (s.st_mode & S_IFDIR)
        {
            return stat_result::directory;
        }
        else if(s.st_mode & S_IFREG)
        {
            return stat_result::file;
        }
        else
        {
            return stat_result::incomplete_implementation;
        }
    }
    else
    {
        throw std::runtime_error("Cannot stat() path '" + path + "'.");
    }
    
    // Otherwise the compiler complains
    return stat_result::incomplete_implementation;
}

#endif
