//
//  logger.hpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//

#ifndef logger_hpp
#define logger_hpp

#include <string>
#include <memory>
#include <spdlog/spdlog.h>

namespace libseq
{
    
    /// Logger class to console
    class logger
    {
        /// All default constructors and operators= deleted
        logger(const logger&) = delete;
        logger(logger&&) = delete;
        logger& operator=(const logger&) = delete;
        logger& operator=(logger&&) = delete;
        
        /// Implementation details for the logger
        class details
        {
        public:
            
            /// Log file name
            std::string logfilename_;
            
            /// Console log object
            std::shared_ptr<spdlog::logger> console_;
            
            /// File log object
            std::shared_ptr<spdlog::logger> logfile_;
        };
        
    public:
        
        /// Initialize the logger
        static void init(const std::string &logfile);
        
        /// Return the log file name
        static const std::string& get_log_file();
        
        /// Log information to both file and console
        template<typename... Args>
        static void info(Args... args)
        {
            // Check for a valid logger initialization
            check();
            
            instance_->logfile_->info(args...);
            instance_->console_->info(args...);
        }
        
        /// Log debug info to file only, and to console also only in debug mode
        template<typename... Args>
        static void debug(Args... args)
        {
            // Check for a valid logger initialization
            check();
            
            instance_->logfile_->info(args...);
#if DEBUG
            instance_->console_->info(args...);
#endif
        }
        
    private:
        
        /// Instance
        static std::unique_ptr<details> instance_;
        
        static inline void check()
        {
            if (instance_ == nullptr)
                throw std::logic_error("You have called the logger before initializing it by calling init.");
        }
    };
    
}


#endif /* logger_hpp */
