//
//  accelerator_sse.hpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//


#ifndef accelerator_sse_hpp
#define accelerator_sse_hpp

#include <unordered_map>
#include <x86intrin.h>

#include "accelerator.hpp"

namespace libseq
{
        
    class accelerator_sse : public accelerator<accelerator_sse, __m128i>
    {
        
        /// Actual name of the class
        static const char name_[];
        
        /// Precomputed bitmasks for all k
        static storage_type mask_[64];
        
    public:
        
        /// Storage type
        using storage_type = __m128i;
        
        /// Return the name of SSE accelerator
        const char* name() const
        {
            return name_;
        }
        
        /// Return the bitmask for the kmer of size k
        storage_type mask(std::size_t k) const
        {
            return mask_[k - 1];
        }
        
        /// Build the implemented representation of a kmer from a string
        storage_type to_forward(const std::string_view &s) const;
    
        /// Build the implemented representation of a reverse-complement kmer from a string
        storage_type to_revcomp(const std::string_view &s) const;
        
        /// Build the implemented representation of a canonical representation of a kmer from a string
        storage_type to_canonical(const std::string_view &s) const;
        
        /// Build a string from the implemented representation of a kmer
        std::string to_string(const storage_type r, std::size_t k) const;
        
        /// Return the hash value of a kmer
        std::size_t hash(storage_type r) const;
            
        /// Compare 2 k-mers, lexicographically
        int compare(storage_type r, storage_type s) const;
        
        /// Equal
        bool equal(storage_type r, storage_type s) const;

    private:
    
        /// Hash class for the current storage type
        class hash_storage_type
        {
        public:
        
            // From https://github.com/skeeto/hash-prospector/blob/master/tests/splitmix64.c
            std::size_t hash(std::size_t x) const
            {
                x += 0x9e3779b97f4a7c15;
                x ^= (x >> 30);
                x *= 0xbf58476d1ce4e5b9;
                x ^= (x >> 27);
                x *= 0x94d049bb133111eb;
                x ^= (x >> 31);
                return x;
            }
        
            std::size_t operator()(const storage_type &r) const
            {
                std::size_t a = _mm_cvtsi128_si64(r);
                std::size_t b = _mm_extract_epi64(r, 1);
            
                std::size_t l = hash(a), u = hash(b);
            
                return l ^ (u << 31 | u >> 33);
            }
        };
        
    };


}

#endif //accelerator_sse_hpp
