//
//  accelerator.hpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//


#ifndef accelerator_hpp
#define accelerator_hpp

#include <string>
#include <string_view>
#include <array>

#include "inherit.hpp"

namespace libseq
{
    
    /// Accelerator base class
    template <typename Derived, typename KmerType>
    class accelerator : public inherit<Derived>
    {
    public:
    
        /// Actual kmer representation
        using storage_type = KmerType;
                
        /// Return the name of the derived class
        const char* name() const
        {
            return this->self().name();
        }
        
        storage_type mask(std::size_t k) const
        {
            return this->self().mask(k);
        }
        
        /// Build the implemented representation of a kmer from a string
        storage_type to_forward(const std::string_view &s) const
        {
            return this->self().to_forward(s);
        }
        
        /// Build the implemented representation of a reverse-complement kmer from a string
        storage_type to_revcomp(const std::string_view &s) const
        {
            return this->self().to_revcomp(s);
        }

        /// Build the implemented representation of a canonical representation of a kmer from a string
        storage_type to_canonical(const std::string_view &s) const
        {
            return this->self().to_canonical(s);
        }

        /// Build a string from the implemented representation of a kmer
        std::string to_string(const storage_type r, std::size_t k) const
        {
            return this->self().to_string(r, k);
        }
    
        /// Return the hash value of a kmer
        auto hash(const storage_type r) const
        {
            return this->self().hash(r);
        }
    
        /// Compare 2 k-mers, lexicographically
        int compare(const storage_type r1, const storage_type r2) const
        {
            return this->self().compare(r1, r2);
        }
    
        /// Equal
        bool equal(const storage_type r1, const storage_type r2) const
        {
            return this->self().equal(r1, r2);
        }
        
    };
    
}

#endif //accelerator_hpp
