//
//  auxbenchfuncs.cpp
//
//  Copyright 2022 Franco Milicchio. All rights reserved.
//

#include "auxbenchfuncs.hpp"

////////////////////////////////////////////////////////////////////////////////
std::string random_string( size_t length )
{
    // I'm lazy for this, see: https://stackoverflow.com/questions/440133/how-do-i-create-a-random-alpha-numeric-string-in-c
    auto randchar = []() -> char
    {
        const char charset[] = "ATCG";
        const size_t max_index = (sizeof(charset) - 1);
        return charset[ rand() % max_index ];
    };
    std::string str(length,0);
    std::generate_n( str.begin(), length, randchar );
    return str;
}

////////////////////////////////////////////////////////////////////////////////
/// parse forward //////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

void parse_kmer_fwd_string(int k, std::string &t)
{
    //Accelerator
    libseq::accelerator_string acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_string::storage_type from;
    
    kmer = std::string_view(t.data(), k);
    
    from = acc.to_forward(kmer);
}

void parse_kmer_fwd_sse(int k, std::string &t)
{
    //Accelerator
    libseq::accelerator_sse acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_sse::storage_type from;
    
    kmer = std::string_view(t.data(), k);
    
    from = acc.to_forward(kmer);
}

void parse_kmer_fwd_dsk(int k, std::string &t)
{
    // Parsed string
    Kmer<65>::Type from;
    
    Data data(t.data());
    
    std::string q(t);
    
    Kmer<65>::ModelDirect model(k);
    Kmer<65>::ModelDirect::Iterator it(model);
    it.setData(data);
    it.first();
    from = it->value(0);
}

////////////////////////////////////////////////////////////////////////////////
/// parse reverse complement ///////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

void parse_kmer_rev_string(int k, std::string &t)
{
    //Accelerator
    libseq::accelerator_string acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_string::storage_type from;
    
    kmer = std::string_view(t.data(), k);
    
    from = acc.to_revcomp(kmer);
}

void parse_kmer_rev_sse(int k, std::string &t)
{
    //Accelerator
    libseq::accelerator_sse acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_sse::storage_type from;
    
    kmer = std::string_view(t.data(), k);
    
    from = acc.to_revcomp(kmer);
}

////////////////////////////////////////////////////////////////////////////////
/// parse canonical ////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

void parse_kmer_can_string(int k, std::string &t)
{
    //Accelerator
    libseq::accelerator_string acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_string::storage_type from;
    
    kmer = std::string_view(t.data(), k);
    
    from = acc.to_canonical(kmer);
}

void parse_kmer_can_sse(int k, std::string &t)
{
    //Accelerator
    libseq::accelerator_sse acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_sse::storage_type from;
    
    kmer = std::string_view(t.data(), k);
    
    from = acc.to_canonical(kmer);
}

void parse_kmer_can_dsk(int k, std::string &t)
{
    // Parsed string
    Kmer<65>::Type from;
    
    Data data(t.data());
    
    Kmer<65>::ModelCanonical model(k);
    Kmer<65>::ModelCanonical::Iterator it(model);
    it.setData(data);
    it.first();
    from = it->value(0);
}

void parse_kmer_can_kmc(int k, std::string &t)
{
    CKmer<2> from;
    
    uchar *tmp = (uchar*)t.data();
    
    from.load(tmp, k);
}

void parse_read_can_string(int k, std::string &t)
{
    std::size_t j, readlength = t.size();
    
    //Accelerator
    libseq::accelerator_string acc;
        
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_string::storage_type from;
    
    // Parse every kmer
    for (j = 0; j < readlength - k + 1; j++)
    {
        kmer = std::string_view(t.data() + j, k);
        
        from = acc.to_canonical(kmer);
        
        //auto to = acc.to_string(from, k);
    }
}

void parse_read_can_sse(int k, std::string &t)
{
    std::size_t j, readlength = t.size();
    
    // Accelerator
    libseq::accelerator_sse acc;
    
    std::string_view kmer;
    
    // Parsed string
    libseq::accelerator_sse::storage_type from;
    
    // Parse every kmer
    for (j = 0; j < readlength - k + 1; j++)
    {
        kmer = std::string_view(t.data() + j, k);
        
        from = acc.to_canonical(kmer);
        
        //auto to = acc.to_string(from, k);
    }
}

void parse_read_can_dsk(int k, std::string &t)
{
    // Parsed string
    Kmer<65>::Type from;

    Data data(t.data());

    // Parse every kmer
    Kmer<65>::ModelCanonical model(k);
    Kmer<65>::ModelCanonical::Iterator it(model);
    it.setData(data);
    
    // Parse every kmer
    for (it.first(); !it.isDone(); it.next())
    {
        from = it->value(0);
    }
}

void parse_read_can_kmc(int k, std::string &t)
{
    std::size_t j, readlength = t.size();
    
    CKmer<2> from;
    
    std::string_view kmer;
    
    // Parse every kmer
    for (j = 0; j < readlength - k + 1; j++)
    {
        kmer = std::string_view(t.data() + j, k);

        uchar *tmp = (uchar*) kmer.data();
        
        from.load(tmp, k);
    }
}

