//
//  partitioner.hpp
//
//  Copyright 2018 Franco Milicchio and Marco Oliva. All rights reserved.
//

#ifndef partitioner_hpp
#define partitioner_hpp

#include <vector>

#include "inherit.hpp"
#include "path.hpp"
#include "logger.hpp"
#include "timer.hpp"
#include "accelerator.hpp"


namespace libseq
{
    
    /// Partitioner base class
    template <typename Derived>
    class partitioner : public inherit<Derived>
    {
    
    protected:
        
        /// Statistics
        struct partitioner_stats
        {
            std::size_t n_reads = 0;
            std::size_t n_kmers = 0;
        };
    
    public:
        
        /// Return the name of the derived class
        inline const char* name() const
        {
            return this->self().name_;
        }
    
        /// Return the output partitions
        inline const std::vector<path>& get_partitions() const
        {
            return this->self().partitions_;
        }
    
        /// Core method, partition the input file according to the hash value of each kmer
        template <typename Implementation, typename StorageType, typename D = std::chrono::milliseconds>
        void run(accelerator<Implementation, StorageType>& accelerator, path input_file, std::size_t k,
        std::size_t np)
        {
            libseq::logger::info("Partitioner type: {}", this->name());
            libseq::logger::info("Started running partitioner with k = {} in {} partitions", k, np);
    
            double time = 0.0;
            libseq::timer t;
            
            this->self().run(accelerator, input_file, k, np);
    
            time = t.elapsed<D>();
    
            libseq::logger::info("Ended hashing of {} kmers in {} reads in {} ({})",
                this->self().stats_.n_kmers,
                this->self().stats_.n_reads,
                time,
                typeid(D).name());
        }
        
    };

}

#endif //partitioner_hpp
