//
//  fastq_mmap.hpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//

#ifndef fastq_mmap_hpp
#define fastq_mmap_hpp

#include <string_view>
#include <string>
#include <tuple>

#include <mio/mmap.hpp>

#include "reader.hpp"

namespace libseq
{
    
    /// Memory-mapped FASTQ reader
    class fastq_mmap : public reader<fastq_mmap>
    {
    public:
        
        // Forward definition
        class iterator;
        
        /// Construct a FASTQ reader with a memory mapped file
        fastq_mmap(const std::string &filename);
        
        /// Return the beginning of the file
        iterator begin() const
        {
            return iterator(mmap_.data(), mmap_.size());
        }
        
        /// Return the ending of the file
        iterator end() const
        {
            return iterator(nullptr, 0);
        }
        
        /// Return the file size in bytes
        std::size_t size() const
        {
            return mmap_.size();
        }

    private:
        
        /// Memory-mapped file
        mio::mmap_source mmap_;
        
    public:
        
        /// Iterator on a memory-mapped FASTQ file
        class iterator
        {
            /// The fastq_mmap class can create and modify the iterator
            friend class fastq_mmap;
            
            /// Constructor called by the fastq_mmap class
            iterator(const char* data, std::size_t size) : map_(data), end_(map_ + size)
            {
                // Is this the end() iterator
                if ((data == nullptr) or (size < 1))
                {
                    // Nope
                    l_1_ = l_2_ = l_3_ = l_4_ = map_ = end_ = nullptr;
                }
                else
                {
                    // Start
                    l_1_ = map_;
                    l_2_ = static_cast<const char*>(memchr(l_1_, '\n', end_ - l_1_)) + 1;
                    l_3_ = static_cast<const char*>(memchr(l_2_, '\n', end_ - l_2_)) + 1;
                    l_4_ = static_cast<const char*>(memchr(l_3_, '\n', end_ - l_3_)) + 1;
                    tmp_ = std::string_view(l_2_, l_3_ - l_2_ - 1);
                }
            }

        public:
            
            /// Value type used for the read
            using value_type = const std::string_view;
            
            /// Difference of iterators type
            using difference_type = std::size_t;
            
            /// Pointer type
            using pointer = const void*;
            
            /// Size type
            using size_type = std::size_t;
            
            /// Reference type
            using reference = const std::string_view&;
            
            /// Iterator type (input only, forward)
            using iterator_category = std::forward_iterator_tag;
            
        public:
            
            /// Check if two interators are equal
            inline bool operator==(iterator other) const
            {
                return (map_ == other.map_) and (end_ == other.end_) and
                       (l_1_ == other.l_1_) and (l_2_ == other.l_2_) and
                       (l_3_ == other.l_3_) and (l_4_ == other.l_4_);
            }
            
            /// Check if two interators are different
            inline bool operator!=(iterator other) const
            {
                return not(*this == other);
            }
            
            /// Return the read pointed at by the iterator
            inline const value_type& operator*() const
            {
                return tmp_;
            }
            
            /// Return the read pointed at by the iterator
            inline value_type& operator*()
            {
                return tmp_;
            }
            
            /// Move to the next read (prefix operator)
            inline iterator& operator++()
            {
                // Next read
                l_1_ = static_cast<const char*>(memchr(l_4_, '\n', end_ - l_4_)) + 1;
                
                // Is the position still valid
                if (l_1_ >= end_)
                {
                    // No room for us to continue
                    l_1_ = l_2_ = l_3_ = l_4_ = map_ = end_ = nullptr;
                }
                else
                {
                    // We assume it is a well-formed FASTQ file
                    l_2_ = static_cast<const char*>(memchr(l_1_, '\n', end_ - l_1_)) + 1;
                    l_3_ = static_cast<const char*>(memchr(l_2_, '\n', end_ - l_2_)) + 1;
                    l_4_ = static_cast<const char*>(memchr(l_3_, '\n', end_ - l_3_)) + 1;
                    tmp_ = std::string_view(l_2_, l_3_ - l_2_ - 1);
                }
                
                return *this;
            }
            
            /// Move to the next read (suffix operator)
            inline iterator operator++(int)
            {
                iterator retval = *this;
                ++(*this);
                return retval;
            }
            
            /// Return the FASTQ read properties as a tuple (header, read, optional, quality)
            inline auto properties()
            {
                const char *next = static_cast<const char*>(memchr(l_4_, '\n', end_ - l_4_)) + 1;
                
                return std::make_tuple(std::string_view(l_1_, l_2_ - l_1_ - 1),
                                       std::string_view(l_2_, l_3_ - l_2_ - 1),
                                       std::string_view(l_3_, l_4_ - l_3_ - 1),
                                       std::string_view(l_4_, next - l_4_ - 1));
            }
            
        private:
            
            /// Current memory-mapped data position
            const char* map_;
            
            /// End-of-file (mmap'ed) position
            const char* end_;
            
            /// Lines in a FASTQ file
            const char *l_1_, *l_2_, *l_3_, *l_4_;
            
            std::string_view tmp_;
        };

    };
    
}

#endif /* fastq_mmap_hpp */
