//
//  accelerator_integer.cpp
//
//  Copyright 2018 Franco Milicchio. All rights reserved.
//


#ifndef accelerator_integer_hpp
#define accelerator_integer_hpp

#include <string>
#include <string_view>
#include <unordered_map>
#include <algorithm>
#include <array>

#include "accelerator.hpp"
#include "precomputed.hpp"

namespace libseq
{
    
    class accelerator_uint128 : public accelerator<accelerator_uint128, std::array<std::size_t, 2>>
    {

    public:
    
        /// Actual name of the class
        static constexpr char name_[] = "accelerator_uint128";
    
        /// Storage type
        using storage_type = std::array<std::size_t, 2>;
    
        /// Build the implemented representation of a kmer from a string
        storage_type to_forward(const std::string_view& s) const;
    
        /// Build a string from the implemented representation of a kmer
        std::string to_string(const storage_type& r, std::size_t k) const;
            
        /// Return the hash value of a kmer
        std::size_t hash(storage_type r) const;
    
        /// Compare 2 k-mers, lexicographically
        int compare(storage_type r1, storage_type r2) const;
    
        /// Equal
        bool equal(storage_type r1, storage_type r2) const;

    private:
    
        /// Hash class for the current storage type
        class hash_storage_type
        {
        public:
        
            // From https://github.com/skeeto/hash-prospector/blob/master/tests/splitmix64.c
            std::size_t hash(std::size_t x) const
            {
                x += 0x9e3779b97f4a7c15;
                x ^= (x >> 30);
                x *= 0xbf58476d1ce4e5b9;
                x ^= (x >> 27);
                x *= 0x94d049bb133111eb;
                x ^= (x >> 31);
                return x;
            }
        
            std::size_t operator()(const storage_type &r) const
            {
                std::size_t l = hash(r[0]), u = hash(r[1]);
            
                return l ^ (u << 31 | u >> 33);
            }
        };
    };

}

#endif //accelerator_integer_hpp
